/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.classpool.plugins.jbosscl;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Collection;
import java.util.HashSet;

import javassist.ClassPool;
import javassist.scopedpool.ScopedClassPoolFactory;
import javassist.scopedpool.ScopedClassPoolRepository;

import org.jboss.classloader.spi.ClassLoaderDomain;
import org.jboss.classloader.spi.ClassLoaderSystem;
import org.jboss.classloading.spi.RealClassLoader;
import org.jboss.classloading.spi.dependency.ClassLoading;
import org.jboss.classloading.spi.dependency.Module;
import org.jboss.classpool.domain.ClassPoolDomain;
import org.jboss.classpool.domain.ClassPoolDomainRegistry;
import org.jboss.classpool.plugins.NonDelegatingClassPool;
import org.jboss.classpool.plugins.temp.TempClassPoolFactory;
import org.jboss.classpool.spi.AbstractClassPool;

/**
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 97735 $
 */
public class JBossClDelegatingClassPoolFactory extends TempClassPoolFactory implements ScopedClassPoolFactory
{
   private final DomainRegistry registry;
   
   private final RegisterModuleCallback registerModuleCallback;
   
   private Collection<ClassLoader> currentClassLoaders = new HashSet<ClassLoader>();
   
   public JBossClDelegatingClassPoolFactory(DomainRegistry registry, RegisterModuleCallback registerModuleCallback)
   {
      this.registry = registry;
      this.registerModuleCallback = registerModuleCallback;
   }
   
   public synchronized AbstractClassPool create(ClassLoader cl, ClassPool src, ScopedClassPoolRepository repository)
   {
      if (this.currentClassLoaders.contains(cl))
      {
         return null;
      }
      currentClassLoaders.add(cl);
      try
      {
      ClassPool parent = getCreateParentClassPools(cl, src, repository);
      
      if (cl instanceof RealClassLoader)
      {
         Module module = registry.getModule(cl);
         if (module == null)
         {
            module = getModuleForClassLoader(cl);
         }
         registerModuleCallback.registerModule(module);
         registerBootstrapLoaders(module, repository);
         ClassPoolDomain domain = getDomain(module, cl, repository);
         // this indicates that the domain corresponds to a cp that is being created
         // to avoid duplicate class pools
         if (domain == null)
         {
            return null;
         }
         return new JBossClDelegatingClassPool(domain, cl, parent, repository, module, registerModuleCallback);
      }
      if (parent == null)
      {
         parent = ClassPool.getDefault();
      }
      return new NonDelegatingClassPool(cl, parent, repository, true);
      }
      finally
      {
         this.currentClassLoaders.remove(cl);
      }
   }

   private synchronized ClassPoolDomain getDomain(Module module, ClassLoader cl, ScopedClassPoolRepository repository)
   {
      ClassLoaderDomain domain = null;
      ClassLoaderSystem sys = registry.getSystem();
      if (module != null && module.getDeterminedParentDomainName() != null)
      {
         //It is scoped
         domain = sys.getDomain(module.getDeterminedDomainName());
      }
      
      if (domain == null)
      {
         domain = registry.getDefaultDomain();
      }
      
      return getDomain(cl, domain, sys, repository);
   }

   private ClassPoolDomain getDomain(ClassLoader cl, ClassLoaderDomain domain,
         ClassLoaderSystem sys, ScopedClassPoolRepository repository)
   {
      ClassPoolDomain poolDomain = ClassPoolDomainRegistry.getInstance().getDomain(domain);
      if (poolDomain == null)
      {
         String parentDomainName = domain.getParentDomainName();
         if (parentDomainName != null)
         {
            ClassLoaderDomain parentDomain = sys.getDomain(parentDomainName);
            if (parentDomain == null)
            {
               throw new RuntimeException("No domain found, domain name: " + parentDomainName);
            }
            ClassPoolDomain parentPoolDomain = this.getDomain(null, parentDomain, sys, repository);
            poolDomain = new JBossClClassPoolDomain(domain.getName(), parentPoolDomain, domain.getParentPolicy(), registry);
         }
         else
         {
            ClassLoader parentUnitLoader = registry.getParentUnitLoader(cl);
            if (this.currentClassLoaders.contains(parentUnitLoader))
            {
               return null;
            }
            ClassPool parentUnitPool = parentUnitLoader == null? null: repository.registerClassLoader(parentUnitLoader);
            if (parentUnitPool == null)
            {
               poolDomain = new JBossClClassPoolDomain(domain.getName(), domain.getParentPolicy(), registry);
            }
            else
            {
               poolDomain = new JBossClClassPoolDomain(domain.getName(), parentUnitPool, domain.getParentPolicy(), registry);
            }
         }
         ClassPoolDomainRegistry.getInstance().addClassPoolDomain(domain, poolDomain);
      }
      return poolDomain;
   }
   
   @Override
   protected ClassPool getCreateParentClassPools(ClassLoader cl, ClassPool src, ScopedClassPoolRepository repository)
   {
      ClassPool parent = super.getCreateParentClassPools(cl, src, repository);
      if (parent == ClassPool.getDefault())
      {
         //In AS BaseClassLoader seems to normally have a null parent
         return null;
      }
      return parent;
   }
   
   private void registerBootstrapLoaders(Module skip, ScopedClassPoolRepository repository)
   {
      Collection<Module> unregistered = registerModuleCallback.getUnregisteredModules();
      if (unregistered.size() > 0)
      {
         for (Module module : unregistered)
         {
            if (module == skip)
            {
               continue;
            }
            ClassLoader loader = getClassLoaderForModule(module);
            if (this.currentClassLoaders.contains(loader))
            {
               continue;
            }
            ClassPool classPool = repository.registerClassLoader(loader);
            if (classPool == null)
            {
               repository.unregisterClassLoader(loader);
            }
         }
      }
   }
   
   private ClassLoader getClassLoaderForModule(final Module module)
   {
      return AccessController.doPrivileged(new PrivilegedAction<ClassLoader>()
      {
      
         public ClassLoader run()
         {
            return ClassLoading.getClassLoaderForModule(module);
         }
      });
   }
   
   private Module getModuleForClassLoader(final ClassLoader classLoader)
   {
      return AccessController.doPrivileged(new PrivilegedAction<Module>()
      {
         public Module run()
         {
            return ClassLoading.getModuleForClassLoader(classLoader);
         }
      });
   }
}