/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.classpool.plugins.jbosscl;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Collections;
import java.util.Map;
import java.util.WeakHashMap;

import javassist.ClassPool;

import org.jboss.classloading.spi.dependency.ClassLoading;
import org.jboss.classloading.spi.dependency.Module;
import org.jboss.classpool.scoped.ScopedClassPool;
import org.jboss.classpool.spi.ClassPoolRepository;
import org.jboss.logging.Logger;

/**
 * Repository for JBossclDelegatingClassPools.
 * 
 * @author <a href="mailto:flavia.rainone@jboss.org">Flavia Rainone</a>
 *
 * @version $Revision: 103631 $
 */

public class JBossClDelegatingClassPoolRepository extends ClassPoolRepository
{
   private final Logger log = Logger.getLogger(JBossClDelegatingClassPoolRepository.class);
   
   private final static JBossClDelegatingClassPoolRepository instance = new JBossClDelegatingClassPoolRepository();
   
   /**
    * Returns the singleton instance.
    * 
    * @return the singleton repository instance
    */
   public static JBossClDelegatingClassPoolRepository getInstance()
   {
      return instance;
   }
   
   /** The registered modules */
   protected Map<Module, ScopedClassPool> registeredModules =
      Collections.synchronizedMap(new WeakHashMap<Module, ScopedClassPool>());
   
   @Override
   public ClassPool registerClassLoader(ClassLoader classLoader)
   {
      ScopedClassPool classPool = (ScopedClassPool) super.registerClassLoader(classLoader);
      if (classPool == null)
      {
         // TODO check this works; was delegate before
         super.unregisterClassLoader(classLoader);
      }
      else
      {
         Module module = getModuleForClassLoader(classLoader);
         this.registeredModules.put(module, classPool);
      }
      return classPool;
   }
   

   public void unregisterClassLoader(ClassLoader classLoader, Module module)
   {
      ScopedClassPool classPool = registeredModules.remove(module);
      if (classLoader == null)
      {
         if (classPool == null)
         {
            //throw new IllegalStateException("Module " + module + " is not registered");
            //TODO JBREFLECT-116
           if (log.isDebugEnabled())
              log.warn("Module " + module + " is not registered");
           return;
         }
         classPool.close();
      }
      else
      {
         unregisterClassLoader(classLoader);
      }
   }
   
   private Module getModuleForClassLoader(final ClassLoader classLoader)
   {
      return AccessController.doPrivileged(new PrivilegedAction<Module>()
      {
         public Module run()
         {
            return ClassLoading.getModuleForClassLoader(classLoader);
         }
      });
   }
}