/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.classpool.plugins.jbosscl;

import org.jboss.classloading.spi.dependency.ClassLoading;
import org.jboss.classpool.scoped.ScopedClassPoolFactory;
import org.jboss.classpool.spi.ClassPoolRepository;

/**
 * This sets up the ClassPools in JBoss AS. It gets installed in JBoss AS under 
 * the name 'JBossClClassPoolConfig' and provides the items for other subsystems
 * used in AS.
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class JBossClClassPoolConfig
{
   private static volatile JBossClClassPoolConfig config;
   
   private final DomainRegistry domainRegistry;
   
   private final RegisterModuleCallback registerModuleCallback;
   
   private ClassLoading classLoading;
   
   private JBossClDelegatingClassPoolFactory classPoolFactory;
   
   private ClassPoolRepository classPoolRepository;
   
   private JBossClClassPoolConfig(DomainRegistry domainRegistry, RegisterModuleCallback registerModuleCallback, JBossClDelegatingClassPoolFactory classPoolFactory)
   {
      this.domainRegistry = domainRegistry;
      this.registerModuleCallback = registerModuleCallback;
      this.classPoolFactory = classPoolFactory;
      classPoolRepository = ClassPoolRepository.getInstance();
      classPoolRepository.setClassPoolFactory(classPoolFactory);
   }
   
   /**
    * Get the singleton instance
    * 
    * @return the singleton instance
    */
   public static JBossClClassPoolConfig getInstance()
   {
      if (config == null)
      {
         initialize();
      }
      return config;
   }
   
   private static synchronized void initialize()
   {
      if (config != null)
         return;
      
      DomainRegistry domainRegistry = VFSClassLoaderDomainRegistry.getInstance();
      RegisterModuleCallback registerModuleCallback = new RegisterModuleCallback(domainRegistry);
      JBossClDelegatingClassPoolFactory classPoolFactory = new JBossClDelegatingClassPoolFactory(domainRegistry, registerModuleCallback);
      
      config = new JBossClClassPoolConfig(domainRegistry, registerModuleCallback, classPoolFactory);
   }
   
   /**
    * Get the domainRegistry
    * 
    * @return the domainRegistry
    */
   public DomainRegistry getDomainRegistry()
   {
      return domainRegistry;
   }
   
   /**
    * Get the classLoading
    * 
    * @return the classLoading
    */
   public ClassLoading getClassLoading()
   {
      return classLoading;
   }

   /**
    * Set the classLoading. This should be set via a property 
    * by the MC once the ClassLoading bean has been installed
    * 
    * @param cl the classLoading to set
    */
   public void setClassLoading(ClassLoading cl)
   {
      if (cl != null)
         cl.addModuleRegistry(registerModuleCallback);
      else if (classLoading != null)
         classLoading.removeModuleRegistry(registerModuleCallback);
      
      classLoading = cl;
   }

   /**
    * Get the registerModuleCallback
    * 
    * @return the registerModuleCallback
    */
   public RegisterModuleCallback getRegisterModuleCallback()
   {
      return registerModuleCallback;
   }

   /**
    * Get the classPoolFactory
    * 
    * @return the classPoolFactory
    */
   public ScopedClassPoolFactory getClassPoolFactory()
   {
      return classPoolFactory;
   }

   /**
    * Set the classPoolFactory
    * 
    * @param classPoolFactory the classPoolFactory to set
    */
   public void setClassPoolFactory(JBossClDelegatingClassPoolFactory classPoolFactory)
   {
      this.classPoolFactory = classPoolFactory;
   }

   /**
    * Get the classPoolRepository
    * @return the classPoolRepository
    */
   public ClassPoolRepository getClassPoolRepository()
   {
      return classPoolRepository;
   }
}
