/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.classpool.spi;

import java.util.Collection;
import java.util.HashSet;
import java.util.Map;

import javassist.ClassPool;
import javassist.scopedpool.ScopedClassPool;
import javassist.scopedpool.ScopedClassPoolFactory;
import javassist.scopedpool.ScopedClassPoolRepository;
import javassist.scopedpool.ScopedClassPoolRepositoryImpl;


/**
 * Singleton classpool repository.
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 97910 $
 */
public class ClassPoolRepository implements ScopedClassPoolRepository
{
   private final static ClassPoolRepository instance = new ClassPoolRepository();
   
   protected ScopedClassPoolRepository delegate;
   private ClassPoolRepositoryCallback callback;
   private Collection<ClassLoader> currentClassLoaders = new HashSet<ClassLoader>();
   
   /**
    * Returns the singleton instance.
    * 
    * @return the singleton repository instance
    */
   public static ClassPoolRepository getInstance()
   {
      return instance;
   }
   
   /**
    * Constructor.
    */
   protected ClassPoolRepository()
   {
      this.delegate = ScopedClassPoolRepositoryImpl.getInstance();
   }
   
   /**
    * Sets the classpool factory that should be used. This method should always
    * be called before the repository is used.
    */
   public void setClassPoolFactory(ScopedClassPoolFactory factory)
   {
      delegate.setClassPoolFactory(factory);
   }
   
   /**
    * Returns the classpool factory.
    */
   public ScopedClassPoolFactory getClassPoolFactory()
   {
      return delegate.getClassPoolFactory();
   }
   
   /**
    * Sets a callback object for notifications on classloader registration.
    *  
    * @param callback the callback object
    * @see ClassPoolRepositoryCallback
    */
   public synchronized void setClassPoolRepositoryCallback(ClassPoolRepositoryCallback callback)
   {
      this.callback = callback;
   }
   
   /**
    * Returns the callback object.
    * @return the callback object. May be {@code null}
    */
   public synchronized ClassPoolRepositoryCallback getClassPoolRepositoryCallback()
   {
      return this.callback;
   }

   public boolean isPrune()
   {
      return delegate.isPrune();
   }

   public void setPrune(boolean prune)
   {
      delegate.setPrune(prune);
   }

   /**
    * Creates a ClassPool corresponding to {@code classLoader}.
    * 
    * @param classLoader the classLoader corresponding to the created ClassPool
    * @param parent the parent of the class pool to be created.
    * @return the created ClassPool.
    */
   public ScopedClassPool createScopedClassPool(ClassLoader classLoader, ClassPool parent)
   {
      return delegate.createScopedClassPool(classLoader, parent);
   }

   /**
    * Finds the ClassPool corresponding to {@code classLoader}
    * 
    * @param classLoader the classLoader
    * @return the ClassPoool that corresponds to {@classLoader}
    */
   public ClassPool findClassPool(ClassLoader classLoader)
   {
      return delegate.findClassPool(classLoader);
   }
   
   /**
    * Get the registered classloaders
    * 
    * @return the registered classloaders
    */
   @SuppressWarnings("unchecked")
   public Map<ClassLoader, ClassPool> getRegisteredCLs()
   {
      return delegate.getRegisteredCLs();
   }

   /**
    * This method will check to see if a register classloader has been undeployed (as in JBoss)
    */
   public void clearUnregisteredClassLoaders()
   {
      delegate.clearUnregisteredClassLoaders();
   }
   
   /**
    * Registers {@code classLoader} and returns the corresponding {@code ClassPool}.
    * 
    * @param classLoader the ClassLoader to be added to this repository
    * @return the ClassPool corresponding to {@code classLoader}
    */
   public synchronized ClassPool registerClassLoader(ClassLoader classLoader)
   {
      if (classLoader == null)
      {
         classLoader = SecurityActions.getContextClassLoader();
      }
      if (currentClassLoaders.contains(classLoader))
      {
         return null;
      }
      // TODO JBREFLECT-63 review this
      currentClassLoaders.add(classLoader);
      ScopedClassPool classPool = (ScopedClassPool) delegate.registerClassLoader(classLoader);
      currentClassLoaders.remove(classLoader);
      
      // TODO review classPool != null check for AOP tests
      if (classPool != null && callback != null)
      {
         callback.classLoaderRegistered(classLoader);
      }
      return classPool;
   }

   /**
    * Unregisters {@code classLoader}.
    * 
    * @param classLoader the ClassLoader to be removed from this repository
    */
   public synchronized void unregisterClassLoader(ClassLoader classLoader)
   {
      delegate.unregisterClassLoader(classLoader);
      if (callback != null)
      {
         callback.classLoaderUnregistered(classLoader);
      }
   }
}