/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.classpool.spi;

import java.util.Iterator;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.ConcurrentHashMap;

import javassist.ClassPool;
import javassist.CtClass;
import javassist.scopedpool.ScopedClassPool;
import javassist.scopedpool.ScopedClassPoolRepository;

import org.jboss.classpool.helpers.ClassLoaderUtils;
import org.jboss.logging.Logger;

/**
 * A ClassPool tailored for usage with JBoss AS.
 * 
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @author <a href="mailto:flavia.rainone@jboss.com">Flavia Rainone</a>
 * @version $Revision: 97733 $
 */
public class AbstractClassPool extends ScopedClassPool
{
   protected final Logger logger = Logger.getLogger(this.getClass());
   
   /** Classnames of classes that will be created - we do not want to look for these in other pools.
    * The main use for this is when a class is created in a parent pool, and we then want to 
    * create a class with the same name in a parent-last child pool. As part of the create process
    * javassist.ClassPool will check if that class is frozen (which in turn will call getCached()
    * and get0()). If the classname exists in this map, get0() and getCached() should return null;   
    */
   protected final ConcurrentHashMap<String, String> generatedClasses = new ConcurrentHashMap<String, String>();

   protected final ConcurrentHashMap<String, Boolean> localResources = new ConcurrentHashMap<String, Boolean>();

   /** Classnames of classes that have been loaded, but were not woven */
   protected final ConcurrentHashMap<String, Boolean> loadedButNotWovenClasses = new ConcurrentHashMap<String, Boolean>();

   /** Causes the AbstractClassPool.getCached() method to search all ClassPools registered in the repository */
   public static final Class<SearchAllRegisteredLoadersSearchStrategy> SEARCH_ALL_STRATEGY = SearchAllRegisteredLoadersSearchStrategy.class;

   /** Causes the AbstractClassPool.getCached() method to search only itself */
   public static final Class<SearchLocalLoaderLoaderSearchStrategy> SEARCH_LOCAL_ONLY_STRATEGY = SearchLocalLoaderLoaderSearchStrategy.class;
   
   private final ClassPoolSearchStrategy searchStrategy;
   
   private final Map<AbstractClassPool, Boolean> children = new WeakHashMap<AbstractClassPool, Boolean>();
   
   static
   {
      ClassPool.doPruning = false;
      ClassPool.releaseUnmodifiedClassFile = false;
   }

   public AbstractClassPool(ClassLoader cl, ClassPool src, ScopedClassPoolRepository repository)
   {
      this(cl, src, repository, false);
   }

   protected AbstractClassPool(ClassPool src, ScopedClassPoolRepository repository)
   {
      this(null, src, repository, true);
   }

   private AbstractClassPool(ClassLoader cl, ClassPool src, ScopedClassPoolRepository repository, boolean isTemp)
   {
      this(cl, src, repository, SEARCH_ALL_STRATEGY, isTemp);
   }

   public AbstractClassPool(ClassLoader cl, ClassPool src, ScopedClassPoolRepository repository, Class<? extends ClassPoolSearchStrategy> searchStrategy)
   {
      this(cl, src, repository, searchStrategy, false);
   }
   
   public AbstractClassPool(ClassPool src, ScopedClassPoolRepository repository, Class<? extends ClassPoolSearchStrategy> searchStrategy)
   {
      this(null, src, repository, searchStrategy, true);
   }
   
   private AbstractClassPool(ClassLoader cl, ClassPool src, ScopedClassPoolRepository repository, Class<? extends ClassPoolSearchStrategy> searchStrategy, boolean isTemp)
   {
      super(cl, src, repository, isTemp);
      if (searchStrategy == SEARCH_ALL_STRATEGY)
      {
         this.searchStrategy = new SearchAllRegisteredLoadersSearchStrategy();
      }
      else if (searchStrategy == SEARCH_LOCAL_ONLY_STRATEGY)
      {
         this.searchStrategy = new SearchLocalLoaderLoaderSearchStrategy();
      }
      else
      {
         try
         {
            this.searchStrategy = searchStrategy.newInstance();
         }
         catch (Exception e)
         {
            throw new RuntimeException("Error instantiating search strategy class " + searchStrategy, e);
         }
      }
      if (logger.isTraceEnabled()) logger.trace(this + " creating pool for loader " + cl + " searchStrategy:" + this.searchStrategy + " isTemp:" + isTemp);

      registerWithParent();
   }
   
   private void registerWithParent()
   {
      if (parent != null && parent instanceof AbstractClassPool)
      {
         ((AbstractClassPool)parent).children.put(this, Boolean.TRUE);
      }
   }
   
   private void unregisterWithParent()
   {
      if (parent != null && parent instanceof AbstractClassPool)
      {
         ((AbstractClassPool)parent).children.remove(this);
      }
   }
   
   public void registerGeneratedClass(String className)
   {
      generatedClasses.put(className, className);
   }

   public boolean isGeneratedClass(String className)
   {
      return generatedClasses.containsKey(className);
   }
   
   public void doneGeneratingClass(String className)
   {
      generatedClasses.remove(className);
   }

   public void close()
   {
      super.close();
      unregisterWithParent();
      
      for (Iterator<AbstractClassPool> childIterator = children.keySet().iterator(); childIterator.hasNext();)
      {
         AbstractClassPool child = childIterator.next();
         childIterator.remove();
         if (child.getClassLoader() != null)
         {
            child.close();
         }
      }
   }

   public CtClass getCached(String classname)
   {
      return searchStrategy.getCached(classname);
   }

   @Override
   public void cacheCtClass(String classname, CtClass c, boolean dynamic)
   {
      boolean trace = logger.isTraceEnabled();
      if (trace) logger.trace(this + " caching " + classname);
      // TODO remove this true when ready
      super.cacheCtClass(classname, c, true);
      if (dynamic)
      {
         if (trace) logger.trace(this + " registering dynamic class " + classname);
         doneGeneratingClass(classname);
         String resourcename = getResourceName(classname);
         localResources.put(resourcename, Boolean.TRUE);
      }
   }

   protected boolean includeInGlobalSearch()
   {
      return true;
   }

   protected String getResourceName(String classname)
   {
      return ClassLoaderUtils.getResourceName(classname);
   }

   protected final boolean isLocalResource(String resourceName, boolean trace)
   {
      String classResourceName = resourceName;
      Boolean isLocal = localResources.get(classResourceName);
      if (isLocal != null)
      {
         if (trace) logger.trace(this + " " + resourceName + " is local " + isLocal);
      
         return isLocal.booleanValue();
      }
      boolean localResource = isLocalClassLoaderResource(classResourceName);
      localResources.put(classResourceName, localResource ? Boolean.TRUE : Boolean.FALSE);
      
      if (trace) logger.trace(this + " " + resourceName + " is local " + localResource);
      
      return localResource;
   }

   protected boolean isLocalClassLoaderResource(String classResourceName)
   {
      return getClassLoader().getResource(classResourceName) != null;
   }

   public String toString()
   {
      ClassLoader cl = null;
      try
      {
         cl = getClassLoader();
      }
      catch(IllegalStateException ignore)
      {
      }
      return this.getClass().getName() + "@" + System.identityHashCode(this) + " " + super.toString() + " - dcl:" + cl;
   }
   
   protected String getClassPoolLogStringForClass(CtClass clazz)
   {
      if (clazz == null)
      {
         return null;
      }
      if (clazz.getClassPool() == null)
      {
         return null;
      }
      return clazz.getClassPool().toString();
   }
   
   /**
    * 
    * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
    * @version $Revision: 97733 $
    */
   public interface ClassPoolSearchStrategy
   {
      CtClass getCached(String classname);
   }

   /**
    * Contains the original AbstractClassPool.getCached()
    * 
    */
   // NOTE: identical to ScopedClassPoolRepository, with two differences:
   // 1. This method uses the shortcuts isLocalResource and getResourceName
   // 2. This method uses the Repository.includeInGlobalSearch method
   @SuppressWarnings("unchecked")
   private class SearchAllRegisteredLoadersSearchStrategy implements ClassPoolSearchStrategy
   {
      Logger logger = Logger.getLogger(this.getClass());
      public CtClass getCached(String classname)
      {
         boolean trace = logger.isTraceEnabled();
         
         if (trace) logger.trace(this + " " + AbstractClassPool.this + " searching all pools for " + classname);
         
         CtClass clazz = getCachedLocally(classname);
         if (clazz == null && getClassLoader0() != null &&
               !isLocalResource(getResourceName(classname), trace) &&
               !generatedClasses.containsKey(classname))
         {
            Map<ClassLoader, ClassPool> registeredCLs = repository.getRegisteredCLs();
            synchronized (registeredCLs)
            {
               for(ClassPool pl : registeredCLs.values())
               {
                  AbstractClassPool pool = (AbstractClassPool) pl;
                  if (pool.isUnloadedClassLoader())
                  {
                     if (trace) logger.trace(this + " pool is unloaded " + pool);
                     repository.unregisterClassLoader(pool.getClassLoader());
                     continue;
                  }

                  //Do not check classpools for scoped classloaders
                  if (!pool.includeInGlobalSearch())
                  {
                     if (trace) logger.trace(this + " pool is scoped " + pool);
                     continue;
                  }

                  if (trace) logger.trace(this + " " + AbstractClassPool.this + " searching for " + classname + " in " + pool);
                  clazz = pool.getCachedLocally(classname);
                  if (clazz != null)
                  {
                     break;
                  }
               }
            }
         }
         if (trace) logger.trace(this + " " + AbstractClassPool.this + " found " + classname + " in pool" + getClassPoolLogStringForClass(clazz));
         return clazz;
      }
   }
   
   /**
    * Checks only the ClassPool's cache 
    */
   private class SearchLocalLoaderLoaderSearchStrategy implements ClassPoolSearchStrategy
   {
      Logger logger = Logger.getLogger(this.getClass());

      public CtClass getCached(String classname)
      {
         boolean trace = logger.isTraceEnabled();
         
         if (trace) logger.trace(this + " " + AbstractClassPool.this + " searching just this pool for " + classname);
         CtClass clazz = getCachedLocally(classname);
         if (trace) logger.trace(this + " " + AbstractClassPool.this + " found " + classname + " in pool" + getClassPoolLogStringForClass(clazz));
         return clazz;
      }
   }
}