/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.classpool.spi;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;

import javassist.ClassPool;
import javassist.scopedpool.ScopedClassPool;
import javassist.scopedpool.ScopedClassPoolFactory;
import javassist.scopedpool.ScopedClassPoolRepository;
import javassist.scopedpool.ScopedClassPoolRepositoryImpl;


/**
 * Singleton classpool repository.
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 99199 $
 */
public class ClassPoolRepository implements ScopedClassPoolRepository
{
   private final static ClassPoolRepository instance = new ClassPoolRepository();
   
   protected final ScopedClassPoolRepository delegate;
   
   private volatile List<ClassPoolRepositoryCallback> callbacks;
   
   private final  ConcurrentMap<ClassLoader, Boolean> currentClassLoaders = new ConcurrentHashMap<ClassLoader, Boolean>();
   
   /**
    * Returns the singleton instance.
    * 
    * @return the singleton repository instance
    */
   public static ClassPoolRepository getInstance()
   {
      return instance;
   }
   
   /**
    * Constructor.
    */
   protected ClassPoolRepository()
   {
      this.delegate = ScopedClassPoolRepositoryImpl.getInstance();
   }
   
   /**
    * Sets the classpool factory that should be used. This method should always
    * be called before the repository is used.
    */
   public void setClassPoolFactory(ScopedClassPoolFactory factory)
   {
      delegate.setClassPoolFactory(factory);
   }
   
   /**
    * Returns the classpool factory.
    */
   public ScopedClassPoolFactory getClassPoolFactory()
   {
      return delegate.getClassPoolFactory();
   }
   
   /**
    * Adds a callback object for notifications on classloader registration.
    *  
    * @param callback the callback object
    * @see ClassPoolRepositoryCallback
    * @throws IllegalArgumentException if the passed in callback was null
    */
   public void addClassPoolRepositoryCallback(ClassPoolRepositoryCallback callback)
   {
      if (callback == null)
         throw new IllegalArgumentException("Null callback");
      if (callbacks == null)
      {
         synchronized(this)
         {
            if (callbacks == null)
            {
               callbacks = new CopyOnWriteArrayList<ClassPoolRepositoryCallback>();
            }
         }
      }
      callbacks.add(callback);
   }
   
   /**
    * Returns the callback objects.
    * @return the callback objects in an unmodifiable list. If there are none an empty list is returned
    */
   public List<ClassPoolRepositoryCallback> getClassPoolRepositoryCallbacks()
   {
      List<ClassPoolRepositoryCallback> cbs = callbacks;
      if (cbs == null || cbs.size() == 0)
         return Collections.<ClassPoolRepositoryCallback>emptyList();
      return Collections.unmodifiableList(callbacks);
   }
   
   /**
    * Removes a callback object. If the callback object is null, or it cannot be found in the list, no error is thrown.
    * @param callback the callback object
    */
   public boolean removeClassPoolRepositoryCallback(ClassPoolRepositoryCallback callback)
   {
      if (callback == null)
         return false;
      if (callbacks == null)
         return false;
      return callbacks.remove(callback);
   }

   public boolean isPrune()
   {
      return delegate.isPrune();
   }

   public void setPrune(boolean prune)
   {
      delegate.setPrune(prune);
   }

   /**
    * Creates a ClassPool corresponding to {@code classLoader}.
    * 
    * @param classLoader the classLoader corresponding to the created ClassPool
    * @param parent the parent of the class pool to be created.
    * @return the created ClassPool.
    */
   public ScopedClassPool createScopedClassPool(ClassLoader classLoader, ClassPool parent)
   {
      return delegate.createScopedClassPool(classLoader, parent);
   }

   /**
    * Finds the ClassPool corresponding to {@code classLoader}
    * 
    * @param classLoader the classLoader
    * @return the ClassPoool that corresponds to {@classLoader}
    */
   public ClassPool findClassPool(ClassLoader classLoader)
   {
      return delegate.findClassPool(classLoader);
   }
   
   /**
    * Get the registered classloaders
    * 
    * @return the registered classloaders
    */
   @SuppressWarnings("unchecked")
   public Map<ClassLoader, ClassPool> getRegisteredCLs()
   {
      return delegate.getRegisteredCLs();
   }

   /**
    * This method will check to see if a register classloader has been undeployed (as in JBoss)
    */
   public void clearUnregisteredClassLoaders()
   {
      delegate.clearUnregisteredClassLoaders();
   }
   
   /**
    * Registers {@code classLoader} and returns the corresponding {@code ClassPool}.
    * 
    * @param classLoader the ClassLoader to be added to this repository
    * @return the ClassPool corresponding to {@code classLoader}
    */
   public ClassPool registerClassLoader(ClassLoader classLoader)
   {
      if (classLoader == null)
      {
         classLoader = SecurityActions.getContextClassLoader();
      }
      
      if (currentClassLoaders.putIfAbsent(classLoader, Boolean.TRUE) != null)
      {
         return null;
      }

      ScopedClassPool classPool = (ScopedClassPool) delegate.registerClassLoader(classLoader);
      currentClassLoaders.remove(classLoader);
      
      // TODO review classPool != null check for AOP tests
      if (callbacks != null && callbacks.size() > 0)
      {
         for (ClassPoolRepositoryCallback callback : callbacks)
         {
            callback.classLoaderRegistered(classLoader);
         }
      }
      return classPool;
   }

   /**
    * Unregisters {@code classLoader}.
    * 
    * @param classLoader the ClassLoader to be removed from this repository
    */
   public void unregisterClassLoader(ClassLoader classLoader)
   {
      delegate.unregisterClassLoader(classLoader);
      if (callbacks != null && callbacks.size() > 0)
      {
         for (ClassPoolRepositoryCallback callback : callbacks)
         {
            callback.classLoaderUnregistered(classLoader);
         }
      }
   }
}