/*
 * JBoss, a division of Red Hat
 * Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ha.client.loadbalance;

import java.util.List;

import org.jboss.ha.framework.interfaces.FamilyClusterInfo;



/**
 * LoadBalancePolicy implementation that, once a target is chosen, always favors 
 * that same target; i.e. no further load balancing occurs. Useful in cases where
 * "sticky session" behavior is desired. Initially the favored 
 * target is randomly selected, so different instances of this class
 * will likely not all select the same target; thus load balancing does occur
 * across different proxies. Favoring the same target does not 
 * mean that fail-over will not occur if the chosen target dies. In this case, 
 * fail-over will occur, and a new favored target will be randomly chosen and 
 * invocation will continously be invoked on this new target until its death.
 * <p> 
 * Each proxy using this policy will choose its own
 * preferred target: the target is not shared accross the proxy family. For 
 * shared behaviour please take a look at 
 * {@link FirstAvailableIdenticalAllProxies}.
 * </p>
 * 
 * @author <a href="mailto:bill@burkecentral.com">Bill Burke</a>.
 * @author <a href="mailto:sacha.labourey@cogito-info.ch">Sacha Labourey</a>.
 * @author brian.stansberry@jboss.com
 * 
 * @version $Revision: 57642 $
 * 
 * @see org.jboss.ha.client.LoadBalancePolicy
 */

public class FirstAvailable implements LoadBalancePolicy
{
   // Constants -----------------------------------------------------
   
   private static final long serialVersionUID = 2008524502721775114L;

   // Attributes ----------------------------------------------------
   
   private Object electedTarget = null;
   
   // Static --------------------------------------------------------
   
   // Constructors --------------------------------------------------
       
    // Public --------------------------------------------------------

   public Object chooseTarget (FamilyClusterInfo clusterFamily)
   {
      List targets = clusterFamily.getTargets ();
      if (targets.size () == 0)
         return null;

      if ( (this.electedTarget != null) && targets.contains (this.electedTarget) )
      {
         return this.electedTarget;
      }
      else
      {
         int cursor = RandomRobin.localRandomizer.nextInt(targets.size());
         this.electedTarget = targets.get(cursor);
         return this.electedTarget;
      }
   }
}
