/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.ha.framework.interfaces;

import java.util.List;

import org.jboss.ha.client.loadbalance.FirstAvailableIdenticalAllProxies;
import org.jboss.ha.client.loadbalance.RoundRobin;

/**
 * Maintain information for a given proxy family. Proxies can statically reference
 * objects implementing this interface: only the content will change as the 
 * cluster topology changes, not the FamilyClusterInfo object itself.
 * Proxies or LoadBalancing policy implementations can use the cursor and object
 * attribute to store arbitrary data that is then shared accross all proxies belonging
 * to the same family. 
 * Initial access to this object is done through the 
 * {@link ClusteringTargetsRepository} singleton.
 *
 * @see org.jboss.ha.framework.interfaces.FamilyClusterInfoImpl
 * @see org.jboss.ha.framework.interfaces.ClusteringTargetsRepository
 *
 * @author  <a href="mailto:sacha.labourey@cogito-info.ch">Sacha Labourey</a>.
 * @version $Revision: 69400 $
 *
 * <p><b>Revisions:</b>
 *
 * <p><b>2002/08/23, Sacha Labourey:</b>
 * <ul>
 * <li> First implementation </li>
 * </ul>
 */

public interface FamilyClusterInfo
{
   public String getFamilyName ();
   
   /**
    * Gets the list of targets for this family.
    * 
    * <strong>NOTE:</strong> Implementations should synchronize on themselves 
    * when executing this method (see JBAS-2071).
    */
   public List getTargets ();
   
   /**
    * Gets a shorthand identifier for the current cluster topology for
    * the clustered service represented by this object. The identifer is
    * generated on the server side; clients can pass this view id to the
    * server side to check whether their current view of the service
    * topology matches what the server sees.
    */
   public long getCurrentViewId ();
   
   /**
    * Remove the given target from the list of targets.
    * 
    * <strong>NOTE:</strong> Implementations should synchronize on themselves 
    * when executing this method (see JBAS-2071).
    * 
    * @param target the target
    * @return the updated list of targets
    */
   public List removeDeadTarget(Object target);
   
   /**
    * Updates the targets and the view id.
    * 
    * <strong>NOTE:</strong> Implementations should synchronize on themselves 
    * when executing this method (see JBAS-2071).
    */
   public List updateClusterInfo (List targets, long viewId);
   
   /**
    * Gets whether this object believes its current 
    * {@link #getTargets() target list} matches what the server-side
    * used to generate the {@link #getCurrentViewId() view id}.
    * <p>
    * Generally, a call to {@link #removeDeadTarget(Object)} would cause
    * this method to return <code>false</code> until a subsequent call
    * to {@link #updateClusterInfo(List, long)} is made.
    * </p>
    */
   public boolean currentMembershipInSyncWithViewId();
   
   /**
    * Force a reload of the view at the next invocation.
    * 
    * <strong>NOTE:</strong> Implementations should synchronize on themselves 
    * when executing this method (see JBAS-2071).
    */
   public void resetView ();
   
   /**
    * Gets a cursor indicating a position in the current 
    * {@link #getTargets() list of targets}. Useful for round-robin load
    * balancing strategies that wish to coordinate between instances.
    * <p>
    * It is not guaranteed that the returned value will be less than the size
    * of the target list; callers need to verify that themselves.
    * </p>
    * 
    * @see #UNINITIALIZED_CURSOR
    * @see RoundRobin
    */
   public int getCursor();
   
   /**
    * Sets a cursor indicating a position in the current 
    * {@link #getTargets() list of targets}. Useful for round-robin load
    * balancing strategies that wish to coordinate between instances.
    * 
    * @param
    */
   public int setCursor (int cursor);
   
   /**
    * Gets an object that may be one of the members of the current
    * {@link #getTargets() list of targets}. Useful for "sticky" load
    * balancing strategies that wish to coordinate between instances.
    * <p>
    * It is not guaranteed that the returned object will be a current member of
    * the target list; callers need to verify that themselves.
    * </p>
    * 
    * @return an object that *may* be an element in the target list, or 
    *         may be <code>null</code>
    *         
    * @see FirstAvailableIdenticalAllProxies       
    */
   public Object getObject();
   
   /**
    * Sets the member of the current {@link #getTargets() list of targets}
    * that "sticky" load balancing strategies that wish to coordinate between 
    * instances should use.
    * 
    * @param  whatever *should* be a member of the current list of targets, or
    *                  <code>null</code>. However, this requirement is not
    *                  enforced.      
    */
   public Object setObject(Object whatever);
   
   /**
    * Initial value returned by {@link #getCursor()}.
    */
   public final static int UNINITIALIZED_CURSOR = 999999999;
}
