/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.ha.client.loadbalance.aop;

import java.util.List;

import org.jboss.aop.joinpoint.Invocation;
import org.jboss.aspects.remoting.ClusterConstants;
import org.jboss.ha.client.loadbalance.AopLoadBalancePolicy;
import org.jboss.ha.framework.interfaces.FamilyClusterInfo;


/**
 * LoadBalancePolicy implementation that, once a target is chosen, always favors 
 * that same target; i.e. no further load balancing occurs. Useful in cases where
 * "sticky session" behavior is desired. Differs from the 
 * {@link org.jboss.ha.client.loadbalance.FirstAvailable superclass} in the way
 * that the {@link #chooseTarget(FamilyClusterInfo, Invocation)} method 
 * initially selects the target. Instead of always randomly selecting the target,
 * the method will check the passed invocation's metadata for the 
 * {@link ClusterConstants#HA_TARGET} key in the 
 * {@link ClusterConstants#CLUSTERED_REMOTING} group.  If an object is found 
 * there and it is a valid target associated with the given FamilyClusterInfo,
 * that target will be used. Otherwise the target will be randomly chosen.
 * This mechanism allows the proxy that contains this policy to specify the 
 * preferred target via invocation metadata.
 * <p> 
 * Each proxy using this policy will choose its own
 * preferred target: the target is not shared accross the proxy family. For 
 * shared behaviour please take a look at 
 * {@link FirstAvailableIdenticalAllProxies}.
 * </p>
 * 
 * @author Brian Stansberry
 *  
 * @version $Revision: 57642 $
 */
public class FirstAvailable 
extends org.jboss.ha.client.loadbalance.FirstAvailable
implements AopLoadBalancePolicy
{
   // Constants -----------------------------------------------------
   
   private static final long serialVersionUID = 2008524502721775114L;

   // Static --------------------------------------------------------
   
   // Constructors --------------------------------------------------
       
    // Public --------------------------------------------------------

   public Object chooseTarget (FamilyClusterInfo clusterFamily, Invocation invocation)
   {
      List targets = clusterFamily.getTargets ();
      if (targets.size () == 0)
         return null;

      if ( (this.electedTarget != null) && targets.contains (this.electedTarget) )
      {
         return this.electedTarget;
      }
      else
      {
         this.electedTarget = electTarget(targets, invocation);
         return this.electedTarget;
      }
   }

   private Object electTarget(List targets, Invocation invocation)
   {
      Object target = invocation.getMetaData(ClusterConstants.CLUSTERED_REMOTING, ClusterConstants.HA_TARGET);
      if (target == null || !targets.contains(target))
         target = electTarget(targets);
      return target;
   }
}
