/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.ha.framework.server;

import java.util.Hashtable;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NameNotFoundException;
import javax.naming.NamingException;

import org.jboss.cache.CacheManager;
import org.jboss.ha.framework.interfaces.HAPartition;
import org.jboss.logging.Logger;

/**
 * Service Locator utility for locating a {@link PojoCacheManager}. Maintains
 * an internal ref to a manager, and if it that is null,
 * will attempt to find it in JNDI using a standard naming pattern.
 * 
 * @author <a href="brian.stansberry@jboss.com">Brian Stansberry</a>
 * @version $Revision: 1.1 $
 */
public class CacheManagerLocator
{
   private static final Logger log = Logger.getLogger(CacheManagerLocator.class);
   
   private static CacheManagerLocator sharedInstance = new CacheManagerLocator();
   
   /** Name of the standard JNDI context under which HAPartitions are bound */
   public static final String STANDARD_JNDI_NAME = "java:CacheManager";
   
   private CacheManager registeredManager;
   
   /**
    * Gets the 
    * @return the shared CacheManagerLocator; will not be <code>null</code>
    */
   public static CacheManagerLocator getCacheManagerLocator()
   {
      return sharedInstance;
   }
   
   /**
    * Allows replacement of the default implementation.
    * 
    * @param shared the locator to use. Cannot be <code>null</code>.
    */
   protected static void setSharedInstance(CacheManagerLocator shared)
   {
      if (shared == null)
         throw new IllegalArgumentException("shared cannot be null");
      
      sharedInstance = shared;
   }
   
   /**
    * Allow subclasses to create a new HAPartitionLocator.
    */
   protected CacheManagerLocator()
   {      
   }
   
   /**
    * Locates and returns the {@link HAPartition} whose partition name matches
    * the given <code>partitionName</code>.
    * 
    * @param jndiProperties any naming properties to pass to new InitialContext() 
    *                       if JNDI lookup is needed.
    * @return the partition. Will not return <code>null</code>
    * 
    * @throws IllegalStateException if no cache manager can be found
    */
   public CacheManager getCacheManager(Hashtable jndiProperties)
   {
      CacheManager manager = registeredManager;
      if (manager == null)
      {
         try
         {
            manager = findInJndi(jndiProperties);
         }
         catch (NamingException e)
         {
            log.error("Problem finding CacheManager in JNDI", e);        
         }
      }
      
      if (manager == null)
         throw new IllegalStateException("CacheManager not found");
      
      return manager;
   }
   
   /**
    * Register a CacheManager with this locator.
    * 
    * @param partition the partition
    */
   public void registerCacheManager(CacheManager manager)
   {
      if (manager != null)
      {
         if (registeredManager != null && manager != registeredManager)
            throw new IllegalStateException("CacheManager already registered");
         registeredManager = manager;
      }
   }
   
   public CacheManager getDirectlyRegisteredManager()
   {
      return registeredManager;
   }
   
   /**
    * Deregister a partition from this locator.
    * 
    * @param partition the partition
    */
   public void deregisterCacheManager()
   {
      registeredManager = null;
   }
   
   /**
    * Find the given CacheManager in JNDI under a 
    * {@link #STANDARD_JNDI_NAME standard binding}.
    * 
    * @param jndiProperties any naming properties to pass to new InitialContext() 
    * 
    * @return the CacheManager, or <code>null</code>
    * 
    * @throws NamingException if there is a problem with the naming context. Will
    *                         not throw NameNotFoundException; if not found, will
    *                         return <code>null</code>.
    *                         
    * @see #STANDARD_JNDI_NAME
    */
   protected CacheManager findInJndi(Hashtable jndiProperties) throws NamingException
   {
      try
      {
         Context ctx = new InitialContext(jndiProperties);
         return (CacheManager) ctx.lookup(STANDARD_JNDI_NAME);
      }
      catch (NameNotFoundException e)
      {
         // just not there
         return null;
      }     
      
   }
}
