/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.ha.framework.interfaces;

import java.io.Serializable;
import java.util.List;
import java.util.Vector;

/** 
 *   HAPartition is an abstraction of the communication framework, providing access
 *   to clustered services such as Distributed State, Distributed Replicant Manager
 *   and RPC.
 *  
 *   @author  <a href="mailto:bill@burkecentral.com">Bill Burke</a>.
 *   @author  <a href="mailto:sacha.labourey@cogito-info.ch">Sacha Labourey</a>.
 *   @author  <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a>
 *   @version $Revision: 104233 $
 *
 * <p><b>Revisions:</b><br>
 * <p><b>28.07.2002 - Sacha Labourey:</b>
 * <ul>
 * <li> Added network-partition merge callback for listeners</li>
 * </ul>
 */

public interface HAPartition
{
   // *******************************
   // *******************************
   // Partition information accessors
   // *******************************
   // *******************************
   
   /**
    * Return the name of this node in the current partition. The name is
    * dynamically determined by the partition. The name will be the String
    * returned by <code>getClusterNode().getName()</code>.
    * 
    * @return The node name
    * 
    * @see #getClusterNode()
    */   
   public String getNodeName();
   /**
    * The name of the partition.
    *
    * @return Name of the current partition
    */   
   public String getPartitionName();

   /**
    * Accessor to the DistributedReplicantManager (DRM) that is linked to this partition.
    * @return the DistributedReplicantManager
    */   
   public DistributedReplicantManager getDistributedReplicantManager();
   
   /**
    * Accessor to the DistributedState (DS) that is linked to this partition.
    * @return the DistributedState service
    * 
    * @deprecated Use JBoss Cache for distributed caching
    */   
   public DistributedState getDistributedStateService ();

   // ***************************
   // ***************************
   // RPC multicast communication
   // ***************************
   // ***************************
   
   /**
    * Gets the default period, in ms, that the various <code>callMethodOnXXX</code>
    * methods that don't specify a <code>methodTimeout</code> parameter will wait for a response.
    */
   public long getMethodCallTimeout();
   
   /**
    * Register an object upon which RPCs associated with the given serviceName
    * will be invoked.
    * The partition receives RPC calls from other nodes in the cluster and demultiplexes
    * them, according to a service name, to a particular service. Consequently, each
    * service must first subscribe with a particular service name in the partition. The subscriber
    * does not need to implement any specific interface: the call is handled
    * dynamically through reflection.
    * @param serviceName Name of the subscribing service (demultiplexing key)
    * @param handler object to be called when receiving a RPC for its key.
    */   
   public void registerRPCHandler(String serviceName, Object handler);
   
   /**
    * Register an object upon which RPCs associated with the given serviceName
    * will be invoked.
    * The partition receives RPC calls from other nodes in the cluster and demultiplexes
    * them, according to a service name, to a particular service. Consequently, each
    * service must first subscribe with a particular service name in the partition. The subscriber
    * does not need to implement any specific interface: the call is handled
    * dynamically through reflection.
    * In cases where the client is using a scoped classloader, the client will need to provide a 
    * reference to the classloader if the service's RPC calls use custom parameter or response object types.
    * The classloader will be used to deserialize the RPC and/or response. 
    * @param serviceName Name of the subscribing service (demultiplexing key)
    * @param handler object to be called when receiving a RPC for its key.
    * @param classloader ClassLoader to be used when marshalling and unmarshalling RPC requests and responses.
    */   
   public void registerRPCHandler(String serviceName, Object handler, ClassLoader classloader);
   
   /**
    * Unregister the service from the partition
    * @param serviceName Name of the service key (on which the demultiplexing occurs)
    * @param subscriber The target object that unsubscribes
    */   
   public void unregisterRPCHandler(String serviceName, Object subscriber);

   
   /**
    * Invoke an RPC call on all nodes of the partition/cluster and return their
    * response values as a list. This convenience method is equivalent to 
    * {@link #callMethodOnCluster(String, String, Object[], Class[], Class, boolean, ResponseFilter, long, boolean) 
    * callAsynchMethodOnCluster(serviceName, methodName, args, types, Object.class, excludeSelf, null, methodTimeout, false)} 
    * where <code>methodTimeout</code> is the value returned by {@link #getMethodCallTimeout()}.
    * 
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param excludeSelf <code>false</code> if the RPC must also be made on the current
    * node of the partition, <code>true</code> if only on remote nodes
    *
    * @return an array of responses from nodes that invoked the RPC
    */
   public List<?> callMethodOnCluster(String serviceName, String methodName,
         Object[] args, Class<?>[] types, boolean excludeSelf) throws InterruptedException;
   
   /**
    * Invoke a synchronous RPC call on all nodes of the partition/cluster and return their
    * response values as a list. This convenience method is equivalent to 
    * {@link #callMethodOnCluster(String, String, Object[], Class[], boolean, ResponseFilter, boolean) 
    * callAsynchMethodOnCluster(serviceName, methodName, args, types, Object.class, excludeSelf, filter, methodTimeout, false)} 
    * where <code>methodTimeout</code> is the value returned by {@link #getMethodCallTimeout()}.
    * 
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param excludeSelf <code>false</code> if the RPC must also be made on the current
    * node of the partition, <code>true</code> if only on remote nodes
    * @param filter response filter instance which allows for early termination 
    * of the synchronous RPC call. Can be <code>null</code>.
    * 
    * @return an array of responses from remote nodes
    */
   public List<?> callMethodOnCluster(String serviceName, String methodName,
         Object[] args, Class<?>[] types, boolean excludeSelf, ResponseFilter filter) throws InterruptedException;
   
   /**
    * Invoke an RPC call on all nodes of the partition/cluster and return their
    * response values as a list.
    * 
    * @param T the expected type of the return values
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param returnType the expected type of the return values, null or 
    *     <code>void.class</code> if no return values are expected
    * @param excludeSelf <code>false</code> if the RPC must also be made on the current
    * node of the partition, <code>true</code> if only on remote nodes
    * @param filter response filter instance which allows for early termination 
    * of the RPC call once acceptable responses are received. Can be <code>null</code>, 
    * in which the call will not return until all nodes have responded.
    * @param methodTimeout max number of ms to wait for response to arrive before
    *                      returning
    * @param unordered <code>true</code> if the HAPartition isn't required to ensure
    * that this RPC is invoked on all nodes in a consistent order with respect
    * to other RPCs originated by the same node
    * @return a list of responses from remote nodes
    */
   public <T> List<T> callMethodOnCluster(String serviceName, String methodName,
         Object[] args, Class<?>[] types, Class<T> returnType, boolean excludeSelf, 
         ResponseFilter filter, long methodTimeout, boolean unordered) throws InterruptedException;

   /**
    * Invoke an RPC call on all nodes of the partition/cluster without waiting
    * for any responses. The call will return immediately after sending a message
    * to the cluster telling nodes to invoke the RPC and will not wait for the nodes to answer. Thus
    * no return values are available. This convenience method is equivalent to 
    * {@link #callAsynchMethodOnCluster(String, String, Object[], Class[], boolean, boolean) 
    * callAsynchMethodOnCluster(serviceName, methodName, args, types, excludeSelf, false)}.
    * 
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param excludeSelf <code>false</code> if the RPC must also be made on the current
    * node of the partition, <code>true</code> if only on remote nodes
    */   
   public void callAsynchMethodOnCluster (String serviceName, String methodName,
         Object[] args, Class<?>[] types, boolean excludeSelf) throws InterruptedException;
   
   /**
    * Invoke an RPC call on all nodes of the partition/cluster without waiting
    * for any responses. The call will return immediately after sending a message
    * to the cluster telling nodes to invoke the RPC and will not wait for the nodes to answer. 
    * Thus no return values are available. 
    * 
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param excludeSelf <code>false</code> if the RPC must also be made on the current
    * node of the partition, <code>true</code> if only on remote nodes
    * @param unordered <code>true</code> if the HAPartition isn't required to ensure
    * that this RPC is invoked on all nodes in a consistent order with respect
    * to other RPCs originated by the same node
    */   
   public void callAsynchMethodOnCluster(String serviceName, String methodName,
         Object[] args, Class<?>[] types, boolean excludeSelf, boolean unordered) throws InterruptedException;

   /**
    * Calls method on Cluster coordinator node only.  The cluster coordinator 
    * node is the first node in the current cluster view. This convenience method is equivalent to 
    * {@link #callMethodOnCoordinatorNode(String, String, Object[], Class[], Class, boolean, long, boolean) 
    * callMethodOnCoordinatorNode(serviceName, methodName, args, types, Object.class, excludeSelf, methodTimeout, false)} 
    * where <code>methodTimeout</code> is the value returned by {@link #getMethodCallTimeout()}.
    * 
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param excludeSelf <code>true</code> if the RPC should not be made on the 
    * current node even if the current node is the coordinator
    * 
    * @return the value returned by the target method
    */
   public Object callMethodOnCoordinatorNode(String serviceName, String methodName,
             Object[] args, Class<?>[] types, boolean excludeSelf) throws Exception;

   /**
    * Calls method on Cluster coordinator node only.  The cluster coordinator 
    * node is the first node in the current cluster view.
    * 
    * @param T the expected type of the return value
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param returnType the expected type of the return value, <code>null</code>
    *                   or <code>void.class</code> if no return value is expected
    * @param excludeSelf <code>true</code> if the RPC should not be made on the 
    * current node even if the current node is the coordinator
    * @param methodTimeout max number of ms to wait for response to arrive before
    *                      returning
    * @param unordered <code>true</code> if the HAPartition isn't required to ensure
    * that this RPC is invoked on all nodes in a consistent order with respect
    * to other RPCs originated by the same node
    * @return the value returned by the target method
    */
   public <T> T callMethodOnCoordinatorNode(String serviceName, String methodName,
             Object[] args, Class<?>[] types, Class<T> returnType, boolean excludeSelf, 
             long methodTimeout, boolean unordered) throws Exception;


   /**
    * Calls method on target node only. This convenience method is equivalent to 
    * {@link #callMethodOnNode(String, String, Object[], Class[], Class, long, ClusterNode, boolean) 
    * callMethodOnNode(serviceName, methodName, args, types, Object.class, methodTimeout, targetNode, false)}
    * where <code>methodTimeout</code> is the value returned by {@link #getMethodCallTimeout()}.
    * 
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param methodTimeout max number of ms to wait for response to arrive before
    *                      returning
    * @param targetNode is the target of the call
    * 
    * @return the value returned by the target method
    */
   public Object callMethodOnNode(String serviceName, String methodName,
           Object[] args, Class<?>[] types, ClusterNode targetNode) throws Exception;


   /**
    * Calls method on target node only. This convenience method is equivalent to 
    * {@link #callMethodOnNode(String, String, Object[], Class[], Class, long, ClusterNode, boolean) 
    * callMethodOnNode(serviceName, methodName, args, types, Object.class, methodTimeout, targetNode, false)}.
    * 
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param methodTimeout max number of ms to wait for response to arrive before
    *                      returning
    * @param targetNode is the target of the call
    * 
    * @return the value returned by the target method
    */
   public Object callMethodOnNode(String serviceName, String methodName,
           Object[] args, Class<?>[] types, long methodTimeout, ClusterNode targetNode) throws Exception;


   /**
    * Calls method synchronously on target node only.
    * 
    * @param T the expected type of the return value
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param returnType the expected type of the return value, <code>null</code>
    *                   or <code>void.class</code> if no return value is expected
    * @param methodTimeout max number of ms to wait for response to arrive before
    *                      returning
    * @param targetNode is the target of the call
    * @param unordered <code>true</code> if the HAPartition isn't required to ensure
    * that this RPC is invoked on all nodes in a consistent order with respect
    * to other RPCs originated by the same node
    * @return the value returned by the target method
    */
   public <T> T callMethodOnNode(String serviceName, String methodName,
          Object[] args, Class<?>[] types, Class<T> returnType, 
          long methodTimeout, ClusterNode targetNode, boolean unordered) throws Exception;

   /**
    * Calls method on target node only. The call will return immediately and 
    * will not wait for the node to answer. Thus no answer is available. This 
    * convenience method is equivalent to 
    * {@link #callAsynchMethodOnNode(String, String, Object[], Class[], ClusterNode, boolean) 
    * callAsynchMethodOnCluster(serviceName, methodName, args, types, methodTimeout, targetNode, false)}.
    * 
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param targetNode is the target of the call
     */
   public void callAsyncMethodOnNode(String serviceName, String methodName,
           Object[] args, Class<?>[] types, ClusterNode targetNode) throws Exception;

   /**
    * Calls method on target node only. The call will return immediately and 
    * will not wait for the node to answer. Thus no answer is available.
    * 
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param targetNode is the target of the call
    * @param unordered <code>true</code> if the HAPartition isn't required to ensure
    * that this RPC is invoked on all nodes in a consistent order with respect
    * to other RPCs originated by the same node
    */
   public void callAsyncMethodOnNode(String serviceName, String methodName,
           Object[] args, Class<?>[] types, ClusterNode targetNode, boolean unordered) throws Exception;

   /**
    * Calls method on Cluster coordinator node only.  The cluster coordinator 
    * node is the first node in the current cluster view.  The call will return immediately and 
    * will not wait for the node to answer. Thus no answer is available. This convenience method is equivalent to 
    * {@link #callAsyncMethodOnCoordinatorNode(String, String, Object[], Class[], boolean, boolean) 
    * callMethodOnCoordinatorNode(serviceName, methodName, args, types, excludeSelf, false)}}.
    * 
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param excludeSelf <code>true</code> if the RPC should not be made on the 
    * current node even if the current node is the coordinator
    */
   public void callAsyncMethodOnCoordinatorNode(String serviceName, String methodName,
             Object[] args, Class<?>[] types, boolean excludeSelf) throws Exception;

   /**
    * Calls method on Cluster coordinator node only.  The cluster coordinator 
    * node is the first node in the current cluster view. The call will return immediately and 
    * will not wait for the node to answer. Thus no answer is available. 
    * 
    * @param serviceName name of the target service name on which calls are invoked
    * @param methodName name of the Java method to be called on remote services
    * @param args array of Java Object representing the set of parameters to be
    * given to the remote method
    * @param types types of the parameters
    * @param excludeSelf <code>true</code> if the RPC should not be made on the 
    * current node even if the current node is the coordinator
    * @param unordered <code>true</code> if the HAPartition isn't required to ensure
    * that this RPC is invoked on all nodes in a consistent order with respect
    * to other RPCs originated by the same node
    * @return the value returned by the target method
    */
   public void callAsyncMethodOnCoordinatorNode(String serviceName, String methodName,
             Object[] args, Class<?>[] types, boolean excludeSelf, boolean unordered) throws Exception;

   // *************************
   // *************************
   // State transfer management
   // *************************
   // *************************
   
   
   /**
    * State management is highly important for clustered services. Consequently, services that wish to manage their state
    * need to subscribe to state transfer events. When a service is started on a cluster node, state is pushed from another node to the  
    * new node.  When another node starts, the node may be asked to provide its state to initialise the newly started node.
    */   
   public interface HAPartitionStateTransfer
   {
      /**
       * Called when a new node needs to be initialized. This is called on any existing node to determine a current state for this service.
       * @return A serializable representation of the state
       */      
      public Serializable getCurrentState ();
      /**
       * This callback method is called when a new service starts on a new node; the state that it should hold is transfered to it through this callback.
       * @param newState The serialized representation of the state of the new service.
       */      
      public void setCurrentState(Serializable newState);
   }
   
   /**
    * Register a service that will participate in state transfer protocol and receive callbacks
    * @param serviceName Name of the service that subscribes for state transfer events. This name must be identical for all identical services in the cluster.
    * @param subscriber Object implementing {@link HAPartitionStateTransfer} and providing or receiving state transfer callbacks
    */   
   public void subscribeToStateTransferEvents (String serviceName, HAPartition.HAPartitionStateTransfer subscriber);
   
   /**
    * Unregister a service from state transfer callbacks.
    * @param serviceName Name of the service that participates in the state transfer protocol
    * @param subscriber Service implementing the state transfer callback methods
    */   
   public void unsubscribeFromStateTransferEvents (String serviceName, HAPartition.HAPartitionStateTransfer subscriber);

   // *************************
   // *************************
   // Group Membership listeners
   // *************************
   // *************************
   
   /**
    * Listener for notifications issued when a new node joins the cluster or an 
    * existing node leaves the cluster (or simply dies).
    */   
   public interface HAMembershipListener
   {
      /** Called when a partition topology change occurs. This callback may be
       * made using the thread that carries messages up from the network and so you 
       * should not execute new cluster calls using this thread. If you need to 
       * do that implement the 
       * {@link AsynchHAMembershipListener asynchronous version of this interface}.
       *
       * @param deadMembers A list of nodes that have died since the previous view
       * @param newMembers A list of nodes that have joined the partition since the previous view
       * @param allMembers A list of nodes that built the current view
       */      
      public void membershipChanged(Vector<ClusterNode> deadMembers, Vector<ClusterNode> newMembers, Vector<ClusterNode> allMembers);
   }

   /** 
    * A tagging interface for HAMembershipListener implementations that require
    * notifications to be issued by a thread separate from the thread that 
    * carries messages up from the network. The ordering of view changes is 
    * preserved, but listeners are free to execute cluster calls.
    */
   public interface AsynchHAMembershipListener extends HAMembershipListener
   {
      // Nothing new
   }
   
   /** 
    * Extends HAMembershipListener to receive a specialized notification when a 
    * network-partition merge occurs. A merge occurs when two or more nodes that 
    * were unaware of each other for some reason (e.g. a crashed switch) detect 
    * each other and form a single group.
    */
   public interface HAMembershipExtendedListener extends HAPartition.HAMembershipListener
   {
      /** 
       * Specialized notification issued instead of 
       * {@link #membershipChanged(List, List, List) the standard one}
       * when a network-partition merge occurs. This notification is made
       * using the thread that carries messages up from the network and so you 
       * should not execute new cluster calls using this thread. If you need to 
       * do that implement the 
       * {@link AsynchHAMembershipExtendedListener asynchronous version of this interface}.
       *
       * @param deadMembers A list of nodes that have died since the previous view
       * @param newMembers A list of nodes that have joined the partition since the previous view
       * @param allMembers A list of nodes that built the current view
       * @param originatingGroups A list of nodes that were previously partioned and that are now merged
       */      
      public void membershipChangedDuringMerge(Vector<ClusterNode> deadMembers, Vector<ClusterNode> newMembers,
            Vector<ClusterNode> allMembers, Vector<List<ClusterNode>> originatingGroups);
   }

   /** 
    * A tagging interface for HAMembershipExtendedListener callbacks that will
    * be performed in a thread separate from the thread that carries messages 
    * up from the network. The ordering of view changes is preserved, but 
    * listeners are free to execute cluster calls.
    */
   public interface AsynchHAMembershipExtendedListener extends HAMembershipExtendedListener
   {
      // Nothing new
   }

   /**
    * Subscribes to receive {@link HAMembershipListener} events.
    * @param listener The membership listener object
    */   
   public void registerMembershipListener(HAMembershipListener listener);
   
   /**
    * Unsubscribes from receiving {@link HAMembershipListener} events.
    * @param listener The listener wishing to unsubscribe
    */ 
   public void unregisterMembershipListener(HAMembershipListener listener);
   
   /**
    * Returns whether this partition will synchronously notify any 
    * HAMembershipListeners of membership changes using the calling thread
    * from the underlying <code>ClusterPartition</code>.
    * 
    * @return <code>true</code> if registered listeners that don't implement
    *         <code>AsynchHAMembershipExtendedListener</code> or
    *         <code>AsynchHAMembershipListener</code> will be notified
    *         synchronously of membership changes; <code>false</code> if
    *         those listeners will be notified asynchronously.  Default
    *         is <code>false</code>.
    */
   public boolean getAllowSynchronousMembershipNotifications();

   /**
    * Sets whether this partition will synchronously notify any 
    * HAMembershipListeners of membership changes using the calling thread
    * from the underlying <code>ClusterPartition</code>.
    * 
    * @param allowSync  <code>true</code> if registered listeners that don't 
    *         implement <code>AsynchHAMembershipExtendedListener</code> or
    *         <code>AsynchHAMembershipListener</code> should be notified
    *         synchronously of membership changes; <code>false</code> if
    *         those listeners can be notified asynchronously.  Default
    *         is <code>false</code>.
    */
   public void setAllowSynchronousMembershipNotifications(boolean allowSync);

   /**
    * Each time the partition topology changes, a new view is computed. A view is a list of members,
    * the first member being the coordinator of the view. Each view also has a distinct identifier.
    * @return The identifier of the current view
    */   
   public long getCurrentViewId();

   /**
    * Return the list of member nodes that built the current view i.e., the current partition.
    * @return An array of Strings containing the node names
    */   
   public Vector<String> getCurrentView ();

   /**
    * Return the member nodes that built the current view i.e., the current partition.
    * @return   An array of ClusterNode listing the current members of the partition.
    *           This array will be in the same order in all nodes in the cluster that
    *           have received the current view.
    */
   public ClusterNode[] getClusterNodes ();

   /**
    * Return member node for the current cluster node.
    * @return ClusterNode containing the current node name
    */
   public ClusterNode getClusterNode ();
}