package org.jboss.ha.framework.server.ispn;

import java.util.Properties;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import org.infinispan.Cache;
import org.infinispan.lifecycle.ComponentStatus;
import org.infinispan.manager.CacheContainer;
import org.infinispan.manager.EmbeddedCacheManager;
import org.jboss.ha.ispn.CacheContainerRegistry;
import org.jgroups.ChannelFactory;

@SuppressWarnings("deprecation")
public class HAPartitionCacheHandler<K, V> implements InfinispanHAPartitionCacheHandler<K, V>
{
   private final CacheContainerRegistry registry;
   private final AtomicReference<Cache<K, V>> cacheReference = new AtomicReference<Cache<K, V>>();
   private final AtomicInteger acquireCount = new AtomicInteger();
   
   private volatile String cacheContainerName;
   private volatile String cacheName;
   
   public HAPartitionCacheHandler(CacheContainerRegistry registry)
   {
      this.registry = registry;
   }
   
   /**
    * {@inheritDoc}
    * @see org.jboss.ha.framework.server.ispn.InfinispanHAPartitionCacheHandler#getCache()
    */
   @Override
   public Cache<K, V> getCache()
   {
      return this.cacheReference.get();
   }
   
   /**
    * {@inheritDoc}
    * @see org.jboss.ha.framework.server.spi.HAPartitionCacheHandler#acquireCache()
    */
   @Override
   public void acquireCache()
   {
      if (this.cacheReference.get() == null)
      {
         CacheContainer container = this.registry.getCacheContainer(this.cacheContainerName);
         
         String name = this.cacheName;
         
         this.cacheReference.compareAndSet(null, (name != null) ? container.<K, V>getCache(name) : container.<K, V>getCache());
      }
      
      this.acquireCount.incrementAndGet();
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.ha.framework.server.spi.HAPartitionCacheHandler#startCache()
    */
   @Override
   public void startCache()
   {
      Cache<K, V> cache = this.cacheReference.get();
      
      if (cache == null)
      {
         throw new IllegalStateException("You must first acquire a cache before starting it.");
      }
      
      if (cache.getStatus() != ComponentStatus.RUNNING)
      {
         cache.start();
      }
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.ha.framework.server.spi.HAPartitionCacheHandler#releaseCache()
    */
   @Override
   public void releaseCache()
   {
      int count = this.acquireCount.decrementAndGet();
      
      if (count == 0)
      {
         Cache<K, V> cache = this.cacheReference.getAndSet(null);
         
         if ((cache != null) && (cache.getStatus() == ComponentStatus.RUNNING))
         {
            cache.stop();
         }
      }
      else if (count < 0)
      {
         // Attempt to resolve
         this.acquireCount.compareAndSet(count, count + 1);
         
         throw new IllegalStateException("Attempt to release cache that was not acquired.");
      }
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.ha.framework.server.spi.HAPartitionCacheHandler#getChannelStackName()
    */
   @Override
   public String getChannelStackName()
   {
      return this.getTransportProperties().getProperty("stack");
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.ha.framework.server.spi.HAPartitionCacheHandler#getCacheChannelFactory()
    */
   @Override
   public ChannelFactory getCacheChannelFactory()
   {
      return (ChannelFactory) this.getTransportProperties().get("channelFactory");
   }
   
   private Properties getTransportProperties()
   {
      Cache<K, V> cache = this.cacheReference.get();
      
      if (cache == null)
      {
         throw new IllegalStateException("Must acquire cache before getting channel stack name");
      }
      
      CacheContainer container = cache.getCacheManager();
      
      if (!(container instanceof EmbeddedCacheManager))
      {
         throw new IllegalStateException(String.format("Cache container [%s] is not of the expected type: %s", container.getClass().getName(), EmbeddedCacheManager.class.getName()));
      }
      
      EmbeddedCacheManager manager = (EmbeddedCacheManager) container;
      
      return manager.getGlobalConfiguration().getTransportProperties();
   }
   
   /**
    * {@inheritDoc}
    * @see org.jboss.ha.framework.server.spi.HAPartitionCacheHandler#getCacheConfigName()
    */
   @Override
   public String getCacheConfigName()
   {
      return this.cacheName;
   }
   
   public void setCacheConfigName(String cacheName)
   {
      this.cacheName = cacheName;
   }
   
   public String getCacheContainerName()
   {
      return this.cacheContainerName;
   }
   
   public void setCacheContainerName(String cacheContainerName)
   {
      this.cacheContainerName = cacheContainerName;
   }
}
