/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.web.tomcat.service.session.distributedcache.ispn;

import java.io.Serializable;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;

import org.infinispan.Cache;
import org.infinispan.atomic.AtomicMap;
import org.infinispan.config.CacheLoaderManagerConfig;
import org.infinispan.context.Flag;
import org.infinispan.distribution.DistributionManager;
import org.infinispan.lifecycle.ComponentStatus;
import org.infinispan.manager.EmbeddedCacheManager;
import org.infinispan.notifications.Listener;
import org.infinispan.notifications.cachelistener.annotation.CacheEntryActivated;
import org.infinispan.notifications.cachelistener.annotation.CacheEntryModified;
import org.infinispan.notifications.cachelistener.annotation.CacheEntryRemoved;
import org.infinispan.notifications.cachelistener.event.CacheEntryActivatedEvent;
import org.infinispan.notifications.cachelistener.event.CacheEntryModifiedEvent;
import org.infinispan.notifications.cachelistener.event.CacheEntryRemovedEvent;
import org.infinispan.notifications.cachemanagerlistener.annotation.ViewChanged;
import org.infinispan.notifications.cachemanagerlistener.event.ViewChangedEvent;
import org.infinispan.remoting.transport.Address;
import org.jboss.ha.framework.server.lock.SharedLocalYieldingClusterLockManager;
import org.jboss.ha.framework.server.lock.TimeoutException;
import org.jboss.ha.ispn.atomic.AtomicMapFactory;
import org.jboss.ha.ispn.invoker.CacheInvoker;
import org.jboss.logging.Logger;
import org.jboss.web.tomcat.service.session.distributedcache.impl.IncomingDistributableSessionDataImpl;
import org.jboss.web.tomcat.service.session.distributedcache.spi.BatchingManager;
import org.jboss.web.tomcat.service.session.distributedcache.spi.DistributableSessionMetadata;
import org.jboss.web.tomcat.service.session.distributedcache.spi.IncomingDistributableSessionData;
import org.jboss.web.tomcat.service.session.distributedcache.spi.LocalDistributableSessionManager;
import org.jboss.web.tomcat.service.session.distributedcache.spi.OutgoingDistributableSessionData;
import org.jboss.web.tomcat.service.session.distributedcache.spi.SessionOwnershipSupport;

/**
 * Distributed cache manager implementation using Infinispan.
 * @author Paul Ferraro
 */
@Listener
public class DistributedCacheManager<T extends OutgoingDistributableSessionData> implements org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager<T>, SessionOwnershipSupport
{
   static String mask(String sessionId)
   {
      if (sessionId == null) return null;
      
      int length = sessionId.length();
      
      if (length <= 8) return sessionId;
      
      return sessionId.substring(0, 2) + "****" + sessionId.substring(length - 6, length);
   }

   static RuntimeException getRuntimeException(String message, Exception e)
   {   
      if (e instanceof RuntimeException) return (RuntimeException) e;
      
      return new RuntimeException(message != null ? message : e.toString(), e);
   }
   
   private static final Logger log = Logger.getLogger(DistributedCacheManager.class);
   
   private static Map<SharedLocalYieldingClusterLockManager.LockResult, LockResult> lockResultMap = lockResultMap();
   private static Map<SharedLocalYieldingClusterLockManager.LockResult, LockResult> lockResultMap()
   {
      Map<SharedLocalYieldingClusterLockManager.LockResult, LockResult> map = new EnumMap<SharedLocalYieldingClusterLockManager.LockResult, LockResult>(SharedLocalYieldingClusterLockManager.LockResult.class);
      map.put(SharedLocalYieldingClusterLockManager.LockResult.ACQUIRED_FROM_CLUSTER, LockResult.ACQUIRED_FROM_CLUSTER);
      map.put(SharedLocalYieldingClusterLockManager.LockResult.ALREADY_HELD, LockResult.ALREADY_HELD);
      map.put(SharedLocalYieldingClusterLockManager.LockResult.NEW_LOCK, LockResult.NEW_LOCK);
      return map;
   }
   
   private final LocalDistributableSessionManager manager;
   private final SharedLocalYieldingClusterLockManager lockManager;
   private final SessionAttributeStorage<T> attributeStorage;
   private final Cache<?, ?> cache;
   private final CacheInvoker invoker;
   private final BatchingManager batchingManager;
   private final boolean passivationEnabled;
   final AtomicMapFactory atomicMapFactory;
   
   // FIXME make configurable
   private long lockTimeout = 30000;
   
   public DistributedCacheManager(LocalDistributableSessionManager manager, Cache<?, ?> cache,
         SharedLocalYieldingClusterLockManager lockManager, SessionAttributeStorage<T> attributeStorage,
         BatchingManager batchingManager, CacheInvoker invoker, AtomicMapFactory atomicMapFactory)
   {
      this.manager = manager;
      this.lockManager = lockManager;
      this.cache = cache;
      this.attributeStorage = attributeStorage;
      this.batchingManager = batchingManager;
      this.invoker = invoker;
      this.atomicMapFactory = atomicMapFactory;

      CacheLoaderManagerConfig loaderManagerConfig = this.cache.getConfiguration().getCacheLoaderManagerConfig();

      this.passivationEnabled = (loaderManagerConfig != null) ? loaderManagerConfig.isPassivation().booleanValue() && !loaderManagerConfig.isShared().booleanValue() : false;
   }

   @SuppressWarnings("unchecked")
   Cache<String, AtomicMap<Object, Object>> getSessionCache()
   {
      return (Cache<String, AtomicMap<Object, Object>>) this.cache;
   }
   
   @SuppressWarnings("unchecked")
   Cache<Address, String> getJvmRouteCache()
   {
      return (Cache<Address, String>) this.cache;
   }
   
   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#start()
    */
   @Override
   public void start()
   {
      if (this.cache.getStatus() != ComponentStatus.RUNNING)
      {
         this.cache.start();
      }
      
      this.cache.addListener(this);
      
      if (this.cache.getConfiguration().getCacheMode().isDistributed())
      {
         EmbeddedCacheManager container = (EmbeddedCacheManager) this.cache.getCacheManager();
         
         container.addListener(this);
         
         this.getJvmRouteCache().putIfAbsent(container.getAddress(), this.manager.getJvmRoute());
      }
   }
   
   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#stop()
    */
   @Override
   public void stop()
   {
      if (this.cache.getConfiguration().getCacheMode().isDistributed())
      {
         EmbeddedCacheManager container = (EmbeddedCacheManager) this.cache.getCacheManager();
         
         this.getJvmRouteCache().remove(container.getAddress());
         
         container.removeListener(this);
      }
      
      this.cache.removeListener(this);
      
      this.cache.stop();
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#getBatchingManager()
    */
   @Override
   public BatchingManager getBatchingManager()
   {
      return this.batchingManager;
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#sessionCreated(java.lang.String)
    */
   @Override
   public void sessionCreated(String sessionId)
   {
      // Do nothing
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#storeSessionData(org.jboss.web.tomcat.service.session.distributedcache.spi.OutgoingDistributableSessionData)
    */
   @Override
   public void storeSessionData(T sessionData)
   {
      final String sessionId = sessionData.getRealId();
      
      if (log.isTraceEnabled())
      {
         log.trace("putSession(): putting session " + mask(sessionId));
      }     
      
      Operation<AtomicMap<Object, Object>> operation = new Operation<AtomicMap<Object, Object>>()
      {
         @Override
         public AtomicMap<Object, Object> invoke(Cache<String, AtomicMap<Object, Object>> cache)
         {
            return DistributedCacheManager.this.atomicMapFactory.getAtomicMap(cache, sessionId, true);
         }
      };
      
      AtomicMap<Object, Object> data = this.invoker.invoke(this.getSessionCache(), operation);
      
      AtomicMapEntry.VERSION.put(data, Integer.valueOf(sessionData.getVersion()));
      AtomicMapEntry.METADATA.put(data, sessionData.getMetadata());
      AtomicMapEntry.TIMESTAMP.put(data, sessionData.getTimestamp());
      
      this.attributeStorage.store(data, sessionData);
   }
   
   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#getSessionData(java.lang.String, boolean)
    */
   @Override
   public IncomingDistributableSessionData getSessionData(String sessionId, boolean initialLoad)
   {
      return this.getData(sessionId, true);
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#getSessionData(java.lang.String, java.lang.String, boolean)
    */
   @Override
   public IncomingDistributableSessionData getSessionData(final String sessionId, String dataOwner, boolean includeAttributes)
   {
      return (dataOwner == null) ? this.getData(sessionId, includeAttributes) : null;
   }
   
   private IncomingDistributableSessionData getData(final String sessionId, boolean includeAttributes)
   {
      Operation<AtomicMap<Object, Object>> operation = new Operation<AtomicMap<Object, Object>>()
      {
         @Override
         public AtomicMap<Object, Object> invoke(Cache<String, AtomicMap<Object, Object>> cache)
         {
            return DistributedCacheManager.this.atomicMapFactory.getAtomicMap(cache, sessionId, false);
         }
      };
      
      AtomicMap<Object, Object> data = this.invoker.invoke(this.getSessionCache(), operation);
      
      // If requested session is no longer in the cache; return null
      if (data == null) return null;
      
      try
      {
         Integer version = AtomicMapEntry.VERSION.get(data);
         Long timestamp = AtomicMapEntry.TIMESTAMP.get(data);
         DistributableSessionMetadata metadata = AtomicMapEntry.METADATA.get(data);
         IncomingDistributableSessionDataImpl result = new IncomingDistributableSessionDataImpl(version, timestamp, metadata);
         
         if (includeAttributes)
         {
            try
            {
               result.setSessionAttributes(this.attributeStorage.load(data));
            }
            catch (Exception e)
            {
               throw getRuntimeException("Failed to load session attributes for session: " + mask(sessionId), e);
            }
         }

         return result;
      }
      catch (Exception e)
      {
         String message = String.format("Problem accessing session [%s]: %s", mask(sessionId), e.toString());
         log.warn(message);
         log.debug(message, e);
         
         // Clean up
         this.removeSessionLocal(sessionId);
         
         return null;
      }
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#removeSession(java.lang.String)
    */
   @Override
   public void removeSession(final String sessionId)
   {
      this.removeSession(sessionId, false);
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#removeSessionLocal(java.lang.String)
    */
   @Override
   public void removeSessionLocal(final String sessionId)
   {
      this.removeSession(sessionId, true);
   }

   private void removeSession(final String sessionId, final boolean local)
   {
      Operation<AtomicMap<Object, Object>> operation = new Operation<AtomicMap<Object, Object>>()
      {
         @Override
         public AtomicMap<Object, Object> invoke(Cache<String, AtomicMap<Object, Object>> cache)
         {
            if (local)
            {
               cache.getAdvancedCache().withFlags(Flag.CACHE_MODE_LOCAL);
            }
            
            return cache.remove(sessionId);
         }
      };
      
      this.invoker.invoke(this.getSessionCache(), operation);
   }
   
   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#removeSessionLocal(java.lang.String, java.lang.String)
    */
   @Override
   public void removeSessionLocal(String sessionId, String dataOwner)
   {
      if (dataOwner == null)
      {
         this.removeSession(sessionId, true);
      }
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#evictSession(java.lang.String)
    */
   @Override
   public void evictSession(final String sessionId)
   {
      Operation<Void> operation = new Operation<Void>()
      {
         @Override
         public Void invoke(Cache<String, AtomicMap<Object, Object>> cache)
         {
            cache.evict(sessionId);
            return null;
         }
      };
      
      this.invoker.invoke(this.getSessionCache(), operation);
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#evictSession(java.lang.String, java.lang.String)
    */
   @Override
   public void evictSession(String sessionId, String dataOwner)
   {
      if (dataOwner == null)
      {
         this.evictSession(sessionId);
      }
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#getSessionIds()
    */
   @Override
   public Map<String, String> getSessionIds()
   {
      Map<String, String> result = new HashMap<String, String>();

      for (Object key: this.cache.keySet())
      {
         if (key instanceof String)
         {
            result.put((String) key, null);
         }
      }
      
      return result;
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#isPassivationEnabled()
    */
   @Override
   public boolean isPassivationEnabled()
   {
      return this.passivationEnabled;
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#setForceSynchronous(boolean)
    */
   @Override
   public void setForceSynchronous(boolean forceSynchronous)
   {
      this.invoker.setForceSynchronous(forceSynchronous);
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#getSessionOwnershipSupport()
    */
   @Override
   public SessionOwnershipSupport getSessionOwnershipSupport()
   {
      return (this.lockManager != null) ? this : null;
   }
   
   @Override
   public LockResult acquireSessionOwnership(String sessionId, boolean newLock) throws TimeoutException, InterruptedException
   {
      LockResult result = lockResultMap.get(this.lockManager.lock(this.getSessionLockKey(sessionId), this.lockTimeout, newLock));
      
      return (result != null) ? result : LockResult.UNSUPPORTED;
   }

   @Override
   public void relinquishSessionOwnership(String sessionId, boolean remove)
   {
      this.lockManager.unlock(this.getSessionLockKey(sessionId), remove);
   }
   
   private Serializable getSessionLockKey(String sessionId)
   {
      return this.cache.getName() + "/" + sessionId;
   }
   
   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#isLocal(java.lang.String)
    */
   @Override
   public boolean isLocal(String realId)
   {
      DistributionManager manager = this.cache.getAdvancedCache().getDistributionManager();
      
      return (manager != null) ? manager.isLocal(realId) : true;
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager#locate(java.lang.String)
    */
   @Override
   public String locate(String sessionId)
   {
      DistributionManager manager = this.cache.getAdvancedCache().getDistributionManager();
      
      if (manager != null)
      {
         for (Address address: manager.locate(sessionId))
         {
            String jvmRoute = this.getJvmRouteCache().get(address);
            
            if (jvmRoute != null)
            {
               return jvmRoute;
            }
         }
      }
      
      return this.manager.getJvmRoute();
   }

   @CacheEntryRemoved
   public void removed(CacheEntryRemovedEvent event)
   {
      if (event.isPre() || event.isOriginLocal()) return;
      
      Object key = event.getKey();
      
      if (!(key instanceof String)) return;
      
      this.manager.notifyRemoteInvalidation((String) key);
   }
   
   @CacheEntryModified
   public void modified(CacheEntryModifiedEvent event)
   {
      if (event.isPre() || event.isOriginLocal()) return;
      
      Object key = event.getKey();
      
      if (!(key instanceof String)) return;
      
      String sessionId = (String) key;
      @SuppressWarnings("unchecked")
      AtomicMap<Object, Object> data = (AtomicMap<Object, Object>) event.getValue();
      
      Integer version = AtomicMapEntry.VERSION.get(data);
      Long timestamp = AtomicMapEntry.TIMESTAMP.get(data);
      DistributableSessionMetadata metadata = AtomicMapEntry.METADATA.get(data);
      
      if (timestamp == null)
      {
         log.warn(String.format("No timestamp attribute found in node modification event for session %s", mask(sessionId)));
         return;
      }
      
      boolean updated = this.manager.sessionChangedInDistributedCache(sessionId, null, version.intValue(), timestamp.longValue(), metadata);
      
      if (!updated)
      {
         log.warn(String.format("Possible concurrency problem: Replicated version id %d is less than or equal to in-memory version for session %s", version, mask(sessionId))); 
      }
   }
   
   @CacheEntryActivated
   public void activated(CacheEntryActivatedEvent event)
   {
      if (event.isPre()) return;
      
      Object key = event.getKey();
      
      if (!(key instanceof String)) return;
      
      if (this.manager.isPassivationEnabled())
      {
         this.manager.sessionActivated();
      }
   }
   
   @ViewChanged
   public void viewChanged(ViewChangedEvent event)
   {
      Collection<Address> oldMembers = event.getOldMembers();
      Collection<Address> newMembers = event.getNewMembers();
      
      // Remove jvm route of crashed member
      for (Address member: oldMembers)
      {
         if (!newMembers.contains(member))
         {
            this.getJvmRouteCache().remove(member);
         }
      }
      
      // Restore our jvm route in cache if we are new
      Address localAddress = event.getLocalAddress();
      if (!oldMembers.contains(localAddress) && newMembers.contains(localAddress))
      {
         this.getJvmRouteCache().put(localAddress, this.manager.getJvmRoute());
      }
   }
   
   // Simplified CacheInvoker.Operation using assigned key/value types
   static interface Operation<R> extends CacheInvoker.Operation<String, AtomicMap<Object, Object>, R>
   {
   }
}
