/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.web.tomcat.service.session.distributedcache.impl.jbc;

import java.util.Map;

import org.jboss.cache.Fqn;
import org.jboss.cache.notifications.annotation.NodeModified;
import org.jboss.cache.notifications.annotation.NodeRemoved;
import org.jboss.cache.notifications.event.NodeModifiedEvent;
import org.jboss.cache.notifications.event.NodeRemovedEvent;
import org.jboss.logging.Logger;
import org.jboss.web.tomcat.service.session.distributedcache.spi.DistributableSessionMetadata;
import org.jboss.web.tomcat.service.session.distributedcache.spi.DistributableSessionTimestamp;
import org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager;
import org.jboss.web.tomcat.service.session.distributedcache.spi.LocalDistributableSessionManager;

/**
 * Listens for removals and modifications in the cache, notifying the 
 * session manager of significant events.
 * 
 * @author Brian Stansberry
 */
@org.jboss.cache.notifications.annotation.CacheListener
public class CacheListener extends CacheListenerBase
{
   // Element within an FQN that is the root of a Pojo attribute map
   private static final int POJO_ATTRIBUTE_FQN_INDEX = SESSION_ID_FQN_INDEX + 1;
   // Element within an FQN that is the root of an individual Pojo attribute
   private static final int POJO_KEY_FQN_INDEX = POJO_ATTRIBUTE_FQN_INDEX + 1;
   // Element within an FQN that is the root of a session's internal pojo storage area
   private static final int POJO_INTERNAL_FQN_INDEX = SESSION_ID_FQN_INDEX + 1;
   // Minimum size of an FQN that is below the root of a session's internal pojo storage area
   private static final int POJO_INTERNAL_FQN_SIZE = POJO_INTERNAL_FQN_INDEX + 1;
   private static Logger log_ = Logger.getLogger(CacheListener.class);
   private boolean fieldBased_;

   CacheListener(JBossCacheWrapper wrapper, LocalDistributableSessionManager manager, String hostname, String webapp, boolean field)
   {      
      super(manager, hostname, webapp);
      fieldBased_ = field;
   }

   // --------------- CacheListener methods ------------------------------------

   @NodeRemoved
   public void nodeRemoved(NodeRemovedEvent event)
   {      
      if (event.isPre())
         return;
      
      boolean local = event.isOriginLocal();
      if (!fieldBased_ && local)
         return;
      
      Fqn fqn = event.getFqn();
      boolean isBuddy = isBuddyFqn(fqn);
      
      if (!local 
            && isFqnSessionRootSized(fqn, isBuddy) 
            && isFqnForOurWebapp(fqn, isBuddy))
      {
         // A session has been invalidated from another node;
         // need to inform manager
         String sessId = getIdFromFqn(fqn, isBuddy);
         manager_.notifyRemoteInvalidation(sessId);
      }
      else if (local && !isBuddy
                  && isPossibleInternalPojoFqn(fqn) 
                  && isFqnForOurWebapp(fqn, isBuddy))
      {
         // One of our sessions' pojos is modified; need to inform
         // the manager so it can mark the session dirty
         String sessId = getIdFromFqn(fqn, isBuddy);
         manager_.notifyLocalAttributeModification(sessId);
      }
   }
   
   @NodeModified
   public void nodeModified(NodeModifiedEvent event)
   {      
      if (event.isPre())
         return;
      
      boolean local = event.isOriginLocal();
      if (!fieldBased_ && local)
         return;
      
      Fqn fqn = event.getFqn();
      boolean isBuddy = isBuddyFqn(fqn);      
      
      if (!local 
             &&isFqnSessionRootSized(fqn, isBuddy)
             &&isFqnForOurWebapp(fqn, isBuddy))
      {
         // Query if we have version value in the distributed cache. 
         // If we have a version value, compare the version and invalidate if necessary.
         Map data = event.getData();
         Integer version = (Integer) data.get(DistributedCacheManager.VERSION_KEY);
         if(version != null)
         {
            String realId = getIdFromFqn(fqn, isBuddy);
            String owner = isBuddy ? getBuddyOwner(fqn) : null;
            // Notify the manager that a session has been updated
            boolean updated = manager_.sessionChangedInDistributedCache(realId, owner, 
                                               version.intValue(), 
                                               (DistributableSessionTimestamp) data.get(DistributedCacheManager.TIMESTAMP_KEY), 
                                               (DistributableSessionMetadata) data.get(DistributedCacheManager.METADATA_KEY));
            if (!updated && !isBuddy)
            {
               log_.warn("Possible concurrency problem: Replicated version id " + 
                         version + " matches in-memory version for session " + realId); 
            }
            /*else 
            {
               We have a local session but got a modification for the buddy tree.
               This means another node is in the process of taking over the session;
               we don't worry about it
            }
             */
         }
         else
         {
            log_.warn("No VERSION_KEY attribute found in " + fqn);
         }
      }
      else if (local && !isBuddy
            && isPossibleInternalPojoFqn(fqn) 
            && isFqnForOurWebapp(fqn, isBuddy))
      {
         // One of our sessions' pojos is modified; need to inform
         // the manager so it can mark the session dirty
         String sessId = getIdFromFqn(fqn, isBuddy);
         manager_.notifyLocalAttributeModification(sessId);
      }
   }
   
   public static String getPojoKeyFromFqn(Fqn fqn, boolean isBuddy)
   {
      return (String) fqn.get(isBuddy ? BUDDY_BACKUP_ROOT_OWNER_SIZE + POJO_KEY_FQN_INDEX: POJO_KEY_FQN_INDEX);
   }
   
   /**
    * Check if the fqn is big enough to be in the internal pojo area but
    * isn't in the regular attribute area.
    * 
    * Structure in the cache is:
    * 
    * /JSESSION
    * ++ /hostname
    * ++++ /contextpath
    * ++++++ /sessionid
    * ++++++++ /ATTRIBUTE
    * ++++++++ /_JBossInternal_
    * ++++++++++ etc etc
    * 
    * If the Fqn size is big enough to be "etc etc" or lower, but the 4th
    * level is "ATTRIBUTE", it must be under _JBossInternal_. We discriminate
    * based on != ATTRIBUTE to avoid having to code to the internal PojoCache
    * _JBossInternal_ name.
    * 
    * @param fqn
    * @return
    */
   public static boolean isPossibleInternalPojoFqn(Fqn fqn)
   {      
      return (fqn.size() > POJO_INTERNAL_FQN_SIZE 
            && JBossCacheService.ATTRIBUTE.equals(fqn.get(POJO_INTERNAL_FQN_INDEX)) == false);
   }
}
