/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.web.tomcat.service.session.distributedcache.impl.jbc;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;

import org.jboss.cache.CacheException;
import org.jboss.cache.Fqn;
import org.jboss.cache.pojo.PojoCache;
import org.jboss.ha.framework.server.PojoCacheManager;
import org.jboss.ha.framework.server.PojoCacheManagerLocator;
import org.jboss.web.tomcat.service.session.distributedcache.spi.ClusteringNotSupportedException;
import org.jboss.web.tomcat.service.session.distributedcache.spi.DistributableSession;
import org.jboss.web.tomcat.service.session.distributedcache.spi.FieldBasedDistributedCacheManager;

/**
 * Adds PojoCache-specific capabilities to JBossCacheService.
 * 
 * @author Brian Stansberry
 *
 */
public class FieldBasedJBossCacheService extends JBossCacheService implements FieldBasedDistributedCacheManager
{
   private final PojoCache pojoCache_;
   
   public FieldBasedJBossCacheService(String cacheConfigName) throws ClusteringNotSupportedException
   {
      this(Util.findPojoCache(cacheConfigName));
   }
   
   /**
    * Create a new FieldBasedJBossCacheService.
    * 
    * @param cache
    */
   public FieldBasedJBossCacheService(PojoCache cache)
   {
      super(cache.getCache());
      this.pojoCache_ = cache;
//      setCache(pojoCache_.getCache());
      if (!isMarshallingAvailable())
      {
         // BES 16/8/2006 -- throw ISE, not ClusteringNotSupportedException, as a
         // misconfig should be treated differently from the absence of clustering 
         // services
         throw new IllegalStateException("replication-granularity value is set to " +
               "'FIELD' but is not supported by the cache service configuration. " +
               "Must set 'useRegionBasedMarshalling' to 'true' in the cache configuration");
      }
   }
   
   @Override
   protected boolean isFieldBased()
   {
      return true;
   }

   private Fqn getFieldFqn(String id, String key)
   {
      return getFieldFqn(hostName_, webAppPath_, id, key);
   }


   private static void breakKeys(String key, List list)
   {
      StringTokenizer token = new StringTokenizer(key, FQN_DELIMITER);
      while(token.hasMoreTokens())
      {
         list.add(token.nextToken());
      }
   }
   public static Fqn getFieldFqn(String hostName, String contextPath, String sessionId, String attributeKey)
   {
      List list = new ArrayList(6);
      list.add(SESSION);
      list.add(hostName);
      list.add(contextPath);
      list.add(sessionId);
      list.add(ATTRIBUTE);
      // Guard against string with delimiter.
      breakKeys(attributeKey, list);
      return new Fqn(list);      
   }

   /**
    * store the pojo instance in the cache. Note that this is for the aop cache.
    * THe pojo needs to be "aspectized".
    * 
    * @param realId the session id with any jvmRoute removed
    * @param key    the attribute key
    * @param pojo
    * @param createRegion TODO
    */
   public Object setPojo(String realId, String key, Object pojo, DistributableSession session)
   {
      if(log_.isTraceEnabled())
      {
         log_.trace("setPojo(): session id: " + realId + " key: " + key + 
                    " object: " + pojo.toString());
      }      
      
      if (session.needRegionForSession())
      {
         Fqn sessionRegion = getSessionFqn(realId);
         getCache().getRegion(sessionRegion, true);
         session.createdRegionForSession();
         if (log_.isTraceEnabled())
         {
            log_.trace("Created region for session at " + sessionRegion);
         }         
      }
      
      // Construct the fqn.
      Fqn fqn = getFieldFqn(realId, key);
      try {
         // Ignore any cache notifications that our own work generates 
//         SessionReplicationContext.startCacheActivity();            
         return pojoCache_.attach(fqn.toString(), pojo);
      } catch (CacheException e) {
         throw new RuntimeException("JBossCacheService: exception occurred in cache setPojo ... ", e);
      }
      finally {
//         SessionReplicationContext.finishCacheActivity();
      }
   }

   /**
    * Remove pojo from the underlying cache store.
    * @param realId the session id with any jvmRoute removed
    * @param key    the attribute key
    * @return pojo that just removed. Null if there none.
    */
   public Object removePojo(String realId, String key)
   {
      if(log_.isTraceEnabled())
      {
         log_.trace("removePojo(): session id: " +realId + " key: " +key);
      }
      // Construct the fqn.
      Fqn fqn = getFieldFqn(realId, key);
      try {
         // Ignore any cache notifications that our own work generates 
//         SessionReplicationContext.startCacheActivity();
         return pojoCache_.detach(fqn.toString());
      } catch (CacheException e) {
         throw new RuntimeException("JBossCacheService: exception occurred in cache removePojo ... ", e);
      }
      finally {
//         SessionReplicationContext.finishCacheActivity();
      }
   }

   /**
    * Remove all the pojos from the underlying cache store locally 
    * without replication.
    * 
    * @param realId the session id with any jvmRoute removed
    */
   public void removePojosLocal(String realId)
   {
      if(log_.isTraceEnabled())
      {
         log_.trace("removePojoLocal(): session id: " +realId);
      }
      // Construct the fqn.
      Fqn fqn = getAttributeFqn(realId);
      try {
         // Ignore any cache notifications that our own work generates 
//         SessionReplicationContext.startCacheActivity();
         cacheWrapper_.removeLocal(fqn);
      }
      finally {
//         SessionReplicationContext.finishCacheActivity();
      }
   }

   /**
    * Remove all the pojos from the underlying cache store locally 
    * without replication.
    * 
    * @param realId the session id with any jvmRoute removed
    */
   public void removePojoLocal(String realId, String key)
   {
      if(log_.isTraceEnabled())
      {
         log_.trace("removePojoLocal(): session id: " + realId + " key: " +key);
      }
      // Construct the fqn.
      Fqn fqn = getFieldFqn(realId, key);
      try {
         // Ignore any cache notifications that our own work generates 
//         SessionReplicationContext.startCacheActivity();
         cacheWrapper_.removeLocal(fqn);
      }
      finally {
//         SessionReplicationContext.finishCacheActivity();
      }
   }
   
   public Set getPojoKeys(String realId)
   {
      Set keys = null;
      Fqn fqn = getAttributeFqn(realId);
      try
      {
         keys = getChildrenNames(fqn);
      }
      catch (CacheException e)
      {
         log_.error("getPojoKeys(): Exception getting keys for session " + realId, e);
      }
      
      return keys;      
   }
   

   /**
    *
    * @param realId the session id with any jvmRoute removed
    * @param key    the attribute key
    * @return Pojo that is associated with the attribute
    */
   public Object getPojo(String realId, String key)
   {
      Fqn fqn = getFieldFqn(realId, key);
      if(log_.isTraceEnabled())
      {
         log_.trace("getPojo(): session id: " +realId + " key: " + key +
                    " fqn: " + fqn);
      }
      
      try 
      {
         return pojoCache_.find(fqn);
      }
      catch (CacheException e) 
      {
         throw new RuntimeException("JBossCacheService: exception occurred in cache find ... ", e);
      }
   }
   
   protected void releaseCacheToManager(String cacheConfigName)
   {      
      try
      {
         PojoCacheManager pcm = PojoCacheManagerLocator.getCacheManagerLocator().getCacheManager(null);
         pcm.releaseCache(cacheConfigName);
      }
      catch (Exception e)
      {
         log_.error("Problem releasing cache to CacheManager -- config is " + cacheConfigName, e);
      }
   }

}
