/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.web.tomcat.service.session.distributedcache.impl.jbc;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.jboss.cache.Cache;
import org.jboss.cache.CacheException;
import org.jboss.cache.Fqn;
import org.jboss.cache.Node;
import org.jboss.web.tomcat.service.session.distributedcache.spi.ClusteringNotSupportedException;
import org.jboss.web.tomcat.service.session.distributedcache.spi.LocalDistributableSessionManager;

/**
 * DistributedCacheManager impl for ReplicationGranularity.ATTRIBUTE.
 */
public class AttributeBasedJBossCacheService extends AbstractJBossCacheService
{   
   public AttributeBasedJBossCacheService(LocalDistributableSessionManager localManager) throws ClusteringNotSupportedException
   {
      super(localManager);
   }
   
   public AttributeBasedJBossCacheService(LocalDistributableSessionManager localManager, Cache cache)
   {
      super(localManager, cache);
   }

   public boolean getSupportsAttributeOperations()
   {
      return true;
   }

   public Object getAttribute(String realId, String key)
   {
      Fqn fqn = getSessionFqn(combinedPath_, realId);
      return getUnMarshalledValue(cacheWrapper_.get(fqn, key));
   }

   public void putAttribute(String realId, String key, Object value)
   {
      Fqn fqn = getSessionFqn(combinedPath_, realId);
      cacheWrapper_.put(fqn, key, getMarshalledValue(value));
   }

   public void putAttribute(String realId, Map<String, Object> map)
   {
      // Duplicate the map with marshalled values
      Map marshalled = new HashMap(map.size());
      Set entries = map.entrySet();
      for (Iterator it = entries.iterator(); it.hasNext(); )
      {
         Map.Entry entry = (Map.Entry) it.next();
         marshalled.put(entry.getKey(), getMarshalledValue(entry.getValue()));
      }
      
      Fqn fqn = getSessionFqn(combinedPath_, realId);
      cacheWrapper_.put(fqn, marshalled);
      
   }

   public Object removeAttribute(String realId, String key)
   {
      Fqn fqn = getSessionFqn(combinedPath_, realId);
      if (log_.isTraceEnabled())
      {
         log_.trace("Remove attribute from distributed store. Fqn: " + fqn + " key: " + key);
      }
      return getUnMarshalledValue(cacheWrapper_.remove(fqn, key));
   }

   public void removeAttributeLocal(String realId, String key)
   {

      Fqn fqn = getSessionFqn(combinedPath_, realId);
      if (log_.isTraceEnabled())
      {
         log_.trace("Remove attribute from distributed store. Fqn: " + fqn + " key: " + key);
      }
      cacheWrapper_.removeLocal(fqn, key);
   }

   /**
    * Obtain the keys associated with this fqn. Note that it is not the fqn children.
    *
    */
   public Set getAttributeKeys(String realId)
   {
      Set keys = null;
      Fqn fqn = getSessionFqn(combinedPath_, realId);
      try
      {
         Node node = getCache().getRoot().getChild(fqn);
         if (node != null)
         {
            keys = node.getKeys();
            keys.removeAll(INTERNAL_KEYS);
         }
      }
      catch (CacheException e)
      {
         log_.error("getAttributeKeys(): Exception getting keys for session " + realId, e);
      }
      
      return keys;
   }

   /**
    * Return all attributes associated with this session id.
    * 
    * @param realId the session id with any jvmRoute removed
    * @return the attributes, or any empty Map if none are found.
    */
   public Map<String, Object> getAttributes(String realId)
   {
      if (realId == null || realId.length() == 0) return Collections.EMPTY_MAP;
      
      Fqn fqn = getSessionFqn(combinedPath_, realId);
      
      Node node = getCache().getRoot().getChild(fqn);
      Map<Object, Object> rawData = node.getData();
      
      return getSessionAttributes(realId, rawData);
   }
   
   /**
    * Returns the session attributes, possibly using the passed in 
    * <code>distributedCacheData</code> as a source.
    * 
    * <strong>Note:</strong> This operation may alter the contents of the 
    * passed in map. If this is unacceptable, pass in a defensive copy.
    */
   protected Map<String, Object> getSessionAttributes(String realId, Map<Object, Object> distributedCacheData)
   {
      Map<String, Object> attrs = new HashMap<String, Object>();         
      for (Map.Entry<Object, Object> entry : distributedCacheData.entrySet())
      {
         if (entry.getKey() instanceof String)
         {
            attrs.put((String) entry.getKey(), getUnMarshalledValue(entry.getValue()));
         }                
      }
      
      return attrs;
   }
   
   protected boolean isFieldBased()
   {
      return false;
   }
   
   protected boolean getStoreAttributesInSingleKey()
   {
      return false;
   }

}
