/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package org.jboss.ha.cachemanager;

import java.util.HashSet;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;

import org.jboss.cache.config.Configuration;
import org.jboss.cache.config.ConfigurationRegistry;

/**
 * {@link ConfigurationRegistry} that can obtain its initial set of 
 * configurations via dependency injection.
 * 
 * @author <a href="brian.stansberry@jboss.com">Brian Stansberry</a>
 * @version $Revision: 1 $
 */
public class DependencyInjectedConfigurationRegistry implements ConfigurationRegistry 
{    
    @SuppressWarnings("deprecation")
    private org.jboss.cache.factories.CacheConfigsXmlParser parser = new org.jboss.cache.factories.CacheConfigsXmlParser();
    private String configResource;
    private Map<String, Configuration> configs = new Hashtable<String, Configuration>();
    private boolean started;
    
    public DependencyInjectedConfigurationRegistry() 
    {
    }
    
    public void start() throws Exception
    {
        if (!started) 
        {
            loadConfigResource(configResource);
            started = true;
        }
    }

   private void loadConfigResource(String resource) throws CloneNotSupportedException
   {
      if (resource != null)
      {
         Map<String, Configuration> parsed = parser.parseConfigs(resource);
         for (Map.Entry<String, Configuration> entry : parsed.entrySet())
         {
            registerConfiguration(entry.getKey(), entry.getValue());
         }
      }
   }
    
    public void stop() 
    {
        if (started) 
        {
            synchronized (configs) 
            {
                configs.clear();
            }
            started = false;
        }
    }
    
    public void setConfigResource(String resource)
    {
       if (started)
       {
          try
          {
             loadConfigResource(resource);
          }
          catch (CloneNotSupportedException e)
          {
             throw new RuntimeException("Configuration in " + resource + 
                                        " does not properly support cloning", e);
          }
       }
       else
       {
          this.configResource = resource;
       }
    }
    
    public void setNewConfigurations(Map<String, Configuration> newConfigs)
    {
       if (newConfigs != null)
       {

          for (Map.Entry<String, Configuration> entry : newConfigs.entrySet())
          {
             try
             {
                registerConfiguration(entry.getKey(), entry.getValue());
             }
             catch (CloneNotSupportedException e)
             {
                throw new RuntimeException("Configuration " + entry.getKey() + 
                                           " does not properly support cloning", e);
             }
          }
       }
    }

    public Set<String> getConfigurationNames()
    {
        return new HashSet<String>(configs.keySet());
    }
    
    public void registerConfiguration(String configName, Configuration config) 
       throws CloneNotSupportedException
    {
        synchronized (configs) {
            if (configs.containsKey(configName))
                throw new IllegalStateException(configName + " already registered");
            configs.put(configName, config.clone());
        }
    }
    
    public void unregisterConfiguration(String configName) 
    {
        synchronized (configs) {
            if (configs.remove(configName) == null)
                throw new IllegalStateException(configName + " not registered");            
        }
    }
    
    public Configuration getConfiguration(String configName) 
    {
       Configuration config = null;
       synchronized (configs)
       {
          config = configs.get(configName);
       }
       
       if (config == null)
          throw new IllegalArgumentException("unknown config " + configName);
        
       // Don't hand out a ref to our master copy 
       try
       {
          return config.clone();
       }
       catch (CloneNotSupportedException e)
       {
          // This should not happen, as we already cloned the config
          throw new RuntimeException("Could not clone configuration " + configName, e);
       }
    }
}
