/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.web.tomcat.service.session.distributedcache.impl.jbc;


import org.jboss.cache.Cache;
import org.jboss.cache.CacheManager;
import org.jboss.cache.Fqn;
import org.jboss.cache.buddyreplication.BuddyManager;
import org.jboss.cache.pojo.PojoCache;
import org.jboss.ha.framework.server.CacheManagerLocator;
import org.jboss.ha.framework.server.PojoCacheManager;
import org.jboss.ha.framework.server.PojoCacheManagerLocator;
import org.jboss.metadata.web.jboss.ReplicationConfig;
import org.jboss.metadata.web.jboss.ReplicationGranularity;
import org.jboss.web.tomcat.service.session.distributedcache.spi.ClusteringNotSupportedException;
import org.jboss.web.tomcat.service.session.distributedcache.spi.LocalDistributableSessionManager;

/**
 * Utility methods related to JBoss distributed sessions.
 * 
 * @author Brian Stansberry
 * @version $Revision: 56542 $
 */
public class Util
{
   private static final int BUDDY_BACKUP_ROOT_OWNER_INDEX = BuddyManager.BUDDY_BACKUP_SUBTREE_FQN.size();
   private static final String DEAD_BUDDY_SUFFIX = ":DEAD";
   private static final String JSESSION = AbstractJBossCacheService.SESSION;
   
   public static ReplicationGranularity getReplicationGranularity(LocalDistributableSessionManager localManager)
   {
      ReplicationConfig config = localManager.getReplicationConfig();
      return config.getReplicationGranularity();
   }  
   
   public static String getCacheConfigName(LocalDistributableSessionManager localManager)
   {
      ReplicationConfig config = localManager.getReplicationConfig();
      return config.getCacheName();
   }
   
   /**
    * Gets a pojo cache with the named configuration
    * 
    * @param cacheConfigName the name of the configuration
    * @return the cache
    * @throws ClusteringNotSupportedException
    * 
    * @deprecated use {@link #findPojoCache(String, PojoCacheManager)
    */
   @Deprecated 
   public static PojoCache findPojoCache(String cacheConfigName) throws ClusteringNotSupportedException
   {
      PojoCacheManager pcm = getManagerForPojoCache(cacheConfigName);
      
      try
      {
         return pcm.getPojoCache(cacheConfigName, true);
      }
      catch (RuntimeException re)
      {
         throw re;
      }
      catch (Exception e)
      {
         throw new RuntimeException("Problem accessing cache " + cacheConfigName, e);
      }
   }
   
   public static PojoCache findPojoCache(String cacheConfigName, PojoCacheManager pcm)
   {      
      if (!pcm.getConfigurationNames().contains(cacheConfigName))
         throw new IllegalStateException("PojoCacheManager does not recognize config " + cacheConfigName);
      
      try
      {
         return pcm.getPojoCache(cacheConfigName, true);
      }
      catch (RuntimeException re)
      {
         throw re;
      }
      catch (Exception e)
      {
         throw new RuntimeException("Problem accessing cache " + cacheConfigName, e);
      }
   }
   
   /**
    * Gets a plain cache with the named configuration
    * 
    * @param cacheConfigName the name of the configuration
    * @return the cache
    * @throws ClusteringNotSupportedException
    * 
    * @deprecated use {@link #findPlainCache(String, CacheManager)
    */
   @Deprecated 
   public static Cache<Object, Object> findPlainCache(String cacheConfigName) throws ClusteringNotSupportedException
   {
      CacheManager pcm = getManagerForCache(cacheConfigName);
      
      try
      {
         return pcm.getCache(cacheConfigName, true);
      }
      catch (RuntimeException re)
      {
         throw re;
      }
      catch (Exception e)
      {
         throw new RuntimeException("Problem accessing cache " + cacheConfigName, e);
      }
   }
   
   public static Cache<Object, Object> findPlainCache(String cacheConfigName, CacheManager cm)
   {      
      if (!cm.getConfigurationNames().contains(cacheConfigName))
         throw new IllegalStateException("CacheManager does not recognize config " + cacheConfigName); 
      
      try
      {
         return cm.getCache(cacheConfigName, true);
      }
      catch (RuntimeException re)
      {
         throw re;
      }
      catch (Exception e)
      {
         throw new RuntimeException("Problem accessing cache " + cacheConfigName, e);
      }
   }
   
   public static PojoCacheManager findPojoCacheManager()  throws ClusteringNotSupportedException
   {
      try
      {
         return PojoCacheManagerLocator.getCacheManagerLocator().getCacheManager(null);
      }
      catch (Throwable t)
      {
         throw new ClusteringNotSupportedException("Could not access CacheManager for JBossWeb clustering", t);
      }
   }
   
   public static CacheManager findPlainCacheManager()  throws ClusteringNotSupportedException
   {
      try
      {
         return CacheManagerLocator.getCacheManagerLocator().getCacheManager(null);
      }
      catch (Throwable t)
      {
         throw new ClusteringNotSupportedException("Could not access CacheManager for JBossWeb clustering", t);
      }
   }
   
   private static PojoCacheManager getManagerForPojoCache(String cacheConfigName) 
      throws ClusteringNotSupportedException
   {
      PojoCacheManager pcm = null;
      try
      {
         pcm = PojoCacheManagerLocator.getCacheManagerLocator().getCacheManager(null);
      }
      catch (Throwable t)
      {
         throw new ClusteringNotSupportedException("Could not access PojoCacheManager for JBossWeb clustering", t);
      }
      
      if (!pcm.getConfigurationNames().contains(cacheConfigName))
         throw new IllegalStateException("PojoCacheManager does not recognize config " + cacheConfigName);
      
      return pcm;
   }
   
   private static CacheManager getManagerForCache(String cacheConfigName) 
      throws ClusteringNotSupportedException
   {
      CacheManager cm = null;
      try
      {
         cm = CacheManagerLocator.getCacheManagerLocator().getCacheManager(null);
      }
      catch (Throwable t)
      {
         throw new ClusteringNotSupportedException("Could not access CacheManager for JBossWeb clustering", t);
      }
      
      if (!cm.getConfigurationNames().contains(cacheConfigName))
         throw new IllegalStateException("CacheManager does not recognize config " + cacheConfigName);
      
      return cm;
   }
   
   public static String maskId(String realId)
   {
      if (realId == null)
      {
         return null;
      }
      else
      {
         int length = realId.length();
         if (length <= 8)
         {
            return realId;
         }
         StringBuilder sb = new StringBuilder(realId.substring(0, 2));
         sb.append("****");
         sb.append(realId.substring(length - 6, length));
         return sb.toString();
      }
   }
   
   public static final int getJSessionIndex(Fqn<String> fqn, boolean isBuddy)
   {
      try
      {
         if (!isBuddy && JSESSION.equals(fqn.get(0)))
         {
            return 0;
         }
         else if (isBuddy)
         {
            if (JSESSION.equals(fqn.get(BUDDY_BACKUP_ROOT_OWNER_INDEX + 1)))
            {
               return BUDDY_BACKUP_ROOT_OWNER_INDEX + 1;
            }
            else if (JSESSION.equals(fqn.get(BUDDY_BACKUP_ROOT_OWNER_INDEX + 2)))
            {
               return BUDDY_BACKUP_ROOT_OWNER_INDEX + 2;
            }
         }
      }
      catch (IndexOutOfBoundsException e)
      {
         // too small; just fall through
      }
      return -1;
   }
   
   public static boolean isBuddyOwnerDead(Fqn<String> fqn)
   {
      String base = (String) fqn.get(BUDDY_BACKUP_ROOT_OWNER_INDEX);
      return base.endsWith(DEAD_BUDDY_SUFFIX);
   }
   
   /**
    * Extracts the owner portion of an buddy subtree Fqn.
    * 
    * @param fqn An Fqn that is a child of the buddy backup root node.
    */
   public static String getBuddyOwner(Fqn<String> fqn)
   {
      String base = (String) fqn.get(BUDDY_BACKUP_ROOT_OWNER_INDEX);
      return base.endsWith(DEAD_BUDDY_SUFFIX) ? base.substring(0, base.length() - DEAD_BUDDY_SUFFIX.length()): base;     
   }
   
   /**
    * Returns either an <code>e</code> or a wrapper RuntimeException, depending on the type
    * of <code>e</code>. If <code>e</code> is an Ea RuntimeException, it
    * is simply returned, otherwise <code>e</code> is wrapped in a RuntimeException
    * which is returned.
    * 
    * @param wrapperMsg an optional message for the wrapper RuntimeException, if
    *                   one is needed. If not provided, the classname of <code>e</code>
    *                   and it's localized message is used
    * @param e the underlying exception
    */
   public static RuntimeException getRuntimeException(String wrapperMsg, Exception e)
   {   
      if (e instanceof RuntimeException)
         return (RuntimeException) e;
      
      wrapperMsg = (wrapperMsg != null) ? wrapperMsg : e.getClass().getName() + " -- " + e.getLocalizedMessage();
      throw new RuntimeException(wrapperMsg, e);
   }
   
   /**
    * Prevent instantiation.
    */
   private Util() {}

}
