/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ha.ispn;

import java.util.Properties;

import org.infinispan.config.Configuration;
import org.infinispan.config.GlobalConfiguration;
import org.infinispan.manager.DefaultCacheManager;
import org.infinispan.manager.EmbeddedCacheManager;
import org.infinispan.remoting.transport.jgroups.JGroupsChannelLookup;
import org.infinispan.remoting.transport.jgroups.JGroupsTransport;
import org.jgroups.Channel;
import org.jgroups.ChannelFactory;

/**
 * An Infinispan cache container factory that injects its channel from a channel factory.
 * 
 * @author Paul Ferraro
 */
@SuppressWarnings("deprecation")
public class DefaultCacheContainerFactory implements CacheContainerFactory
{
   public static final String CHANNEL_ID = "channelId";
   public static final String CHANNEL_FACTORY = "channelFactory";
   
   private final ChannelFactory channelFactory;
   
   /**
    * Creates a new cache container factory using the specified channel factory.
    * @param channelFactory a channel factory
    */
   public DefaultCacheContainerFactory(ChannelFactory channelFactory)
   {
      this.channelFactory = channelFactory;
   }
   
   /**
    * {@inheritDoc}
    * @see org.jboss.ha.ispn.CacheContainerFactory#createCacheContainer(org.jboss.ha.ispn.CacheContainerConfiguration)
    */
   @Override
   public EmbeddedCacheManager createCacheContainer(CacheContainerConfiguration configuration)
   {
      GlobalConfiguration globalConfiguration = configuration.getGlobalConfiguration();
      if (globalConfiguration.getTransportClass() != null)
      {
         globalConfiguration.setStrictPeerToPeer(false);
         Properties properties = globalConfiguration.getTransportProperties();
         
         if (!properties.containsKey(JGroupsTransport.CHANNEL_LOOKUP))
         {
            properties.put(CHANNEL_FACTORY, this.channelFactory);
            properties.setProperty(CHANNEL_ID, globalConfiguration.getClusterName());
            properties.setProperty(JGroupsTransport.CHANNEL_LOOKUP, ChannelFactoryChannelLookup.class.getName());
         }
      }

      EmbeddedCacheManager manager = new DefaultCacheManager(globalConfiguration, configuration.getDefaultConfiguration(), false);

      // Add named configurations
      for (Configuration config: configuration.getNamedConfigurations())
      {
         manager.defineConfiguration(config.getName(), config);
      }
      
      return new DefaultCacheContainer(manager);
   }
   
   /**
    * JGroupsChannelLookup implementation that uses a channel factory.
    */
   public static class ChannelFactoryChannelLookup implements JGroupsChannelLookup
   {
      public static final String STACK = "stack";
      public static final String DEFAULT_STACK = "udp";
      public static final String AUTO_START = "auto-start";
      public static final String DEFAULT_AUTO_START = Boolean.toString(true);
      public static final String AUTO_STOP = "auto-stop";
      public static final String DEFAULT_AUTO_STOP = Boolean.toString(true);
      
      private boolean autoStart = true;
      private boolean autoStop = true;
      
      @Override
      public Channel getJGroupsChannel(Properties properties)
      {
         this.autoStart = Boolean.valueOf(properties.getProperty(AUTO_START, DEFAULT_AUTO_START));
         this.autoStop = Boolean.valueOf(properties.getProperty(AUTO_STOP, DEFAULT_AUTO_STOP));
         
         String stack = properties.getProperty(STACK, DEFAULT_STACK);
         String id = properties.getProperty(CHANNEL_ID);
         ChannelFactory factory = (ChannelFactory) properties.get(CHANNEL_FACTORY);
         
         try
         {
            return factory.createMultiplexerChannel(stack, id);
         }
         catch (Exception e)
         {
            throw new IllegalArgumentException(e);
         }
      }

      @Override
      public boolean shouldStartAndConnect()
      {
         return this.autoStart;
      }

      @Override
      public boolean shouldStopAndDisconnect()
      {
         return this.autoStop;
      }
   }
}
