/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ha.ispn.config.xml;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collections;
import java.util.Map;

import javax.xml.XMLConstants;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.transform.Source;
import javax.xml.transform.sax.SAXSource;
import javax.xml.validation.SchemaFactory;

import org.infinispan.config.InfinispanConfiguration;
import org.infinispan.config.parsing.NamespaceFilter;
import org.jboss.ha.ispn.config.CacheContainerRegistryConfiguration;
import org.jboss.ha.ispn.config.CacheContainerRegistryConfigurationSource;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

/**
 * A source for a cache container registry that parses an XML file containing a series of infinispan cache container configurations.
 * @author Paul Ferraro
 */
public class XMLCacheContainerRegistryConfigurationSource implements CacheContainerRegistryConfigurationSource
{
   private static final String SCHEMA = "META-INF/infinispan-configs.xsd";
   private static final String DEFAULT_XML = "infinispan-configs.xml";
   
   private static final String INFINISPAN_NAMESPACE = NamespaceFilter.ISPN_NS;
   private static final String INFINISPAN_SCHEMA = InfinispanConfiguration.resolveSchemaPath();

   private final URL infinispanSchemaURL;
   private final URL schemaURL;
   private final URL xmlURL;
   
   /**
    * Creates a new XMLCacheContainerRegistryConfigurationSource using the {@link #DEFAULT_XML} classpath resource
    * containing the cache container registry configuration.
    */
   public XMLCacheContainerRegistryConfigurationSource()
   {
      this(DEFAULT_XML);
   }
   
   /**
    * Creates a new XMLCacheContainerRegistryConfigurationSource using the specified location containing
    * the cache container registry configuration.
    * @param url the location of the cache container registry configuration
    */
   public XMLCacheContainerRegistryConfigurationSource(URL url)
   {
      this.xmlURL = url;
      this.schemaURL = findResource(SCHEMA);
      this.infinispanSchemaURL = findResource(INFINISPAN_SCHEMA);
   }
   
   /**
    * Creates a new XMLCacheContainerRegistryConfigurationSource using the specified classpath resource containing
    * the cache container registry configuration.
    * @param resource the classpath resource containing the cache container registry configuration
    */
   public XMLCacheContainerRegistryConfigurationSource(String resource)
   {
      this(findResource(resource));
   }
   
   private static URL findResource(String resource)
   {
      try
      {
         return new URL(resource);
      }
      catch (MalformedURLException e)
      {
         URL url = Thread.currentThread().getContextClassLoader().getResource(resource);
         
         if (url == null)
         {
            throw new IllegalArgumentException("Failed to locate resource: " + resource);
         }
         
         return url;
      }
   }
   
   /**
    * {@inheritDoc}
    * @see org.jboss.ha.ispn.config.CacheContainerRegistryConfigurationSource#getRegistryConfiguration()
    */
   @Override
   public CacheContainerRegistryConfiguration getRegistryConfiguration() throws JAXBException, SAXException
   {
      SchemaFactory schemaFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
      // Resolve Infinispan schema against schema found in classpath
      Map<String, URL> schemaURLs = Collections.singletonMap(INFINISPAN_NAMESPACE, this.infinispanSchemaURL);
      schemaFactory.setResourceResolver(new NamespaceResourceResolver(Collections.singletonMap(XMLConstants.W3C_XML_SCHEMA_NS_URI, schemaURLs)));
      
      Unmarshaller unmarshaller = JAXBContext.newInstance(CacheContainerRegistryConfiguration.class).createUnmarshaller();
      unmarshaller.setSchema(schemaFactory.newSchema(this.schemaURL));
      
      XMLReader reader = new PropertyReplacementFilter(XMLReaderFactory.createXMLReader());
      Source source = new SAXSource(reader, new InputSource(this.xmlURL.toString()));
      return (CacheContainerRegistryConfiguration) unmarshaller.unmarshal(source);
   }
}
