/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ha.ispn;

import java.security.AccessController;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.infinispan.Cache;
import org.infinispan.config.Configuration;
import org.infinispan.config.GlobalConfiguration;
import org.infinispan.lifecycle.ComponentStatus;
import org.infinispan.manager.CacheContainer;
import org.infinispan.manager.DefaultCacheManager;
import org.infinispan.manager.EmbeddedCacheManager;
import org.infinispan.notifications.Listener;
import org.infinispan.notifications.cachemanagerlistener.annotation.CacheStopped;
import org.infinispan.notifications.cachemanagerlistener.event.CacheStoppedEvent;
import org.infinispan.remoting.transport.Address;
import org.jboss.util.loading.ContextClassLoaderSwitcher;

/**
 * Cache manager wrapper to work around ISPN-658
 * @author Paul Ferraro
 */
@Listener
public class DefaultCacheContainer implements EmbeddedCacheManager
{
   // Need to cast since ContextClassLoaderSwitcher.NewInstance does not generically implement PrivilegedAction<ContextClassLoaderSwitcher>
   @SuppressWarnings("unchecked")
   private final ContextClassLoaderSwitcher switcher = (ContextClassLoaderSwitcher) AccessController.doPrivileged(ContextClassLoaderSwitcher.INSTANTIATOR);
   
   private final EmbeddedCacheManager container;
   private final ConcurrentMap<String, EmbeddedCacheManager> containers = new ConcurrentHashMap<String, EmbeddedCacheManager>();

   public DefaultCacheContainer(EmbeddedCacheManager manager)
   {
      this.container = manager;
   }
   
   @Override
   public <K, V> Cache<K, V> getCache()
   {
      return this.getCache(CacheContainer.DEFAULT_CACHE_NAME);
   }

   @Override
   public <K, V> Cache<K, V> getCache(String cacheName)
   {
      return this.getCache(cacheName, cacheName.equals(CacheContainer.DEFAULT_CACHE_NAME) ? this.container.getDefaultConfiguration() : this.container.defineConfiguration(cacheName, new Configuration()));
   }
   
   // Until ISPN-658 is fixed, we need to create a separate adhoc cache manager if requested cache uses DIST mode.
   private <K, V> Cache<K, V> getCache(String cacheName, Configuration config)
   {
      // Since infinispan doesn't init the shared cache container until the 1st cache is started
      // we need to make sure any classes are loaded with this object's classloader
      // not the thread context class loader, which is likely application specific
      // and may otherwise cause a class loader leak when the application is undeployed
      ContextClassLoaderSwitcher.SwitchContext context = this.switcher.getSwitchContext(this.getClass().getClassLoader());

      try
      {
         if (config.getCacheMode().isDistributed())
         {
            GlobalConfiguration globalConfig = this.container.getGlobalConfiguration();
            
            // Create unique cluster name using cache name
            String containerName = this.appendCacheName(globalConfig.getCacheManagerName(), cacheName);
            EmbeddedCacheManager container = this.containers.get(containerName);
            
            if (container == null)
            {
               GlobalConfiguration global = globalConfig.clone();
               global.setCacheManagerName(containerName);
               global.setClusterName(this.appendCacheName(globalConfig.getClusterName(), cacheName));
               global.getTransportProperties().setProperty(DefaultCacheContainerFactory.CHANNEL_ID, global.getClusterName());
               
               // Create single use cache manager
               container = new SingletonCacheManager(this, cacheName, global, config);
               
               EmbeddedCacheManager existing = this.containers.putIfAbsent(containerName, container);
               
               if (existing == null)
               {
                  container.addListener(this);
                  
                  for (Object listener: this.container.getListeners())
                  {
                     container.addListener(listener);
                  }
                  
                  // The cache manager should stop when the cache stops
                  container.start();
               }
               else
               {
                  container = existing;
               }
            }
            
            return container.getCache(cacheName);
         }
         
         return this.container.getCache(cacheName);
      }
      finally
      {
         context.reset();
      }
   }

   private String appendCacheName(String name, String cacheName)
   {
      return name + "/" + cacheName;
   }
   
   @Override
   public synchronized void start()
   {
      this.container.start();
   }

   @Override
   public synchronized void stop()
   {
      this.container.stop();
   }

   @Override
   public void addListener(Object listener)
   {
      this.container.addListener(listener);
   }

   @Override
   public void removeListener(Object listener)
   {
      this.container.removeListener(listener);
   }

   @Override
   public Set<Object> getListeners()
   {
      return this.container.getListeners();
   }

   @Override
   public Configuration defineConfiguration(String cacheName, Configuration configurationOverride)
   {
      return this.container.defineConfiguration(cacheName, configurationOverride);
   }

   @Override
   public Configuration defineConfiguration(String cacheName, String templateCacheName, Configuration configurationOverride)
   {
      return this.container.defineConfiguration(cacheName, templateCacheName, configurationOverride);
   }

   @Override
   public String getClusterName()
   {
      return this.container.getClusterName();
   }

   @Override
   public List<Address> getMembers()
   {
      return this.container.getMembers();
   }

   @Override
   public Address getAddress()
   {
      return this.container.getAddress();
   }

   @Override
   public boolean isCoordinator()
   {
      return this.container.isCoordinator();
   }

   @Override
   public ComponentStatus getStatus()
   {
      return this.container.getStatus();
   }

   @Override
   public GlobalConfiguration getGlobalConfiguration()
   {
      return this.container.getGlobalConfiguration();
   }

   @Override
   public Configuration getDefaultConfiguration()
   {
      return this.container.getDefaultConfiguration();
   }

   @Override
   public Set<String> getCacheNames()
   {
      return this.container.getCacheNames();
   }
   
   /**
    * {@inheritDoc}
    * @see org.infinispan.manager.EmbeddedCacheManager#getCoordinator()
    */
   @Override
   public Address getCoordinator()
   {
      return this.container.getCoordinator();
   }

   /**
    * {@inheritDoc}
    * @see org.infinispan.manager.EmbeddedCacheManager#isDefaultRunning()
    */
   @Override
   public boolean isDefaultRunning()
   {
      return this.container.isDefaultRunning();
   }

   /**
    * {@inheritDoc}
    * @see org.infinispan.manager.EmbeddedCacheManager#isRunning(java.lang.String)
    */
   @Override
   public boolean isRunning(String cacheName)
   {
      EmbeddedCacheManager container = this.containers.get(this.appendCacheName(this.container.getGlobalConfiguration().getCacheManagerName(), cacheName));

      return (container != null) ? container.isDefaultRunning() : this.container.isRunning(cacheName);
   }

   @CacheStopped
   public synchronized void cacheStopped(CacheStoppedEvent event)
   {
      EmbeddedCacheManager container = event.getCacheManager();
      
      this.containers.remove(container.getGlobalConfiguration().getCacheManagerName());
      
      container.stop();
   }
   
   static class SingletonCacheManager extends DefaultCacheManager
   {
      private final EmbeddedCacheManager parent;
      private final String cacheName;
      
      /**
       * Creates a new SingletonCacheManager
       * @param globalConfiguration
       * @param defaultConfiguration
       */
      SingletonCacheManager(EmbeddedCacheManager parent, String cacheName, GlobalConfiguration globalConfiguration, Configuration defaultConfiguration)
      {
         super(globalConfiguration, defaultConfiguration, false);
         
         this.parent = parent;
         this.cacheName = cacheName;
      }
      
      /**
       * {@inheritDoc}
       * @see org.infinispan.manager.DefaultCacheManager#getCache(java.lang.String)
       */
      @Override
      public <K, V> Cache<K, V> getCache(String name)
      {
         return this.cacheName.equals(name) ? super.<K, V>getCache(CacheContainer.DEFAULT_CACHE_NAME) : this.parent.<K, V>getCache(name);
      }

      /**
       * {@inheritDoc}
       * @see org.infinispan.manager.DefaultCacheManager#isRunning(java.lang.String)
       */
      @Override
      public boolean isRunning(String cacheName)
      {
         return this.cacheName.equals(cacheName) ? super.isRunning(CacheContainer.DEFAULT_CACHE_NAME) : this.parent.isRunning(cacheName);
      }

      /**
       * {@inheritDoc}
       * @see org.infinispan.manager.DefaultCacheManager#addListener(java.lang.Object)
       */
      @Override
      public void addListener(Object listener)
      {
         super.addListener(listener);
         // This is hacky...
         // Better solution - use custom notifier that allows
         // some level of control over listeners order
         if (listener != this.parent)
         {
            // Make sure our listener is last since our listener stops the container!
            super.removeListener(this.parent);
            super.addListener(this.parent);
         }
      }
   }
}
