/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ha.ispn.config.xml;

import java.net.URL;
import java.util.Map;

import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;

/**
 * A custom resource resolver that can override the locations for resources by namespace.
 * 
 * @author Paul Ferraro
 */
public class NamespaceResourceResolver implements LSResourceResolver
{
   private final Map<String, Map<String, URL>> types;
   private final DOMImplementationLS implementation;
   
   /**
    * Creates a new NamespaceResourceResolver using the specified set of target resource URLs, identified by namespace, per resource type.
    * Uses the default DOM load-store implementation.
    * @param types a map of resource type to map of namespace to resource location
    */
   public NamespaceResourceResolver(Map<String, Map<String, URL>> types)
   {
      this(types, defaultImplementation());
   }
   
   /**
    * Creates a new NamespaceResourceResolver using the specified set of target resource URLs, identified by namespace, per resource type.
    * @param types a map of resource type to map of namespace to resource location
    * @param implementation a DOM load-store implementation
    */
   public NamespaceResourceResolver(Map<String, Map<String, URL>> types, DOMImplementationLS implementation)
   {
      this.types = types;
      this.implementation = implementation;
   }
   
   private static DOMImplementationLS defaultImplementation()
   {
      try
      {
         return (DOMImplementationLS) DOMImplementationRegistry.newInstance().getDOMImplementation("LS");
      }
      catch (IllegalAccessException e)
      {
         throw new IllegalStateException(e);
      }
      catch (InstantiationException e)
      {
         throw new IllegalStateException(e);
      }
      catch (ClassNotFoundException e)
      {
         throw new IllegalStateException(e);
      }
   }

   /**
    * {@inheritDoc}
    * @see org.w3c.dom.ls.LSResourceResolver#resolveResource(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
    */
   @Override
   public LSInput resolveResource(String type, String namespaceURI, String publicId, String systemId, String baseURI)
   {
      LSInput input = this.implementation.createLSInput();
      
      input.setPublicId(publicId);
      input.setSystemId(systemId);
      input.setBaseURI(baseURI);
      
      Map<String, URL> namespaces = this.types.get(type);
      
      if (namespaces != null)
      {
         URL url = namespaces.get(namespaceURI);
         
         if (url != null)
         {
            // Substitute our system id, since we have a matching type and namespace
            input.setSystemId(url.toString());
         }
      }
      
      return input;
   }
}
