/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.dna.sequencer.zip;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.jboss.dna.common.monitor.ProgressMonitor;
import org.jboss.dna.graph.sequencers.SequencerContext;
import org.jboss.dna.graph.sequencers.SequencerOutput;
import org.jboss.dna.graph.sequencers.StreamSequencer;

/**
 * @author Michael Trezzi
 */
public class ZipSequencer implements StreamSequencer {

    /**
     * {@inheritDoc}
     * 
     * @see org.jboss.dna.graph.sequencers.StreamSequencer#sequence(java.io.InputStream,
     *      org.jboss.dna.graph.sequencers.SequencerOutput, org.jboss.dna.graph.sequencers.SequencerContext,
     *      org.jboss.dna.common.monitor.ProgressMonitor)
     */
    public void sequence( InputStream stream,
                          SequencerOutput output,
                          SequencerContext context,
                          ProgressMonitor progressMonitor ) {
        try {
            ZipInputStream in = new ZipInputStream(stream);
            ZipEntry entry = in.getNextEntry();
            byte[] buf = new byte[1024];

            // Create top-level node
            output.setProperty("zip:content", "jcr:primaryType", "zip:content");
            while (entry != null) {

                if (entry.isDirectory()) { // If entry is directory, create nt:folder node
                    output.setProperty("zip:content/" + entry.getName(), "jcr:primaryType", "nt:folder");
                } else { // If entry is File, create nt:file
                    output.setProperty("zip:content/" + entry.getName() + "/jcr:content", "jcr:primaryType", "nt:resource");
                    int n;
                    ByteArrayOutputStream baout = new ByteArrayOutputStream();
                    while ((n = in.read(buf, 0, 1024)) > -1) {
                        baout.write(buf, 0, n);
                    }
                    output.setProperty("zip:content/" + entry.getName() + "/jcr:content", "jcr:data", baout.toByteArray());
                    // all other nt:file properties should be generated by other sequencers (mimetype, encoding,...)
                }
                in.closeEntry();
                entry = in.getNextEntry();

            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }
}
