/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.ejb3.deployers.metadata.processor;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.logging.Logger;
import org.jboss.metadata.ear.jboss.JBossAppMetaData;
import org.jboss.metadata.ejb.jboss.JBossEnterpriseBeanMetaData;
import org.jboss.metadata.ejb.jboss.JBossEnterpriseBeansMetaData;
import org.jboss.metadata.ejb.jboss.JBossMetaData;
import org.jboss.metadata.process.ProcessingException;
import org.jboss.metadata.process.processor.JBossMetaDataProcessor;

/**
 * An implementation of {@link JBossMetaDataProcessor} which sets
 * the containername in the {@link JBossEnterpriseBeanMetaData}, if not already set.
 * 
 *
 * @author Jaikiran Pai
 * @version $Revision: $
 */
public class ContainerNameMetaDataProcessor implements JBossMetaDataProcessor<JBossMetaData>
{

   private static Logger logger = Logger.getLogger(ContainerNameMetaDataProcessor.class);
   
   private DeploymentUnit deploymentUnit;

   /**
    * Creates a {@link ContainerNameMetaDataProcessor} for a {@link DeploymentUnit}
    * @param deploymentUnit
    */
   public ContainerNameMetaDataProcessor(DeploymentUnit deploymentUnit)
   {
      this.deploymentUnit = deploymentUnit;
   }

   /**
    * Sets the container name in each of {@link JBossEnterpriseBeanMetaData}, present within the passed
    * {@link JBossMetaData}, if it's not already set.
    *  
    * @see org.jboss.metadata.process.processor.JBossMetaDataProcessor#process(org.jboss.metadata.ejb.jboss.JBossMetaData)
    */
   public JBossMetaData process(JBossMetaData metadata) throws ProcessingException
   {
      if (!metadata.isEJB3x())
      {
         return metadata;
      }
      JBossEnterpriseBeansMetaData enterpriseBeans = metadata.getEnterpriseBeans();
      if (enterpriseBeans == null || enterpriseBeans.isEmpty())
      {
         return metadata;
      }
      for (JBossEnterpriseBeanMetaData enterpriseBean : enterpriseBeans)
      {
         // if already set, then move to next
         if (enterpriseBean.getContainerName() != null)
         {
            continue;
         }
         String containerName;
         try
         {
            // get the container name
            containerName = this.getContainerName(this.deploymentUnit, enterpriseBean);
         }
         catch (MalformedObjectNameException moe)
         {
            throw new ProcessingException("Could not generate a container name for bean " + enterpriseBean.getEjbName(), moe);
         }
         enterpriseBean.setContainerName(containerName);
      }
      return metadata;
   }

   /**
    * Generates and returns a container name for the passed {@link JBossEnterpriseBeanMetaData}
    * which belongs to the passed {@link DeploymentUnit}
    *  
    * @param unit The deployment unit which contains the bean
    * @param enterpriseBeanMetaData The bean metadata
    * @return
    * @throws MalformedObjectNameException If generated container name is malformed
    */
   private String getContainerName(DeploymentUnit unit, JBossEnterpriseBeanMetaData enterpriseBeanMetaData)
         throws MalformedObjectNameException
   {
      // TODO the base ejb3 jmx object name comes from Ejb3Module.BASE_EJB3_JMX_NAME, but
      // we don't need any reference to ejb3-core. Right now just hard code here, we need
      // a better way/place for this later
      StringBuilder containerName = new StringBuilder("jboss.j2ee:service=EJB3" + ",");

      // Get the top level unit for this unit (ex: the top level might be an ear and this unit might be the jar
      // in that ear
      DeploymentUnit toplevelUnit = unit.getTopLevel();
      if (toplevelUnit != null)
      {
         // if top level is an ear, then create the name with the ear reference
         if (isEar(toplevelUnit))
         {
            containerName.append("ear=");
            containerName.append(toplevelUnit.getSimpleName());
            containerName.append(",");

         }
      }
      // now work on the passed unit, to get the jar name
      if (unit.getSimpleName() == null)
      {
         containerName.append("*");
      }
      else
      {
         containerName.append("jar=");
         containerName.append(unit.getSimpleName());
      }
      // now the ejbname
      containerName.append(",name=");
      containerName.append(enterpriseBeanMetaData.getEjbName());

      if (logger.isTraceEnabled())
      {
         logger.trace("Container name generated for ejb = " + enterpriseBeanMetaData.getEjbName() + " in unit " + unit
               + " is " + containerName);
      }
      ObjectName containerJMXName = new ObjectName(containerName.toString());
      return containerJMXName.getCanonicalName();
   }

   /**
   * Returns true if this <code>unit</code> represents an .ear deployment
   *
   * @param unit
   * @return
   */
   private boolean isEar(DeploymentUnit unit)
   {
      return unit.getSimpleName().endsWith(".ear") || unit.getAttachment(JBossAppMetaData.class) != null;
   }

}
