/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
  *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb3.proxy.impl.handler.session;

import java.io.Serializable;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

import org.jboss.aop.Advisor;
import org.jboss.aop.advice.Interceptor;
import org.jboss.aop.joinpoint.Invocation;
import org.jboss.aop.util.MethodHashing;
import org.jboss.ejb3.common.lang.SerializableMethod;
import org.jboss.ejb3.common.registrar.spi.Ejb3Registrar;
import org.jboss.ejb3.common.registrar.spi.Ejb3RegistrarLocator;
import org.jboss.ejb3.proxy.impl.invocation.StatefulRemoteInvocation;
import org.jboss.ejb3.proxy.spi.container.InvokableContext;

/**
 * SessionLocalProxyInvocationHandler
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public class SessionLocalProxyInvocationHandler extends SessionProxyInvocationHandlerBase implements Serializable
{

   // --------------------------------------------------------------------------------||
   // Class Members ------------------------------------------------------------------||
   // --------------------------------------------------------------------------------||

   private static final long serialVersionUID = 1L;

   // --------------------------------------------------------------------------------||
   // Constructor --------------------------------------------------------------------||
   // --------------------------------------------------------------------------------||

   /**
    * Constructor
    * 
    * @param containerName The name of the target container
    * @param containerGuid The globally-unique name of the container
    * @param interceptors The interceptors to apply to invocations upon this handler
    */
   public SessionLocalProxyInvocationHandler(final String containerName, final String containerGuid,
         final Interceptor[] interceptors)
   {
      super(containerName, containerGuid, interceptors, null, null);
   }

   /**
    * Constructor
    * 
    * @param containerName The name of the target container
    * @param containerGuid The globally-unique name of the container
    * @param businessInterfaceType The possibly null businessInterfaceType
    *   marking this invocation hander as specific to a given
    *   EJB3 Business Interface
    * @param interceptors The interceptors to apply to invocations upon this handler
    */
   public SessionLocalProxyInvocationHandler(final String containerName, final String containerGuid,
         final Interceptor[] interceptors, final String businessInterfaceType, final Serializable sessionId)
   {
      super(containerName, containerGuid, interceptors, businessInterfaceType, sessionId);
   }

   // --------------------------------------------------------------------------------||
   // Required Implementations -------------------------------------------------------||
   // --------------------------------------------------------------------------------||
   
   /**
    * {@inheritDoc}
    * @see org.jboss.ejb3.proxy.impl.handler.session.SessionProxyInvocationHandlerBase#getContainer(java.lang.reflect.Method, java.lang.Object[])
    */
   @Override
   protected InvokableContext getContainer(final Method method, final Object[] args)
   {
      // Lookup
      Object obj = Ejb3RegistrarLocator.locateRegistrar().lookup(this.getContainerName());

      // Ensure of correct type
      assert obj instanceof InvokableContext : "Container retrieved from " + Ejb3Registrar.class.getSimpleName()
            + " was not of expected type " + InvokableContext.class.getName() + " but was instead " + obj;

      final InvokableContext container = (InvokableContext) obj;

      final InvokableContext proxyToContainer = (InvokableContext) Proxy.newProxyInstance(InvokableContext.class
            .getClassLoader(), new Class<?>[]
      {InvokableContext.class}, new LocalInvokableContextHandler(method, args, this.getTarget(), this.getInterceptors(),
            container));

      // Return
      return proxyToContainer;
   }
   
   // --------------------------------------------------------------------------------||
   // Internal Helper Members --------------------------------------------------------||
   // --------------------------------------------------------------------------------||
   
   /**
    * Invocation handler to adapt interceptors to local invocations
    * upon the container
    * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
    */
   private static final class LocalInvokableContextHandler implements InvocationHandler
   {

      private final Method realMethod;

      private final Object[] realArgs;

      private final Object session;

      private final Interceptor[] interceptors;
      final InvokableContext container;

      LocalInvokableContextHandler(final Method realMethod, final Object[] realArgs, final Object session,
            final Interceptor[] interceptors,final InvokableContext container)
      {
         assert realMethod != null : "method must be specified";
         assert realArgs != null : "realArgs must be specified";
         assert interceptors != null : "interceptors must be specified";
         assert container != null : "container must be specified";
         this.realMethod = realMethod;
         this.realArgs = realArgs;
         this.session = session;
         this.interceptors = interceptors;
         this.container = container;
         
      }

      public Object invoke(final Object proxy, final Method method, final Object[] args) throws Throwable
      {
         final long hash = MethodHashing.calculateHash(realMethod);
         final LocalContainerInvocation invocation = new LocalContainerInvocation(interceptors, hash, realMethod,
               realMethod, null, session);
         invocation.setTargetObject(container);
         invocation.setArguments(args);
         return invocation.invokeNext();
      }
   }
   
   /**
    * Adaptor invocation implementation to intercept based on real methods, 
    * yet invoke the target {@link InvokableContext} at the end 
    * of the interceptor chain
    * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
    */
   private static class LocalContainerInvocation extends StatefulRemoteInvocation{

      public LocalContainerInvocation(Interceptor[] interceptors, long methodHash, Method advisedMethod,
            Method unadvisedMethod, Advisor advisor, Object id)
      {
         super(interceptors, methodHash, advisedMethod, unadvisedMethod, advisor, id);
      }

      @Override
      public Object invokeTarget() throws Throwable
      {
         try
         {
            final Method methodToInvoke =InvokableContext.class.getMethod("invoke", Object.class,
                  SerializableMethod.class, Object[].class);
            return methodToInvoke.invoke(getTargetObject(), arguments);
         }
         catch (Throwable t)
         {
            throw handleErrors(getTargetObject(), getMethod(), arguments, t);
         }
      }
      
      @Override
      public Invocation copy()
      {
         final LocalContainerInvocation copy = new LocalContainerInvocation(this.interceptors, this.methodHash, this.advisedMethod,
               this.unadvisedMethod, this.advisor, this.id);
         copy.setMetaData(this.getMetaData());
         copy.setArguments(this.getArguments());
         copy.setTargetObject(targetObject);
         return copy;
      }
   }

}
