/*
 * JBoss, Home of Professional Open Source
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.embedded.api.server;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.logging.Logger;

/**
 * {@link ClassLoader} implementation which is capable of booting
 * the JBoss Application Server installation located at a known
 * $JBOSS_HOME 
 * 
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public class JBossHomeClassLoader extends URLClassLoader
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Logger
    */
   private static final Logger log = Logger.getLogger(JBossHomeClassLoader.class.getName());

   /**
    * Name of the boot library directory, relative to $JBOSS_HOME
    */
   private static final String DIR_NAME_LIB = "lib/";

   /**
    * Libraries, relative to $JBOSS_HOME/lib, which must be visible to 
    * the context ClassLoader (and the loading CL of the server)
    * in order to boot the Application Server.  This is a carefully
    * pruned list in order to not leak out class references, which would
    * in turn result in {@link NoClassDefFoundError}
    */
   private static final String[] JBOSS_LIB_BOOT_LIST =
   {
         // Concurrent
         "concurrent.jar",
         // Logging
         "log4j-boot.jar",
         "jboss-logging-spi.jar",
         "jboss-logging-log4j.jar",
         "jboss-logging-jdk.jar",
         "jboss-logmanager.jar",
         "jboss-logbridge.jar",
         // Common jars
         "jboss-common-core.jar",
         "jboss-xml-binding.jar",
         // Bootstrap
         "jboss-bootstrap-spi.jar", "jboss-bootstrap-spi-as.jar", "jboss-bootstrap-spi-mc.jar",
         "jboss-bootstrap-impl-base.jar", "jboss-bootstrap-impl-as.jar", "jboss-bootstrap-impl-mc.jar",
         "jboss-bootstrap-api-as.jar", "jboss-bootstrap-api-mc.jar",
         "jboss-bootstrap-api.jar",
         // Microcontainer
         "javassist.jar", "jboss-reflect.jar", "jboss-mdr.jar", "jboss-dependency.jar", "jboss-kernel.jar",
         "jboss-metatype.jar", "jboss-managed.jar",
         // Deployers
         "jboss-deployers-core-spi.jar", "jboss-deployers-client-spi.jar",
         // Fixme ClassLoading
         "jboss-vfs.jar", "jboss-classloading-spi.jar", "jboss-classloader.jar", "jboss-classloading.jar",
         "jboss-classloading-vfs.jar",
         // Fixme aop
         "jboss-aop.jar", "jboss-aop-mc-int.jar", "trove.jar"};

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Constructs a new instance using the specified URLs; not exposed publicly, instead use
    * {@link JBossHomeClassLoader#newInstance(URL, URL[])} outside this class
    */
   private JBossHomeClassLoader(final URL[] urls, final ClassLoader parent) throws IllegalArgumentException,
         IllegalStateException
   {
      // Invoke super
      super(urls, parent);
   }

   //-------------------------------------------------------------------------------------||
   // Factory Methods --------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Constructs a new instance, using the required boot libraries of the specified $JBOSS_HOME
    * as well as the the additional URLs specified (optional)
    * 
    * @param jbossHome The location to JBOSS_HOME
    * @param urls Additional URLs which this ClassLoader should use; may be null
    * @param parent The parent to assign to the new ClassLoader
    * @throws IllegalArgumentException If JBOSS_HOME is not specified
    */
   public static JBossHomeClassLoader newInstance(final URL jbossHome, final URL[] urls, final ClassLoader parent)
   {
      // Precondition checks
      if (jbossHome == null)
      {
         throw new IllegalArgumentException("JBOSS_HOME must be specified");
      }

      // Get $JBOSS_HOME/lib URLs
      final URL[] jbossLibBootUrls = getJBossLibBootUrls(jbossHome);

      // Merge with the specified URLs passed in
      final URL[] merged = merge(jbossLibBootUrls, urls);

      // Create and return a new instance
      return new JBossHomeClassLoader(merged, parent);
   }

   //-------------------------------------------------------------------------------------||
   // Internal Helper Methods ------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Merges the arrays of all URLs into a unified view, with no duplicate 
    * elements
    * 
    * @param urls Any number of arrays to be represented as a union
    * @throws IllegalArgumentException If the URLs specified are null
    */
   private static URL[] merge(final URL[]... urls) throws IllegalArgumentException
   {
      // Precondition check
      assert urls != null : "URLs must be specified";

      // Create a set
      final Set<URL> urlSet = new HashSet<URL>();

      // Merge
      for (final URL[] current : urls)
      {
         for (final URL single : current)
         {
            urlSet.add(single);
         }
      }

      // Return
      return urlSet.toArray(new URL[]
      {});

   }

   /**
    * Obtains all boot libraries for which ClassLoader
    * visibility is required given the specified $JBOSS_HOME
    */
   private static URL[] getJBossLibBootUrls(final URL jbossHome)
   {
      // Get a URL to $JBOSS_HOME/lib
      final URL jbossLib;
      try
      {
         jbossLib = new URL(jbossHome, DIR_NAME_LIB);
      }
      catch (final MalformedURLException murle)
      {
         throw new RuntimeException("Error in constructing $JBOSS_HOME/lib URL", murle);
      }

      final Collection<URL> urls = new HashSet<URL>();
      // Add all appropriate JBOSS_HOME/lib binaries
      for (final String bootLib : JBOSS_LIB_BOOT_LIST)
      {
         final URL url;
         try
         {
            url = new URL(jbossLib, bootLib);
         }
         catch (final MalformedURLException murle)
         {
            throw new RuntimeException("Could not construct full URL for $JBOSS_HOME boot library: " + bootLib, murle);
         }
         urls.add(url);
         log.finer("Using: " + url);

         // Check file for existence 
         final URI uri;
         try
         {
            uri = url.toURI();
         }
         catch (final URISyntaxException urise)
         {
            throw new RuntimeException(urise);
         }
         final File file = new File(uri);
         if (!file.exists())
         {
            throw new IllegalStateException("Attempting to add a library to the ClassLoader which does not exist: "
                  + file + "; perhaps a corrupt $JBOSS_HOME installation? ");
         }
      }

      // Return
      log.fine("Using: " + urls);
      return urls.toArray(new URL[]
      {});
   }

}
