/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.embedded.api.server;

import java.security.AccessController;
import java.security.PrivilegedAction;

/**
 * SecurityActions
 * 
 * Utility class for security actions, so as to provide
 * some centralization without leaking privileged actions
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
class SecurityActions
{

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * In place to prevent instantiation 
    */
   private SecurityActions()
   {
   }

   //-------------------------------------------------------------------------------------||
   // Protected Actions ------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Action to obtain the Thread Context ClassLoader
    */
   private enum GetTcclAction implements PrivilegedAction<ClassLoader> {
      INSTANCE;
      public ClassLoader run()
      {
         // Return the TCCL
         return Thread.currentThread().getContextClassLoader();
      }

   }

   //-------------------------------------------------------------------------------------||
   // Utility Methods --------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Obtains the Thread Context ClassLoader
    * 
    * @return
    */
   static ClassLoader getTccl()
   {
      return AccessController.doPrivileged(GetTcclAction.INSTANCE);
   }

   /**
    * Obtains the system property with the specified key
    * 
    * @param key
    * @return
    * @throws IllegalArgumentException If the key is null
    */
   static String getSystemProperty(final String key) throws IllegalArgumentException
   {
      // Precondition check
      if (key == null)
      {
         throw new IllegalArgumentException("key was null");
      }

      // Get sysprop
      return AccessController.doPrivileged(new PrivilegedAction<String>()
      {
         public String run()
         {
            return System.getProperty(key);
         }
      });
   }

   /**
    * Sets the system property with the specified key and value
    * 
    * @param key
    * @param value
    * @return
    * @throws IllegalArgumentException If the key or value is null
    */
   static void setSystemProperty(final String key, final String value) throws IllegalArgumentException
   {
      // Precondition check
      if (key == null)
      {
         throw new IllegalArgumentException("key was null");
      }
      if (value == null)
      {
         throw new IllegalArgumentException("value was null");
      }

      // Get sysprop
      AccessController.doPrivileged(new PrivilegedAction<Void>()
      {
         public Void run()
         {
            System.setProperty(key, value);
            return null;
         }
      });
   }
}
