package org.hsqldb;

import org.hsqldb.lib.Collection;
import org.hsqldb.lib.HashMappedList;
import org.hsqldb.lib.HashSet;
import org.hsqldb.lib.HsqlArrayList;
import org.hsqldb.lib.IntKeyHashMap;
import org.hsqldb.lib.IntValueHashMap;
import org.hsqldb.lib.Iterator;
import org.hsqldb.lib.Set;
import org.hsqldb.lib.StringUtil;

class GranteeManager
  implements GrantConstants
{
  static final String SYSTEM_AUTHORIZATION_NAME = "_SYSTEM";
  static final String DBA_ADMIN_ROLE_NAME = "DBA";
  static final String PUBLIC_ROLE_NAME = "PUBLIC";
  static final String[] emptyRightsList = new String[0];
  static final IntKeyHashMap hRightsLists = new IntKeyHashMap();
  private HashMappedList map = new HashMappedList();
  private HashMappedList roleMap = new HashMappedList();
  static final IntValueHashMap rightsStringLookup = new IntValueHashMap(7);

  public GranteeManager(Database paramDatabase)
    throws HsqlException
  {
    addRole("DBA");
    getRole("DBA").setAdminDirect();
  }

  void grant(String paramString, Object paramObject, int paramInt)
    throws HsqlException
  {
    Grantee localGrantee = get(paramString);
    if (localGrantee == null)
      throw Trace.error(232, paramString);
    if (isImmutable(paramString))
      throw Trace.error(243, paramString);
    localGrantee.grant(paramObject, paramInt);
    localGrantee.updateAllRights();
    if (localGrantee.isRole)
      updateAllRights(localGrantee);
  }

  void grant(String paramString1, String paramString2)
    throws HsqlException
  {
    Grantee localGrantee1 = get(paramString1);
    if (localGrantee1 == null)
      throw Trace.error(232, paramString1);
    if (isImmutable(paramString1))
      throw Trace.error(243, paramString1);
    Grantee localGrantee2 = get(paramString2);
    if (localGrantee2 == null)
      throw Trace.error(239, paramString2);
    if (paramString2.equals(paramString1))
      throw Trace.error(244, paramString1);
    if (localGrantee2.hasRole(paramString1))
      throw Trace.error(244, Trace.getMessage(245) + " GRANT " + paramString1 + " TO " + paramString2);
    if (localGrantee1.getDirectRoles().contains(paramString2))
      throw Trace.error(245, paramString2);
    localGrantee1.grant(paramString2);
    localGrantee1.updateAllRights();
    if (localGrantee1.isRole)
      updateAllRights(localGrantee1);
  }

  void revoke(String paramString1, String paramString2)
    throws HsqlException
  {
    Grantee localGrantee = get(paramString1);
    if (localGrantee == null)
      throw Trace.error(232, paramString1);
    localGrantee.revoke(paramString2);
    localGrantee.updateAllRights();
    if (localGrantee.isRole)
      updateAllRights(localGrantee);
  }

  void revoke(String paramString, Object paramObject, int paramInt)
    throws HsqlException
  {
    Grantee localGrantee = get(paramString);
    localGrantee.revoke(paramObject, paramInt);
    localGrantee.updateAllRights();
    if (localGrantee.isRole)
      updateAllRights(localGrantee);
  }

  void removeEmptyRole(Grantee paramGrantee)
  {
    String str = paramGrantee.getName();
    for (int i = 0; i < this.map.size(); i++)
    {
      Grantee localGrantee = (Grantee)this.map.get(i);
      localGrantee.roles.remove(str);
    }
  }

  void removeDbObject(Object paramObject)
  {
    for (int i = 0; i < this.map.size(); i++)
    {
      Grantee localGrantee = (Grantee)this.map.get(i);
      localGrantee.revokeDbObject(paramObject);
    }
  }

  void updateAllRights(Grantee paramGrantee)
  {
    String str = paramGrantee.getName();
    Grantee localGrantee;
    for (int i = 0; i < this.map.size(); i++)
    {
      localGrantee = (Grantee)this.map.get(i);
      if (!localGrantee.isRole)
        continue;
      localGrantee.updateNestedRoles(str);
    }
    for (i = 0; i < this.map.size(); i++)
    {
      localGrantee = (Grantee)this.map.get(i);
      if (localGrantee.isRole)
        continue;
      localGrantee.updateAllRights();
    }
  }

  public boolean removeGrantee(String paramString)
  {
    if (isReserved(paramString))
      return false;
    Grantee localGrantee = (Grantee)this.map.remove(paramString);
    if (localGrantee == null)
      return false;
    localGrantee.clearPrivileges();
    updateAllRights(localGrantee);
    if (localGrantee.isRole)
    {
      this.roleMap.remove(paramString);
      removeEmptyRole(localGrantee);
    }
    return true;
  }

  public Grantee addGrantee(String paramString)
    throws HsqlException
  {
    if (this.map.containsKey(paramString))
      throw Trace.error(241, paramString);
    Grantee localGrantee1 = null;
    if (!isReserved(paramString))
    {
      localGrantee1 = get("PUBLIC");
      if (localGrantee1 == null)
        Trace.doAssert(false, Trace.getMessage(242));
    }
    Grantee localGrantee2 = new Grantee(paramString, localGrantee1, this);
    this.map.put(paramString, localGrantee2);
    return localGrantee2;
  }

  boolean isGrantee(String paramString)
  {
    return this.map.containsKey(paramString);
  }

  static int getCheckRight(String paramString)
    throws HsqlException
  {
    int i = getRight(paramString);
    if (i != 0)
      return i;
    throw Trace.error(249, paramString);
  }

  static int getRight(String paramString)
  {
    return rightsStringLookup.get(paramString, 0);
  }

  static String getRightsList(int paramInt)
  {
    if (paramInt == 0)
      return null;
    if (paramInt == 15)
      return "ALL";
    return StringUtil.getList(getRightsArray(paramInt), ",", "");
  }

  static String[] getRightsArray(int paramInt)
  {
    if (paramInt == 0)
      return emptyRightsList;
    String[] arrayOfString = (String[])hRightsLists.get(paramInt);
    if (arrayOfString != null)
      return arrayOfString;
    arrayOfString = getRightsArraySub(paramInt);
    hRightsLists.put(paramInt, arrayOfString);
    return arrayOfString;
  }

  private static String[] getRightsArraySub(int paramInt)
  {
    if (paramInt == 0)
      return emptyRightsList;
    HsqlArrayList localHsqlArrayList = new HsqlArrayList();
    Iterator localIterator = rightsStringLookup.keySet().iterator();
    while (localIterator.hasNext())
    {
      String str = (String)localIterator.next();
      if (str.equals("ALL"))
        continue;
      int i = rightsStringLookup.get(str, 0);
      if ((paramInt & i) == 0)
        continue;
      localHsqlArrayList.add(str);
    }
    return (String[])localHsqlArrayList.toArray(new String[localHsqlArrayList.size()]);
  }

  HashSet getGrantedClassNames()
    throws HsqlException
  {
    int i = this.map.size();
    HashSet localHashSet = new HashSet();
    for (int j = 0; j < i; j++)
    {
      Grantee localGrantee = (Grantee)this.map.get(j);
      if (localGrantee == null)
        continue;
      Iterator localIterator = localGrantee.getGrantedClassNames(false).iterator();
      while (localIterator.hasNext())
        localHashSet.add(localIterator.next());
    }
    return localHashSet;
  }

  public Grantee get(String paramString)
  {
    return (Grantee)this.map.get(paramString);
  }

  public Collection getGrantees()
  {
    return this.map.values();
  }

  public static boolean validRightString(String paramString)
  {
    return getRight(paramString) != 0;
  }

  public static boolean isImmutable(String paramString)
  {
    return (paramString.equals("_SYSTEM")) || (paramString.equals("DBA"));
  }

  public static boolean isReserved(String paramString)
  {
    return (paramString.equals("_SYSTEM")) || (paramString.equals("DBA")) || (paramString.equals("PUBLIC"));
  }

  String addRole(String paramString)
    throws HsqlException
  {
    if (paramString == null)
      Trace.doAssert(false, Trace.getMessage(236));
    Grantee localGrantee = null;
    if (validRightString(paramString))
      throw Trace.error(237, paramString);
    localGrantee = addGrantee(paramString);
    localGrantee.isRole = true;
    boolean bool = this.roleMap.add(paramString, localGrantee);
    if (!bool)
      throw Trace.error(238, paramString);
    return paramString;
  }

  void dropRole(String paramString)
    throws HsqlException
  {
    if (paramString.equals("DBA"))
      throw Trace.error(33);
    if (!isRole(paramString))
      throw Trace.error(239, paramString);
    removeGrantee(paramString);
    this.roleMap.remove(paramString);
  }

  public Set getRoleNames()
  {
    return this.roleMap.keySet();
  }

  Grantee getRole(String paramString)
    throws HsqlException
  {
    if (!isRole(paramString))
      Trace.doAssert(false, "No role '" + paramString + "'");
    Grantee localGrantee = (Grantee)this.roleMap.get(paramString);
    if (localGrantee == null)
      throw Trace.error(234, paramString);
    return localGrantee;
  }

  boolean isRole(String paramString)
    throws HsqlException
  {
    return this.roleMap.containsKey(paramString);
  }

  static
  {
    rightsStringLookup.put("ALL", 15);
    rightsStringLookup.put("SELECT", 1);
    rightsStringLookup.put("UPDATE", 8);
    rightsStringLookup.put("DELETE", 2);
    rightsStringLookup.put("INSERT", 4);
  }
}

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     org.hsqldb.GranteeManager
 * JD-Core Version:    0.6.0
 */