package org.hsqldb;

import java.io.IOException;
import org.hsqldb.index.RowIterator;
import org.hsqldb.lib.ArrayUtil;
import org.hsqldb.lib.HashMappedList;
import org.hsqldb.lib.HashSet;
import org.hsqldb.lib.HsqlArrayList;
import org.hsqldb.lib.Iterator;
import org.hsqldb.lib.StringUtil;
import org.hsqldb.persist.CachedObject;
import org.hsqldb.persist.DataFileCache;
import org.hsqldb.persist.HsqlDatabaseProperties;
import org.hsqldb.persist.Logger;
import org.hsqldb.persist.PersistentStore;
import org.hsqldb.rowio.RowInputInterface;
import org.hsqldb.store.ValuePool;

public class Table extends BaseTable
{
  public static final int SYSTEM_TABLE = 0;
  public static final int SYSTEM_SUBQUERY = 1;
  public static final int TEMP_TABLE = 2;
  public static final int MEMORY_TABLE = 3;
  public static final int CACHED_TABLE = 4;
  public static final int TEMP_TEXT_TABLE = 5;
  public static final int TEXT_TABLE = 6;
  public static final int VIEW = 7;
  static final int SYSTEM_VIEW = 8;
  public HashMappedList columnList;
  private int[] primaryKeyCols;
  private int[] primaryKeyTypes;
  private int[] primaryKeyColsSequence;
  int[] bestRowIdentifierCols;
  boolean bestRowIdentifierStrict;
  int[] bestIndexForColumn;
  Index bestIndex;
  int identityColumn;
  NumberSequence identitySequence;
  NumberSequence rowIdSequence;
  Constraint[] constraintList;
  HsqlArrayList[] triggerLists;
  private int[] colTypes;
  private int[] colSizes;
  private int[] colScales;
  private boolean[] colNullable;
  private Expression[] colDefaults;
  private int[] defaultColumnMap;
  private boolean hasDefaultValues;
  boolean sqlEnforceSize;
  protected int columnCount;
  public Database database;
  protected DataFileCache cache;
  protected HsqlNameManager.HsqlName tableName;
  private int tableType;
  protected boolean isReadOnly;
  protected boolean isTemp;
  protected boolean isCached;
  protected boolean isText;
  protected boolean isMemory;
  private boolean isView;
  protected boolean isLogged;
  protected int indexType;
  protected boolean onCommitPreserve;
  PersistentStore rowStore;
  Index[] indexList;
  HashSet constraintPath;
  HashMappedList tableUpdateList;

  Table(Database paramDatabase, HsqlNameManager.HsqlName paramHsqlName, int paramInt)
    throws HsqlException
  {
    this.database = paramDatabase;
    this.sqlEnforceSize = paramDatabase.sqlEnforceStrictSize;
    this.identitySequence = new NumberSequence(null, 0L, 1L, -5);
    this.rowIdSequence = new NumberSequence(null, 0L, 1L, -5);
    switch (paramInt)
    {
    case 1:
      this.isTemp = true;
      this.isMemory = true;
      break;
    case 0:
      this.isMemory = true;
      break;
    case 4:
      if (DatabaseURL.isFileBasedDatabaseType(paramDatabase.getType()))
      {
        this.cache = paramDatabase.logger.getCache();
        this.isCached = true;
        this.isLogged = (!this.database.isFilesReadOnly());
        this.indexType = 1;
        this.rowStore = new RowStore();
      }
      else
      {
        paramInt = 3;
      }
    case 3:
      this.isMemory = true;
      this.isLogged = (!this.database.isFilesReadOnly());
      break;
    case 2:
      this.isMemory = true;
      this.isTemp = true;
      break;
    case 5:
      if (!DatabaseURL.isFileBasedDatabaseType(paramDatabase.getType()))
        throw Trace.error(63);
      this.isTemp = true;
      this.isText = true;
      this.isReadOnly = true;
      this.indexType = 2;
      this.rowStore = new RowStore();
      break;
    case 6:
      if (!DatabaseURL.isFileBasedDatabaseType(paramDatabase.getType()))
        throw Trace.error(63);
      this.isText = true;
      this.indexType = 2;
      this.rowStore = new RowStore();
      break;
    case 7:
    case 8:
      this.isView = true;
    }
    this.tableType = paramInt;
    this.tableName = paramHsqlName;
    this.primaryKeyCols = null;
    this.primaryKeyTypes = null;
    this.identityColumn = -1;
    this.columnList = new HashMappedList();
    this.indexList = new Index[0];
    this.constraintList = new Constraint[0];
    this.triggerLists = new HsqlArrayList[12];
    if ((paramDatabase.isFilesReadOnly()) && (isFileBased()))
      setIsReadOnly(true);
  }

  boolean equals(Session paramSession, String paramString)
  {
    return this.tableName.name.equals(paramString);
  }

  boolean equals(String paramString)
  {
    return this.tableName.name.equals(paramString);
  }

  boolean equals(HsqlNameManager.HsqlName paramHsqlName)
  {
    return this.tableName.equals(paramHsqlName);
  }

  public final boolean isText()
  {
    return this.isText;
  }

  public final boolean isTemp()
  {
    return this.isTemp;
  }

  public final boolean isReadOnly()
  {
    return isDataReadOnly();
  }

  final boolean isView()
  {
    return this.isView;
  }

  final int getIndexType()
  {
    return this.indexType;
  }

  public final int getTableType()
  {
    return this.tableType;
  }

  public boolean isDataReadOnly()
  {
    return this.isReadOnly;
  }

  protected void setIsReadOnly(boolean paramBoolean)
  {
    this.isReadOnly = paramBoolean;
    this.database.setMetaDirty(true);
  }

  void checkDataReadOnly()
    throws HsqlException
  {
    if (isDataReadOnly())
      throw Trace.error(32);
  }

  void setDataReadOnly(boolean paramBoolean)
    throws HsqlException
  {
    if ((!paramBoolean) && (this.database.isFilesReadOnly()) && (isFileBased()))
      throw Trace.error(32);
    setIsReadOnly(paramBoolean);
  }

  boolean isFileBased()
  {
    return (this.isCached) || (this.isText);
  }

  protected void setDataSource(Session paramSession, String paramString, boolean paramBoolean1, boolean paramBoolean2)
    throws HsqlException
  {
    throw Trace.error(22);
  }

  protected String getDataSource()
  {
    return null;
  }

  protected boolean isDescDataSource()
  {
    return false;
  }

  public void setHeader(String paramString)
    throws HsqlException
  {
    throw Trace.error(150);
  }

  public String getHeader()
  {
    return null;
  }

  public boolean isConnected()
  {
    return true;
  }

  public void connect(Session paramSession)
    throws HsqlException
  {
    throw Trace.error(255);
  }

  public void disconnect(Session paramSession)
    throws HsqlException
  {
    throw Trace.error(255);
  }

  void addConstraint(Constraint paramConstraint)
  {
    this.constraintList = ((Constraint[])ArrayUtil.toAdjustedArray(this.constraintList, paramConstraint, this.constraintList.length, 1));
  }

  Constraint[] getConstraints()
  {
    return this.constraintList;
  }

  Constraint getPrimaryConstraint()
  {
    return this.primaryKeyCols.length == 0 ? null : this.constraintList[0];
  }

  Index getUniqueConstraintIndexForColumns(int[] paramArrayOfInt)
  {
    if (ArrayUtil.areEqual(getPrimaryIndex().getColumns(), paramArrayOfInt, paramArrayOfInt.length, true))
      return getPrimaryIndex();
    int i = 0;
    int j = this.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = this.constraintList[i];
      if ((localConstraint.getType() == 2) && (ArrayUtil.areEqual(localConstraint.getMainColumns(), paramArrayOfInt, paramArrayOfInt.length, true)))
        return localConstraint.getMainIndex();
      i++;
    }
    return null;
  }

  Constraint getConstraintForColumns(Table paramTable, int[] paramArrayOfInt1, int[] paramArrayOfInt2)
  {
    int i = 0;
    int j = this.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = this.constraintList[i];
      if (localConstraint.isEquivalent(paramTable, paramArrayOfInt1, this, paramArrayOfInt2))
        return localConstraint;
      i++;
    }
    return null;
  }

  Constraint getUniqueConstraintForColumns(int[] paramArrayOfInt)
  {
    int i = 0;
    int j = this.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = this.constraintList[i];
      if (localConstraint.isEquivalent(paramArrayOfInt, 2))
        return localConstraint;
      i++;
    }
    return null;
  }

  Constraint getUniqueOrPKConstraintForIndex(Index paramIndex)
  {
    int i = 0;
    int j = this.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = this.constraintList[i];
      if ((localConstraint.getMainIndex() == paramIndex) && ((localConstraint.getType() == 2) || (localConstraint.getType() == 4)))
        return localConstraint;
      i++;
    }
    return null;
  }

  int getNextConstraintIndex(int paramInt1, int paramInt2)
  {
    int i = paramInt1;
    int j = this.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = this.constraintList[i];
      if (localConstraint.getType() == paramInt2)
        return i;
      i++;
    }
    return -1;
  }

  void addColumn(Column paramColumn)
    throws HsqlException
  {
    if (findColumn(paramColumn.columnName.name) >= 0)
      throw Trace.error(27, paramColumn.columnName.name);
    if (paramColumn.isIdentity())
    {
      Trace.check((paramColumn.getType() == 4) || (paramColumn.getType() == -5), 16, paramColumn.columnName.name);
      Trace.check(this.identityColumn == -1, 24, paramColumn.columnName.name);
      this.identityColumn = this.columnCount;
    }
    if (this.primaryKeyCols != null)
      Trace.doAssert(false, "Table.addColumn");
    this.columnList.add(paramColumn.columnName.name, paramColumn);
    this.columnCount += 1;
  }

  void addColumns(Result.ResultMetaData paramResultMetaData, int paramInt)
    throws HsqlException
  {
    for (int i = 0; i < paramInt; i++)
    {
      Column localColumn = new Column(this.database.nameManager.newHsqlName(paramResultMetaData.colLabels[i], paramResultMetaData.isLabelQuoted[i]), true, paramResultMetaData.colTypes[i], paramResultMetaData.colSizes[i], paramResultMetaData.colScales[i], false, null);
      addColumn(localColumn);
    }
  }

  void addColumns(Select paramSelect)
    throws HsqlException
  {
    int i = paramSelect.iResultLen;
    for (int j = 0; j < i; j++)
    {
      Expression localExpression = paramSelect.exprColumns[j];
      Column localColumn = new Column(this.database.nameManager.newHsqlName(localExpression.getAlias(), localExpression.isAliasQuoted()), true, localExpression.getDataType(), localExpression.getColumnSize(), localExpression.getColumnScale(), false, null);
      addColumn(localColumn);
    }
  }

  public HsqlNameManager.HsqlName getName()
  {
    return this.tableName;
  }

  public int getId()
  {
    return this.tableName.hashCode();
  }

  void rename(Session paramSession, String paramString, boolean paramBoolean)
    throws HsqlException
  {
    String str = this.tableName.name;
    this.tableName.rename(paramString, paramBoolean);
    renameTableInCheckConstraints(paramSession, str, paramString);
  }

  int getInternalColumnCount()
  {
    return this.columnCount;
  }

  protected Table duplicate()
    throws HsqlException
  {
    Table localTable = new Table(this.database, this.tableName, this.tableType);
    localTable.onCommitPreserve = this.onCommitPreserve;
    return localTable;
  }

  void checkColumnsMatch(int[] paramArrayOfInt1, Table paramTable, int[] paramArrayOfInt2)
    throws HsqlException
  {
    if (paramArrayOfInt1.length != paramArrayOfInt2.length)
      throw Trace.error(5);
    for (int i = 0; i < paramArrayOfInt1.length; i++)
    {
      if ((paramArrayOfInt1[i] >= this.columnCount) || (paramArrayOfInt2[i] >= paramTable.columnCount))
        throw Trace.error(5);
      if (getColumn(paramArrayOfInt1[i]).getType() == paramTable.getColumn(paramArrayOfInt2[i]).getType())
        continue;
      throw Trace.error(57);
    }
  }

  Table moveDefinition(int[] paramArrayOfInt, Column paramColumn, int paramInt1, int paramInt2)
    throws HsqlException
  {
    Table localTable = duplicate();
    for (int i = 0; i < this.columnCount + 1; i++)
    {
      if (i == paramInt1)
        if (paramInt2 == 0)
        {
          if (paramColumn != null)
          {
            localTable.addColumn(paramColumn);
            continue;
          }
        }
        else if (paramInt2 > 0)
          localTable.addColumn(paramColumn);
        else
          if (paramInt2 < 0)
            continue;
      if (i == this.columnCount)
        break;
      localTable.addColumn(getColumn(i));
    }
    Object localObject1 = this.primaryKeyCols.length == 0 ? null : this.primaryKeyCols;
    if (localObject1 != null)
    {
      localObject2 = ArrayUtil.toAdjustedColumnArray(localObject1, paramInt1, paramInt2);
      if (localObject1.length != localObject2.length)
        throw Trace.error(25);
      localObject1 = localObject2;
    }
    localTable.createPrimaryKey(getIndex(0).getName(), localObject1, false);
    localTable.constraintList = this.constraintList;
    Object localObject2 = null;
    if (paramArrayOfInt != null)
      localObject2 = getIndex(paramArrayOfInt, paramInt1);
    if (localObject2 != null)
    {
      if (((Index)localObject2).isConstraint())
        throw Trace.error(252);
      throw Trace.error(123);
    }
    for (int j = 1; j < this.indexList.length; j++)
    {
      if ((paramArrayOfInt != null) && (ArrayUtil.find(paramArrayOfInt, j) != -1))
        continue;
      localTable.createAdjustedIndex(this.indexList[j], paramInt1, paramInt2);
    }
    localTable.triggerLists = this.triggerLists;
    return (Table)(Table)localTable;
  }

  Index getIndex(int[] paramArrayOfInt, int paramInt)
  {
    for (int i = 1; i < this.indexList.length; i++)
    {
      if ((paramArrayOfInt != null) && (ArrayUtil.find(paramArrayOfInt, i) != -1))
        continue;
      Index localIndex = this.indexList[i];
      int[] arrayOfInt = localIndex.getColumns();
      if (ArrayUtil.find(arrayOfInt, paramInt) != -1)
        return localIndex;
    }
    return null;
  }

  private void copyIndexes(Table paramTable, int paramInt1, int paramInt2, int paramInt3)
    throws HsqlException
  {
    for (int i = 1; i < getIndexCount(); i++)
    {
      Index localIndex1 = this.indexList[i];
      if (paramInt1 == i)
        continue;
      Index localIndex2 = paramTable.createAdjustedIndex(localIndex1, paramInt2, paramInt3);
      if (localIndex2 != null)
        continue;
      throw Trace.error(123);
    }
  }

  Table moveDefinitionPK(int[] paramArrayOfInt, boolean paramBoolean)
    throws HsqlException
  {
    if (((hasPrimaryKey()) && (paramArrayOfInt != null)) || ((!hasPrimaryKey()) && (paramArrayOfInt == null)))
      throw Trace.error(25);
    Table localTable = duplicate();
    for (int i = 0; i < this.columnCount; i++)
      localTable.addColumn(getColumn(i).duplicate(paramBoolean));
    localTable.createPrimaryKey(getIndex(0).getName(), paramArrayOfInt, true);
    localTable.constraintList = this.constraintList;
    for (i = 1; i < getIndexCount(); i++)
    {
      Index localIndex = getIndex(i);
      localTable.createAdjustedIndex(localIndex, -1, 0);
    }
    localTable.triggerLists = this.triggerLists;
    return localTable;
  }

  void updateConstraintsTables(Session paramSession, Table paramTable, int paramInt1, int paramInt2)
    throws HsqlException
  {
    int i = 0;
    int j = this.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = this.constraintList[i];
      localConstraint.replaceTable(paramTable, this, paramInt1, paramInt2);
      if (localConstraint.constType == 3)
        recompileCheckConstraint(paramSession, localConstraint);
      i++;
    }
  }

  private void recompileCheckConstraints(Session paramSession)
    throws HsqlException
  {
    int i = 0;
    int j = this.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = this.constraintList[i];
      if (localConstraint.constType == 3)
        recompileCheckConstraint(paramSession, localConstraint);
      i++;
    }
  }

  private void recompileCheckConstraint(Session paramSession, Constraint paramConstraint)
    throws HsqlException
  {
    String str = paramConstraint.core.check.getDDL();
    Tokenizer localTokenizer = new Tokenizer(str);
    Parser localParser = new Parser(paramSession, this.database, localTokenizer);
    Expression localExpression = localParser.parseExpression();
    paramConstraint.core.check = localExpression;
    localExpression.setLikeOptimised();
    Select localSelect = Expression.getCheckSelect(paramSession, this, localExpression);
    paramConstraint.core.checkFilter = localSelect.tFilter[0];
    paramConstraint.core.checkFilter.setAsCheckFilter();
    paramConstraint.core.mainTable = this;
  }

  void checkColumnInCheckConstraint(String paramString)
    throws HsqlException
  {
    int i = 0;
    int j = this.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = this.constraintList[i];
      if ((localConstraint.constType == 3) && (localConstraint.hasColumn(this, paramString)))
        throw Trace.error(197, localConstraint.getName());
      i++;
    }
  }

  void checkColumnInFKConstraint(int paramInt)
    throws HsqlException
  {
    int i = 0;
    int j = this.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = this.constraintList[i];
      if ((localConstraint.hasColumn(paramInt)) && ((localConstraint.getType() == 1) || (localConstraint.getType() == 0)))
        throw Trace.error(197, localConstraint.getName().name);
      i++;
    }
  }

  void checkColumnInFKConstraint(int paramInt1, int paramInt2)
    throws HsqlException
  {
    int i = 0;
    int j = this.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = this.constraintList[i];
      if ((localConstraint.hasColumn(paramInt1)) && (localConstraint.getType() == 0) && ((paramInt2 == localConstraint.getUpdateAction()) || (paramInt2 == localConstraint.getDeleteAction())))
        throw Trace.error(197, localConstraint.getName().name);
      i++;
    }
  }

  private void renameColumnInCheckConstraints(String paramString1, String paramString2, boolean paramBoolean)
    throws HsqlException
  {
    int i = 0;
    int j = this.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = this.constraintList[i];
      if (localConstraint.constType == 3)
      {
        Expression.Collector localCollector = new Expression.Collector();
        localCollector.addAll(localConstraint.core.check, 2);
        Iterator localIterator = localCollector.iterator();
        while (localIterator.hasNext())
        {
          Expression localExpression = (Expression)localIterator.next();
          if (localExpression.getColumnName() != paramString1)
            continue;
          localExpression.setColumnName(paramString2, paramBoolean);
        }
      }
      i++;
    }
  }

  private void renameTableInCheckConstraints(Session paramSession, String paramString1, String paramString2)
    throws HsqlException
  {
    int i = 0;
    int j = this.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = this.constraintList[i];
      if (localConstraint.constType == 3)
      {
        Expression.Collector localCollector = new Expression.Collector();
        localCollector.addAll(localConstraint.core.check, 2);
        Iterator localIterator = localCollector.iterator();
        while (localIterator.hasNext())
        {
          Expression localExpression = (Expression)localIterator.next();
          if (localExpression.getTableName() != paramString1)
            continue;
          localExpression.setTableName(paramString2);
        }
      }
      i++;
    }
    recompileCheckConstraints(paramSession);
  }

  public int getColumnCount()
  {
    return this.columnCount;
  }

  public int getIndexCount()
  {
    return this.indexList.length;
  }

  int getIdentityColumn()
  {
    return this.identityColumn;
  }

  int getColumnNr(String paramString)
    throws HsqlException
  {
    int i = findColumn(paramString);
    if (i == -1)
      throw Trace.error(28, paramString);
    return i;
  }

  int findColumn(String paramString)
  {
    int i = this.columnList.getIndex(paramString);
    return i;
  }

  public Index getPrimaryIndex()
  {
    return getIndex(0);
  }

  public int[] getPrimaryKey()
  {
    return this.primaryKeyCols;
  }

  public int[] getPrimaryKeyTypes()
  {
    return this.primaryKeyTypes;
  }

  public boolean hasPrimaryKey()
  {
    return this.primaryKeyCols.length != 0;
  }

  int[] getBestRowIdentifiers()
  {
    return this.bestRowIdentifierCols;
  }

  boolean isBestRowIdentifiersStrict()
  {
    return this.bestRowIdentifierStrict;
  }

  void setBestRowIdentifiers()
  {
    Object localObject = null;
    int i = 0;
    boolean bool = false;
    int j = 0;
    if (this.colNullable == null)
      return;
    this.bestIndex = null;
    this.bestIndexForColumn = new int[this.columnList.size()];
    ArrayUtil.fillArray(this.bestIndexForColumn, -1);
    for (int k = 0; k < this.indexList.length; k++)
    {
      Index localIndex = this.indexList[k];
      int[] arrayOfInt = localIndex.getColumns();
      int m = localIndex.getVisibleColumns();
      if (k == 0)
      {
        if (!hasPrimaryKey())
          continue;
        bool = true;
      }
      if (this.bestIndexForColumn[arrayOfInt[0]] == -1)
        this.bestIndexForColumn[arrayOfInt[0]] = k;
      if (!localIndex.isUnique())
      {
        if (this.bestIndex != null)
          continue;
        this.bestIndex = localIndex;
      }
      else
      {
        int n = 0;
        for (int i1 = 0; i1 < m; i1++)
        {
          if (this.colNullable[arrayOfInt[i1]] != 0)
            continue;
          n++;
        }
        if (this.bestIndex != null)
          this.bestIndex = localIndex;
        if (n == m)
        {
          if ((localObject != null) && (i == j) && (m >= i))
            continue;
          localObject = arrayOfInt;
          i = m;
          j = m;
          bool = true;
        }
        else
        {
          if ((bool) || ((localObject != null) && (m >= i) && (n <= j)))
            continue;
          localObject = arrayOfInt;
          i = m;
          j = n;
        }
      }
    }
    this.bestRowIdentifierCols = ((localObject == null) || (i == localObject.length) ? localObject : ArrayUtil.arraySlice(localObject, 0, i));
    this.bestRowIdentifierStrict = bool;
    if (hasPrimaryKey())
      this.bestIndex = getPrimaryIndex();
  }

  void setDefaultExpression(int paramInt, Expression paramExpression)
  {
    Column localColumn = getColumn(paramInt);
    localColumn.setDefaultExpression(paramExpression);
    this.colDefaults[paramInt] = localColumn.getDefaultExpression();
    resetDefaultsFlag();
  }

  void resetDefaultsFlag()
  {
    this.hasDefaultValues = false;
    for (int i = 0; i < this.columnCount; i++)
      this.hasDefaultValues = ((this.hasDefaultValues) || (this.colDefaults[i] != null));
  }

  DataFileCache getCache()
  {
    return this.cache;
  }

  Index getIndexForColumn(Session paramSession, int paramInt)
  {
    int i = this.bestIndexForColumn[paramInt];
    if ((i == -1) && ((this.tableType == 1) || (this.tableType == 0)))
      try
      {
        HsqlNameManager.HsqlName localHsqlName = this.database.nameManager.newAutoName("IDX");
        createIndex(paramSession, new int[] { paramInt }, localHsqlName, false, false, false);
        i = this.bestIndexForColumn[paramInt];
      }
      catch (Exception localException)
      {
      }
    return i == -1 ? null : getIndex(i);
  }

  Index getIndexForColumns(boolean[] paramArrayOfBoolean)
  {
    Object localObject = null;
    int i = 0;
    for (int j = 0; j < this.indexList.length; j++)
    {
      Index localIndex = this.indexList[j];
      boolean bool = ArrayUtil.containsAllTrueElements(paramArrayOfBoolean, localIndex.colCheck);
      if ((!bool) || (localIndex.getVisibleColumns() <= i))
        continue;
      i = localIndex.getVisibleColumns();
      localObject = localIndex;
    }
    return localObject;
  }

  Index getIndexForColumns(int[] paramArrayOfInt, boolean paramBoolean)
    throws HsqlException
  {
    int i = 0;
    int j = getIndexCount();
    while (i < j)
    {
      Index localIndex = getIndex(i);
      int[] arrayOfInt = localIndex.getColumns();
      if ((ArrayUtil.haveEqualArrays(arrayOfInt, paramArrayOfInt, paramArrayOfInt.length)) && ((!paramBoolean) || (localIndex.isUnique())))
        return localIndex;
      i++;
    }
    return null;
  }

  public int[] getIndexRootsArray()
  {
    int[] arrayOfInt = new int[getIndexCount()];
    for (int i = 0; i < getIndexCount(); i++)
      arrayOfInt[i] = this.indexList[i].getRoot();
    return arrayOfInt;
  }

  String getIndexRoots()
  {
    String str = StringUtil.getList(getIndexRootsArray(), " ", "");
    StringBuffer localStringBuffer = new StringBuffer(str);
    localStringBuffer.append(' ');
    localStringBuffer.append(this.identitySequence.peek());
    return localStringBuffer.toString();
  }

  public void setIndexRoots(int[] paramArrayOfInt)
    throws HsqlException
  {
    Trace.check(this.isCached, 22);
    for (int i = 0; i < getIndexCount(); i++)
    {
      int j = paramArrayOfInt[i];
      CachedRow localCachedRow = null;
      if (j != -1)
        localCachedRow = (CachedRow)this.rowStore.get(j);
      Node localNode = null;
      if (localCachedRow != null)
        localNode = localCachedRow.getNode(i);
      this.indexList[i].setRoot(null, localNode);
    }
  }

  void setIndexRoots(String paramString)
    throws HsqlException
  {
    Trace.check(this.isCached, 22);
    Tokenizer localTokenizer = new Tokenizer(paramString);
    int[] arrayOfInt = new int[getIndexCount()];
    for (int i = 0; i < getIndexCount(); i++)
    {
      int j = localTokenizer.getInt();
      arrayOfInt[i] = j;
    }
    setIndexRoots(arrayOfInt);
    long l = localTokenizer.getBigint();
    this.identitySequence.reset(l);
  }

  void createPrimaryKey(int[] paramArrayOfInt)
    throws HsqlException
  {
    createPrimaryKey(null, paramArrayOfInt, false);
  }

  void createPrimaryKey()
    throws HsqlException
  {
    createPrimaryKey(null, null, false);
  }

  void createPrimaryKey(HsqlNameManager.HsqlName paramHsqlName, int[] paramArrayOfInt, boolean paramBoolean)
    throws HsqlException
  {
    if (this.primaryKeyCols != null)
      Trace.doAssert(false, "Table.createPrimaryKey(column)");
    if (paramArrayOfInt == null)
      paramArrayOfInt = new int[0];
    else
      for (i = 0; i < paramArrayOfInt.length; i++)
      {
        if (paramBoolean)
          getColumn(paramArrayOfInt[i]).setNullable(false);
        getColumn(paramArrayOfInt[i]).setPrimaryKey(true);
      }
    this.primaryKeyCols = paramArrayOfInt;
    this.colTypes = new int[this.columnCount];
    this.colDefaults = new Expression[this.columnCount];
    this.colSizes = new int[this.columnCount];
    this.colScales = new int[this.columnCount];
    this.colNullable = new boolean[this.columnCount];
    this.defaultColumnMap = new int[this.columnCount];
    for (int i = 0; i < this.columnCount; i++)
      setColumnTypeVars(i);
    this.primaryKeyTypes = new int[this.primaryKeyCols.length];
    ArrayUtil.copyColumnValues(this.colTypes, this.primaryKeyCols, this.primaryKeyTypes);
    this.primaryKeyColsSequence = new int[this.primaryKeyCols.length];
    ArrayUtil.fillSequence(this.primaryKeyColsSequence);
    resetDefaultsFlag();
    HsqlNameManager.HsqlName localHsqlName = paramHsqlName != null ? paramHsqlName : this.database.nameManager.newAutoName("IDX");
    createPrimaryIndex(paramArrayOfInt, localHsqlName);
    setBestRowIdentifiers();
  }

  void setColumnTypeVars(int paramInt)
  {
    Column localColumn = getColumn(paramInt);
    this.colTypes[paramInt] = localColumn.getType();
    this.colSizes[paramInt] = localColumn.getSize();
    this.colScales[paramInt] = localColumn.getScale();
    this.colNullable[paramInt] = localColumn.isNullable();
    this.defaultColumnMap[paramInt] = paramInt;
    if (localColumn.isIdentity())
      this.identitySequence.reset(localColumn.identityStart, localColumn.identityIncrement);
    this.colDefaults[paramInt] = localColumn.getDefaultExpression();
  }

  HsqlNameManager.HsqlName makeSysPKName()
    throws HsqlException
  {
    return this.database.nameManager.newAutoName("PK");
  }

  void createPrimaryIndex(int[] paramArrayOfInt, HsqlNameManager.HsqlName paramHsqlName)
    throws HsqlException
  {
    int[] arrayOfInt = new int[paramArrayOfInt.length];
    for (int i = 0; i < paramArrayOfInt.length; i++)
      arrayOfInt[i] = this.colTypes[paramArrayOfInt[i]];
    Index localIndex = new Index(this.database, paramHsqlName, this, paramArrayOfInt, arrayOfInt, true, true, true, false, paramArrayOfInt, arrayOfInt, this.isTemp);
    addIndex(localIndex);
  }

  private Index createAdjustedIndex(Index paramIndex, int paramInt1, int paramInt2)
    throws HsqlException
  {
    int[] arrayOfInt1 = (int[])ArrayUtil.resizeArray(paramIndex.getColumns(), paramIndex.getVisibleColumns());
    int[] arrayOfInt2 = ArrayUtil.toAdjustedColumnArray(arrayOfInt1, paramInt1, paramInt2);
    if (arrayOfInt2.length != paramIndex.getVisibleColumns())
      return null;
    return createIndexStructure(arrayOfInt2, paramIndex.getName(), paramIndex.isUnique(), paramIndex.isConstraint, paramIndex.isForward);
  }

  Index createIndex(Session paramSession, int[] paramArrayOfInt, HsqlNameManager.HsqlName paramHsqlName, boolean paramBoolean1, boolean paramBoolean2, boolean paramBoolean3)
    throws HsqlException
  {
    int i = createIndexStructureGetNo(paramArrayOfInt, paramHsqlName, paramBoolean1, paramBoolean2, paramBoolean3);
    Index localIndex1 = this.indexList[i];
    Index localIndex2 = getPrimaryIndex();
    RowIterator localRowIterator = localIndex2.firstRow(paramSession);
    int j = 0;
    Object localObject1 = null;
    Object localObject2;
    Node localNode;
    try
    {
      while (localRowIterator.hasNext())
      {
        Row localRow = localRowIterator.next();
        localObject2 = localRow.getNode(i - 1);
        localNode = Node.newNode(localRow, i, this);
        localNode.nNext = ((Node)localObject2).nNext;
        ((Node)localObject2).nNext = localNode;
        j++;
        localIndex1.insert(paramSession, localRow, i);
      }
      return localIndex1;
    }
    catch (OutOfMemoryError localOutOfMemoryError)
    {
      localObject1 = Trace.error(72);
    }
    catch (HsqlException localHsqlException)
    {
      localObject1 = localHsqlException;
    }
    localRowIterator = localIndex2.firstRow(paramSession);
    for (int k = 0; k < j; k++)
    {
      localObject2 = localRowIterator.next();
      localNode = ((Row)localObject2).getNode(0);
      int m = i;
      while (true)
      {
        m--;
        if (m <= 0)
          break;
        localNode = localNode.nNext;
      }
      localNode.nNext = localNode.nNext.nNext;
    }
    this.indexList = ((Index[])ArrayUtil.toAdjustedArray(this.indexList, null, i, -1));
    setBestRowIdentifiers();
    throw ((Throwable)localObject1);
  }

  Index createIndexStructure(int[] paramArrayOfInt, HsqlNameManager.HsqlName paramHsqlName, boolean paramBoolean1, boolean paramBoolean2, boolean paramBoolean3)
    throws HsqlException
  {
    int i = createIndexStructureGetNo(paramArrayOfInt, paramHsqlName, paramBoolean1, paramBoolean2, paramBoolean3);
    return this.indexList[i];
  }

  int createIndexStructureGetNo(int[] paramArrayOfInt, HsqlNameManager.HsqlName paramHsqlName, boolean paramBoolean1, boolean paramBoolean2, boolean paramBoolean3)
    throws HsqlException
  {
    if (this.primaryKeyCols == null)
      Trace.doAssert(false, "createIndex");
    int i = paramArrayOfInt.length;
    int[] arrayOfInt1 = new int[i];
    int[] arrayOfInt2 = new int[i];
    for (int j = 0; j < i; j++)
    {
      arrayOfInt1[j] = paramArrayOfInt[j];
      arrayOfInt2[j] = this.colTypes[arrayOfInt1[j]];
    }
    int[] arrayOfInt3 = getPrimaryKey();
    int[] arrayOfInt4 = getPrimaryKeyTypes();
    Index localIndex = new Index(this.database, paramHsqlName, this, arrayOfInt1, arrayOfInt2, false, paramBoolean1, paramBoolean2, paramBoolean3, arrayOfInt3, arrayOfInt4, this.isTemp);
    int k = addIndex(localIndex);
    setBestRowIdentifiers();
    return k;
  }

  private int addIndex(Index paramIndex)
  {
    for (int i = 0; i < this.indexList.length; i++)
    {
      Index localIndex = this.indexList[i];
      int j = paramIndex.getIndexOrderValue() - localIndex.getIndexOrderValue();
      if (j < 0)
        break;
    }
    this.indexList = ((Index[])ArrayUtil.toAdjustedArray(this.indexList, paramIndex, i, 1));
    return i;
  }

  boolean isIndexingMutable()
  {
    return !isIndexCached();
  }

  void checkDropIndex(String paramString, HashSet paramHashSet, boolean paramBoolean)
    throws HsqlException
  {
    Index localIndex = getIndex(paramString);
    if (localIndex == null)
      throw Trace.error(26, paramString);
    if ((!paramBoolean) && (localIndex.equals(getIndex(0))))
      throw Trace.error(25, paramString);
    int i = 0;
    int j = this.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = this.constraintList[i];
      if ((paramHashSet == null) || (!paramHashSet.contains(localConstraint)))
      {
        if (localConstraint.isIndexFK(localIndex))
          throw Trace.error(50, paramString);
        if (localConstraint.isIndexUnique(localIndex))
          throw Trace.error(56, paramString);
      }
      i++;
    }
  }

  public boolean isEmpty(Session paramSession)
  {
    if (getIndexCount() == 0)
      return true;
    return getIndex(0).isEmpty(paramSession);
  }

  int[] getColumnMap()
  {
    return this.defaultColumnMap;
  }

  int[] getNewColumnMap()
  {
    return new int[this.columnCount];
  }

  boolean[] getNewColumnCheckList()
  {
    return new boolean[this.columnCount];
  }

  public Object[] getEmptyRowData()
  {
    return new Object[this.columnCount];
  }

  Object[] getNewRowData(Session paramSession, boolean[] paramArrayOfBoolean)
    throws HsqlException
  {
    Object[] arrayOfObject = new Object[this.columnCount];
    if ((paramArrayOfBoolean != null) && (this.hasDefaultValues))
      for (int i = 0; i < this.columnCount; i++)
      {
        Expression localExpression = this.colDefaults[i];
        if ((paramArrayOfBoolean[i] != 0) || (localExpression == null))
          continue;
        arrayOfObject[i] = localExpression.getValue(paramSession, this.colTypes[i]);
      }
    return arrayOfObject;
  }

  void dropIndex(Session paramSession, String paramString)
    throws HsqlException
  {
    int i = getIndexIndex(paramString);
    this.indexList = ((Index[])ArrayUtil.toAdjustedArray(this.indexList, null, i, -1));
    setBestRowIdentifiers();
    dropIndexFromRows(paramSession, i);
  }

  void dropIndexFromRows(Session paramSession, int paramInt)
    throws HsqlException
  {
    RowIterator localRowIterator = getPrimaryIndex().firstRow(paramSession);
    while (localRowIterator.hasNext())
    {
      Row localRow = localRowIterator.next();
      int i = paramInt - 1;
      for (Node localNode = localRow.getNode(0); i-- > 0; localNode = localNode.nNext);
      localNode.nNext = localNode.nNext.nNext;
    }
  }

  void moveData(Session paramSession, Table paramTable, int paramInt1, int paramInt2)
    throws HsqlException
  {
    Object localObject = null;
    Column localColumn = null;
    if ((paramInt2 >= 0) && (paramInt1 != -1))
    {
      localColumn = getColumn(paramInt1);
      localObject = localColumn.getDefaultValue(paramSession);
    }
    RowIterator localRowIterator = paramTable.getPrimaryIndex().firstRow(paramSession);
    while (localRowIterator.hasNext())
    {
      Row localRow1 = localRowIterator.next();
      Object[] arrayOfObject1 = localRow1.getData();
      Object[] arrayOfObject2 = getEmptyRowData();
      if ((paramInt2 == 0) && (paramInt1 != -1))
        localObject = Column.convertObject(paramSession, arrayOfObject1[paramInt1], localColumn.getType(), localColumn.getSize(), localColumn.getScale());
      ArrayUtil.copyAdjustArray(arrayOfObject1, arrayOfObject2, localObject, paramInt1, paramInt2);
      setIdentityColumn(paramSession, arrayOfObject2);
      enforceNullConstraints(arrayOfObject2);
      Row localRow2 = newRow(arrayOfObject2);
      indexRow(paramSession, localRow2);
    }
    paramTable.drop();
  }

  int insert(Session paramSession, Result paramResult)
    throws HsqlException
  {
    Record localRecord = paramResult.rRoot;
    int i = 0;
    fireAll(paramSession, 3);
    while (localRecord != null)
    {
      insertRow(paramSession, localRecord.data);
      localRecord = localRecord.next;
      i++;
    }
    fireAll(paramSession, 0);
    return i;
  }

  void insert(Session paramSession, Object[] paramArrayOfObject)
    throws HsqlException
  {
    fireAll(paramSession, 3);
    insertRow(paramSession, paramArrayOfObject);
    fireAll(paramSession, 0);
  }

  private void insertRow(Session paramSession, Object[] paramArrayOfObject)
    throws HsqlException
  {
    if (this.triggerLists[9] != null)
      fireAll(paramSession, 9, null, paramArrayOfObject);
    setIdentityColumn(paramSession, paramArrayOfObject);
    checkRowDataInsert(paramSession, paramArrayOfObject);
    insertNoCheck(paramSession, paramArrayOfObject);
    if (this.triggerLists[6] != null)
    {
      fireAll(paramSession, 6, null, paramArrayOfObject);
      checkRowDataInsert(paramSession, paramArrayOfObject);
    }
  }

  void insertIntoTable(Session paramSession, Result paramResult)
    throws HsqlException
  {
    insertResult(paramSession, paramResult);
    if (!this.isLogged)
      return;
    for (Record localRecord = paramResult.rRoot; localRecord != null; localRecord = localRecord.next)
      this.database.logger.writeInsertStatement(paramSession, this, localRecord.data);
  }

  private void insertNoCheck(Session paramSession, Object[] paramArrayOfObject)
    throws HsqlException
  {
    Row localRow = newRow(paramArrayOfObject);
    indexRow(paramSession, localRow);
    if (paramSession != null)
      paramSession.addInsertAction(this, localRow);
    if (this.isLogged)
      this.database.logger.writeInsertStatement(paramSession, this, paramArrayOfObject);
  }

  public void insertNoCheckFromLog(Session paramSession, Object[] paramArrayOfObject)
    throws HsqlException
  {
    Row localRow = newRow(paramArrayOfObject);
    updateIdentityValue(paramArrayOfObject);
    indexRow(paramSession, localRow);
    if (paramSession != null)
      paramSession.addInsertAction(this, localRow);
  }

  void insertNoCheckRollback(Session paramSession, Row paramRow, boolean paramBoolean)
    throws HsqlException
  {
    Row localRow = restoreRow(paramRow);
    indexRow(paramSession, localRow);
    if ((paramBoolean) && (this.isLogged))
      this.database.logger.writeInsertStatement(paramSession, this, paramRow.getData());
  }

  int insertSys(Result paramResult)
    throws HsqlException
  {
    Record localRecord = paramResult.rRoot;
    for (int i = 0; localRecord != null; i++)
    {
      insertData(null, localRecord.data);
      localRecord = localRecord.next;
    }
    return i;
  }

  int insertResult(Session paramSession, Result paramResult)
    throws HsqlException
  {
    Record localRecord = paramResult.rRoot;
    for (int i = 0; localRecord != null; i++)
    {
      Object[] arrayOfObject = (Object[])ArrayUtil.resizeArrayIfDifferent(localRecord.data, this.columnCount);
      insertData(paramSession, arrayOfObject);
      localRecord = localRecord.next;
    }
    return i;
  }

  public void insertFromScript(Object[] paramArrayOfObject)
    throws HsqlException
  {
    updateIdentityValue(paramArrayOfObject);
    insertData(null, paramArrayOfObject);
  }

  public void insertData(Session paramSession, Object[] paramArrayOfObject)
    throws HsqlException
  {
    Row localRow = newRow(paramArrayOfObject);
    indexRow(paramSession, localRow);
    commitRowToStore(localRow);
  }

  public void insertSys(Object[] paramArrayOfObject)
    throws HsqlException
  {
    Row localRow = newRow(paramArrayOfObject);
    indexRow(null, localRow);
  }

  protected void insertFromTextSource(CachedRow paramCachedRow)
    throws HsqlException
  {
    Object[] arrayOfObject = paramCachedRow.getData();
    updateIdentityValue(arrayOfObject);
    enforceFieldValueLimits(arrayOfObject, this.defaultColumnMap);
    enforceNullConstraints(arrayOfObject);
    indexRow(null, paramCachedRow);
  }

  protected void enforceNullConstraints(Object[] paramArrayOfObject)
    throws HsqlException
  {
    for (int i = 0; i < this.columnCount; i++)
    {
      if ((paramArrayOfObject[i] != null) || (this.colNullable[i] != 0))
        continue;
      Trace.throwerror(10, "column: " + getColumn(i).columnName.name + " table: " + this.tableName.name);
    }
  }

  protected void setIdentityColumn(Session paramSession, Object[] paramArrayOfObject)
    throws HsqlException
  {
    if (this.identityColumn != -1)
    {
      Object localObject = (Number)paramArrayOfObject[this.identityColumn];
      if (localObject == null)
      {
        if (this.colTypes[this.identityColumn] == 4)
          localObject = ValuePool.getInt((int)this.identitySequence.getValue());
        else
          localObject = ValuePool.getLong(this.identitySequence.getValue());
        paramArrayOfObject[this.identityColumn] = localObject;
      }
      else
      {
        this.identitySequence.getValue(((Number)localObject).longValue());
      }
      if (paramSession != null)
        paramSession.setLastIdentity((Number)localObject);
    }
  }

  protected void updateIdentityValue(Object[] paramArrayOfObject)
    throws HsqlException
  {
    if (this.identityColumn != -1)
    {
      Number localNumber = (Number)paramArrayOfObject[this.identityColumn];
      if (localNumber != null)
        this.identitySequence.getValue(localNumber.longValue());
    }
  }

  void enforceFieldValueLimits(Object[] paramArrayOfObject, int[] paramArrayOfInt)
    throws HsqlException
  {
    if (this.sqlEnforceSize)
    {
      if (paramArrayOfInt == null)
        paramArrayOfInt = this.defaultColumnMap;
      for (int i = 0; i < paramArrayOfInt.length; i++)
      {
        int j = paramArrayOfInt[i];
        if (((this.colTypes[j] != 93) && (this.colSizes[j] == 0)) || (paramArrayOfObject[j] == null))
          continue;
        paramArrayOfObject[j] = Column.enforceSize(paramArrayOfObject[j], this.colTypes[j], this.colSizes[j], this.colScales[j], true);
      }
    }
  }

  void fireAll(Session paramSession, int paramInt, Object[] paramArrayOfObject1, Object[] paramArrayOfObject2)
  {
    if (!this.database.isReferentialIntegrity())
      return;
    HsqlArrayList localHsqlArrayList = this.triggerLists[paramInt];
    if (localHsqlArrayList == null)
      return;
    int i = 0;
    int j = localHsqlArrayList.size();
    while (i < j)
    {
      TriggerDef localTriggerDef = (TriggerDef)localHsqlArrayList.get(i);
      localTriggerDef.pushPair(paramSession, paramArrayOfObject1, paramArrayOfObject2);
      i++;
    }
  }

  void fireAll(Session paramSession, int paramInt)
  {
    if (this.triggerLists[paramInt] != null)
      fireAll(paramSession, paramInt, null, null);
  }

  void addTrigger(TriggerDef paramTriggerDef)
  {
    if (this.triggerLists[paramTriggerDef.vectorIndex] == null)
      this.triggerLists[paramTriggerDef.vectorIndex] = new HsqlArrayList();
    this.triggerLists[paramTriggerDef.vectorIndex].add(paramTriggerDef);
  }

  void dropTrigger(String paramString)
  {
    int i = 12;
    for (int j = 0; j < i; j++)
    {
      HsqlArrayList localHsqlArrayList = this.triggerLists[j];
      if (localHsqlArrayList == null)
        continue;
      for (int k = localHsqlArrayList.size() - 1; k >= 0; k--)
      {
        TriggerDef localTriggerDef = (TriggerDef)localHsqlArrayList.get(k);
        if (!localTriggerDef.name.name.equals(paramString))
          continue;
        localHsqlArrayList.remove(k);
        localTriggerDef.terminate();
      }
      if (!localHsqlArrayList.isEmpty())
        continue;
      this.triggerLists[j] = null;
    }
  }

  void dropTriggers()
  {
    int i = 12;
    for (int j = 0; j < i; j++)
    {
      HsqlArrayList localHsqlArrayList = this.triggerLists[j];
      if (localHsqlArrayList == null)
        continue;
      for (int k = localHsqlArrayList.size() - 1; k >= 0; k--)
      {
        TriggerDef localTriggerDef = (TriggerDef)localHsqlArrayList.get(k);
        localTriggerDef.terminate();
      }
      this.triggerLists[j] = null;
    }
  }

  static void checkCascadeDelete(Session paramSession, Table paramTable, HashMappedList paramHashMappedList, Row paramRow, boolean paramBoolean, HashSet paramHashSet)
    throws HsqlException
  {
    int i = 0;
    int j = paramTable.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = paramTable.constraintList[i];
      if ((localConstraint.getType() == 1) && (localConstraint.getRef() != null))
      {
        RowIterator localRowIterator = localConstraint.findFkRef(paramSession, paramRow.getData(), paramBoolean);
        if (localRowIterator.hasNext())
          try
          {
            Object localObject1;
            if (localConstraint.core.deleteAction == 3)
            {
              if (localConstraint.core.mainTable == localConstraint.core.refTable)
              {
                localObject1 = localRowIterator.next();
                if (paramRow.equals(localObject1))
                {
                  localRowIterator.release();
                  break label650;
                }
              }
              throw Trace.error(8, 100, new Object[] { localConstraint.core.fkName.name, localConstraint.core.refTable.getName().name });
            }
            else
            {
              localObject1 = localConstraint.getRef();
              int k = ((Table)localObject1).getNextConstraintIndex(0, 1) != -1 ? 1 : 0;
              if ((!paramBoolean) && (k == 0))
              {
                localRowIterator.release();
              }
              else
              {
                Index localIndex = localConstraint.getRefIndex();
                int[] arrayOfInt1 = localConstraint.getMainColumns();
                int[] arrayOfInt2 = localConstraint.getRefColumns();
                Object[] arrayOfObject1 = paramRow.getData();
                int m = (localConstraint.getDeleteAction() == 2) || (localConstraint.getDeleteAction() == 4) ? 1 : 0;
                HashMappedList localHashMappedList = null;
                if (m != 0)
                {
                  localHashMappedList = (HashMappedList)paramHashMappedList.get(localObject1);
                  if (localHashMappedList == null)
                  {
                    localHashMappedList = new HashMappedList();
                    paramHashMappedList.add(localObject1, localHashMappedList);
                  }
                }
                while (true)
                {
                  Row localRow = localRowIterator.next();
                  if ((localRow == null) || (localRow.isCascadeDeleted()) || (localIndex.compareRowNonUnique(paramSession, arrayOfObject1, arrayOfInt1, localRow.getData()) != 0))
                    break;
                  if (m != 0)
                  {
                    Object[] arrayOfObject2 = ((Table)localObject1).getEmptyRowData();
                    System.arraycopy(localRow.getData(), 0, arrayOfObject2, 0, arrayOfObject2.length);
                    if (localConstraint.getDeleteAction() == 2)
                      for (n = 0; n < arrayOfInt2.length; n++)
                        arrayOfObject2[arrayOfInt2[n]] = null;
                    for (int n = 0; n < arrayOfInt2.length; n++)
                    {
                      Column localColumn = ((Table)localObject1).getColumn(arrayOfInt2[n]);
                      arrayOfObject2[arrayOfInt2[n]] = localColumn.getDefaultValue(paramSession);
                    }
                    if ((k != 0) && (paramHashSet.add(localConstraint)))
                    {
                      checkCascadeUpdate(paramSession, (Table)localObject1, null, localRow, arrayOfObject2, arrayOfInt2, null, paramHashSet);
                      paramHashSet.remove(localConstraint);
                    }
                    if ((paramBoolean) && ((localObject1 != paramTable) || (!localRow.equals(paramRow))))
                      mergeUpdate(localHashMappedList, localRow, arrayOfObject2, arrayOfInt2);
                  }
                  else if (k != 0)
                  {
                    if (localObject1 != paramTable)
                    {
                      if (paramHashSet.add(localConstraint))
                      {
                        checkCascadeDelete(paramSession, (Table)localObject1, paramHashMappedList, localRow, paramBoolean, paramHashSet);
                        paramHashSet.remove(localConstraint);
                      }
                    }
                    else if (localRow != paramRow)
                      checkCascadeDelete(paramSession, (Table)localObject1, paramHashMappedList, localRow, paramBoolean, paramHashSet);
                  }
                  if ((!paramBoolean) || (m != 0) || (localRow.isCascadeDeleted()))
                    continue;
                  ((Table)localObject1).deleteNoRefCheck(paramSession, localRow);
                }
              }
            }
          }
          finally
          {
            localRowIterator.release();
          }
      }
      label650: i++;
    }
  }

  static void checkCascadeUpdate(Session paramSession, Table paramTable1, HashMappedList paramHashMappedList, Row paramRow, Object[] paramArrayOfObject, int[] paramArrayOfInt, Table paramTable2, HashSet paramHashSet)
    throws HsqlException
  {
    int i = 0;
    int j = paramTable1.constraintList.length;
    while (i < j)
    {
      Constraint localConstraint = paramTable1.constraintList[i];
      if ((localConstraint.getType() == 0) && (localConstraint.getRef() != null))
      {
        if (((paramTable2 == null) || (localConstraint.getMain() != paramTable2)) && (ArrayUtil.countCommonElements(paramArrayOfInt, localConstraint.getRefColumns()) != 0))
          localConstraint.hasMainRef(paramSession, paramArrayOfObject);
      }
      else if ((localConstraint.getType() == 1) && (localConstraint.getRef() != null))
      {
        int[] arrayOfInt1 = ArrayUtil.commonElements(paramArrayOfInt, localConstraint.getMainColumns());
        if (arrayOfInt1 != null)
        {
          int[] arrayOfInt2 = localConstraint.getMainColumns();
          int[] arrayOfInt3 = localConstraint.getRefColumns();
          int k = 1;
          for (int m = 0; m < arrayOfInt2.length; m++)
          {
            if (paramRow.getData()[arrayOfInt2[m]].equals(paramArrayOfObject[arrayOfInt2[m]]))
              continue;
            k = 0;
            break;
          }
          if (k == 0)
          {
            RowIterator localRowIterator = localConstraint.findFkRef(paramSession, paramRow.getData(), false);
            if (localRowIterator.hasNext())
            {
              if (localConstraint.core.updateAction == 3)
                throw Trace.error(8, 100, new Object[] { localConstraint.core.fkName.name, localConstraint.core.refTable.getName().name });
              Table localTable = localConstraint.getRef();
              int n = localTable.getNextConstraintIndex(0, 1) != -1 ? 1 : 0;
              Index localIndex = localConstraint.getRefIndex();
              HashMappedList localHashMappedList = (HashMappedList)paramHashMappedList.get(localTable);
              if (localHashMappedList == null)
              {
                localHashMappedList = new HashMappedList();
                paramHashMappedList.add(localTable, localHashMappedList);
              }
              for (Row localRow = localRowIterator.next(); (localRow != null) && (localIndex.compareRowNonUnique(paramSession, paramRow.getData(), arrayOfInt2, localRow.getData()) == 0); localRow = localRowIterator.next())
              {
                Object[] arrayOfObject = localTable.getEmptyRowData();
                System.arraycopy(localRow.getData(), 0, arrayOfObject, 0, arrayOfObject.length);
                int i1;
                if (localConstraint.getUpdateAction() == 2)
                  for (i1 = 0; i1 < arrayOfInt3.length; i1++)
                    arrayOfObject[arrayOfInt3[i1]] = null;
                if (localConstraint.getUpdateAction() == 4)
                {
                  for (i1 = 0; i1 < arrayOfInt3.length; i1++)
                  {
                    Column localColumn = localTable.getColumn(arrayOfInt3[i1]);
                    arrayOfObject[arrayOfInt3[i1]] = localColumn.getDefaultValue(paramSession);
                  }
                  if (paramHashSet.add(localConstraint))
                  {
                    checkCascadeUpdate(paramSession, localTable, paramHashMappedList, localRow, arrayOfObject, arrayOfInt3, null, paramHashSet);
                    paramHashSet.remove(localConstraint);
                  }
                }
                else
                {
                  for (i1 = 0; i1 < arrayOfInt2.length; i1++)
                    arrayOfObject[arrayOfInt3[i1]] = paramArrayOfObject[arrayOfInt2[i1]];
                  if (paramHashSet.add(localConstraint))
                  {
                    checkCascadeUpdate(paramSession, localTable, paramHashMappedList, localRow, arrayOfObject, arrayOfInt1, paramTable1, paramHashSet);
                    paramHashSet.remove(localConstraint);
                  }
                }
                mergeUpdate(localHashMappedList, localRow, arrayOfObject, arrayOfInt3);
              }
            }
          }
        }
      }
      i++;
    }
  }

  static void mergeUpdate(HashMappedList paramHashMappedList, Row paramRow, Object[] paramArrayOfObject, int[] paramArrayOfInt)
  {
    Object[] arrayOfObject = (Object[])paramHashMappedList.get(paramRow);
    if (arrayOfObject != null)
      for (int i = 0; i < paramArrayOfInt.length; i++)
        arrayOfObject[paramArrayOfInt[i]] = paramArrayOfObject[paramArrayOfInt[i]];
    paramHashMappedList.add(paramRow, paramArrayOfObject);
  }

  static boolean mergeKeepUpdate(Session paramSession, HashMappedList paramHashMappedList, int[] paramArrayOfInt1, int[] paramArrayOfInt2, Row paramRow, Object[] paramArrayOfObject)
    throws HsqlException
  {
    Object[] arrayOfObject = (Object[])paramHashMappedList.get(paramRow);
    if (arrayOfObject != null)
    {
      if ((Index.compareRows(paramSession, paramRow.getData(), paramArrayOfObject, paramArrayOfInt1, paramArrayOfInt2) != 0) && (Index.compareRows(paramSession, paramArrayOfObject, arrayOfObject, paramArrayOfInt1, paramArrayOfInt2) != 0))
        return false;
      for (int i = 0; i < paramArrayOfInt1.length; i++)
        paramArrayOfObject[paramArrayOfInt1[i]] = arrayOfObject[paramArrayOfInt1[i]];
      paramHashMappedList.put(paramRow, paramArrayOfObject);
    }
    else
    {
      paramHashMappedList.add(paramRow, paramArrayOfObject);
    }
    return true;
  }

  static void clearUpdateLists(HashMappedList paramHashMappedList)
  {
    for (int i = 0; i < paramHashMappedList.size(); i++)
    {
      HashMappedList localHashMappedList = (HashMappedList)paramHashMappedList.get(i);
      localHashMappedList.clear();
    }
  }

  int delete(Session paramSession, HsqlArrayList paramHsqlArrayList)
    throws HsqlException
  {
    HashSet localHashSet = this.constraintPath == null ? new HashSet() : this.constraintPath;
    this.constraintPath = null;
    HashMappedList localHashMappedList1 = this.tableUpdateList == null ? new HashMappedList() : this.tableUpdateList;
    this.tableUpdateList = null;
    Object localObject;
    if (this.database.isReferentialIntegrity())
      for (i = 0; i < paramHsqlArrayList.size(); i++)
      {
        localObject = (Row)paramHsqlArrayList.get(i);
        localHashSet.clear();
        checkCascadeDelete(paramSession, this, localHashMappedList1, (Row)localObject, false, localHashSet);
      }
    this.database.txManager.checkDelete(paramSession, paramHsqlArrayList);
    HashMappedList localHashMappedList2;
    for (int i = 0; i < localHashMappedList1.size(); i++)
    {
      localObject = (Table)localHashMappedList1.getKey(i);
      localHashMappedList2 = (HashMappedList)localHashMappedList1.get(i);
      this.database.txManager.checkDelete(paramSession, localHashMappedList2);
    }
    fireAll(paramSession, 4);
    if (this.database.isReferentialIntegrity())
      for (i = 0; i < paramHsqlArrayList.size(); i++)
      {
        localObject = (Row)paramHsqlArrayList.get(i);
        localHashSet.clear();
        checkCascadeDelete(paramSession, this, localHashMappedList1, (Row)localObject, true, localHashSet);
      }
    for (i = 0; i < paramHsqlArrayList.size(); i++)
    {
      localObject = (Row)paramHsqlArrayList.get(i);
      if (((Row)localObject).isCascadeDeleted())
        continue;
      deleteNoRefCheck(paramSession, (Row)localObject);
    }
    for (i = 0; i < localHashMappedList1.size(); i++)
    {
      localObject = (Table)localHashMappedList1.getKey(i);
      localHashMappedList2 = (HashMappedList)localHashMappedList1.get(i);
      ((Table)localObject).updateRowSet(paramSession, localHashMappedList2, null, false);
      localHashMappedList2.clear();
    }
    fireAll(paramSession, 1);
    localHashSet.clear();
    this.constraintPath = localHashSet;
    this.tableUpdateList = localHashMappedList1;
    return paramHsqlArrayList.size();
  }

  private void deleteNoRefCheck(Session paramSession, Row paramRow)
    throws HsqlException
  {
    Object[] arrayOfObject = paramRow.getData();
    fireAll(paramSession, 10, arrayOfObject, null);
    deleteNoCheck(paramSession, paramRow, true);
    fireAll(paramSession, 7, arrayOfObject, null);
  }

  private void deleteNoCheck(Session paramSession, Row paramRow, boolean paramBoolean)
    throws HsqlException
  {
    if (paramRow.isCascadeDeleted())
      return;
    Object[] arrayOfObject = paramRow.getData();
    paramRow = paramRow.getUpdatedRow();
    for (int i = this.indexList.length - 1; i >= 0; i--)
    {
      Node localNode = paramRow.getNode(i);
      this.indexList[i].delete(paramSession, localNode);
    }
    paramRow.delete();
    if (paramSession != null)
      paramSession.addDeleteAction(this, paramRow);
    if ((paramBoolean) && (this.isLogged))
      this.database.logger.writeDeleteStatement(paramSession, this, arrayOfObject);
  }

  public void deleteNoCheckFromLog(Session paramSession, Object[] paramArrayOfObject)
    throws HsqlException
  {
    Row localRow = null;
    RowIterator localRowIterator;
    Object localObject;
    if (hasPrimaryKey())
    {
      localRowIterator = getPrimaryIndex().findFirstRow(paramSession, paramArrayOfObject, this.primaryKeyColsSequence);
      localRow = localRowIterator.next();
    }
    else if (this.bestIndex == null)
    {
      localRowIterator = getPrimaryIndex().firstRow(paramSession);
      do
      {
        localRow = localRowIterator.next();
        if (localRow == null)
          break;
      }
      while (Index.compareRows(paramSession, localRow.getData(), paramArrayOfObject, this.defaultColumnMap, this.colTypes) != 0);
    }
    else
    {
      localRowIterator = this.bestIndex.findFirstRow(paramSession, paramArrayOfObject);
      while (true)
      {
        localRow = localRowIterator.next();
        if (localRow == null)
          break;
        localObject = localRow.getData();
        if (this.bestIndex.compareRowNonUnique(paramSession, paramArrayOfObject, this.bestIndex.getColumns(), localObject) != 0)
          localRow = null;
        else
          if (Index.compareRows(paramSession, localObject, paramArrayOfObject, this.defaultColumnMap, this.colTypes) != 0)
            continue;
      }
    }
    if (localRow == null)
      return;
    this.database.txManager.checkDelete(paramSession, localRow);
    for (int i = this.indexList.length - 1; i >= 0; i--)
    {
      localObject = localRow.getNode(i);
      this.indexList[i].delete(paramSession, (Node)localObject);
    }
    localRow.delete();
    if (paramSession != null)
      paramSession.addDeleteAction(this, localRow);
  }

  void deleteNoCheckRollback(Session paramSession, Row paramRow, boolean paramBoolean)
    throws HsqlException
  {
    paramRow = this.indexList[0].findRow(paramSession, paramRow);
    for (int i = this.indexList.length - 1; i >= 0; i--)
    {
      Node localNode = paramRow.getNode(i);
      this.indexList[i].delete(paramSession, localNode);
    }
    paramRow.delete();
    removeRowFromStore(paramRow);
    if ((paramBoolean) && (this.isLogged))
      this.database.logger.writeDeleteStatement(paramSession, this, paramRow.getData());
  }

  int update(Session paramSession, HashMappedList paramHashMappedList, int[] paramArrayOfInt)
    throws HsqlException
  {
    HashSet localHashSet = this.constraintPath == null ? new HashSet() : this.constraintPath;
    this.constraintPath = null;
    HashMappedList localHashMappedList1 = this.tableUpdateList == null ? new HashMappedList() : this.tableUpdateList;
    this.tableUpdateList = null;
    Object[] arrayOfObject;
    for (int i = 0; i < paramHashMappedList.size(); i++)
    {
      arrayOfObject = (Object[])paramHashMappedList.get(i);
      setIdentityColumn(paramSession, arrayOfObject);
      enforceFieldValueLimits(arrayOfObject, paramArrayOfInt);
      enforceNullConstraints(arrayOfObject);
    }
    Object localObject1;
    if (this.database.isReferentialIntegrity())
      for (i = 0; i < paramHashMappedList.size(); i++)
      {
        arrayOfObject = (Object[])paramHashMappedList.get(i);
        localObject1 = (Row)paramHashMappedList.getKey(i);
        checkCascadeUpdate(paramSession, this, localHashMappedList1, (Row)localObject1, arrayOfObject, paramArrayOfInt, null, localHashSet);
      }
    fireAll(paramSession, 5);
    HashMappedList localHashMappedList2 = (HashMappedList)localHashMappedList1.get(this);
    Object localObject2;
    if (localHashMappedList2 != null)
    {
      for (j = 0; j < localHashMappedList2.size(); j++)
      {
        localObject1 = (Row)localHashMappedList2.getKey(j);
        localObject2 = (Object[])localHashMappedList2.get(j);
        mergeKeepUpdate(paramSession, paramHashMappedList, paramArrayOfInt, this.colTypes, (Row)localObject1, localObject2);
      }
      localHashMappedList2.clear();
    }
    for (int j = 0; j < localHashMappedList1.size(); j++)
    {
      localObject1 = (Table)localHashMappedList1.getKey(j);
      localObject2 = (HashMappedList)localHashMappedList1.get(j);
      this.database.txManager.checkDelete(paramSession, (HashMappedList)localObject2);
    }
    this.database.txManager.checkDelete(paramSession, paramHashMappedList);
    for (j = 0; j < localHashMappedList1.size(); j++)
    {
      localObject1 = (Table)localHashMappedList1.getKey(j);
      localObject2 = (HashMappedList)localHashMappedList1.get(j);
      ((Table)localObject1).updateRowSet(paramSession, (HashMappedList)localObject2, null, false);
      ((HashMappedList)localObject2).clear();
    }
    updateRowSet(paramSession, paramHashMappedList, paramArrayOfInt, true);
    fireAll(paramSession, 2);
    localHashSet.clear();
    this.constraintPath = localHashSet;
    this.tableUpdateList = localHashMappedList1;
    clearUpdateLists(this.tableUpdateList);
    return paramHashMappedList.size();
  }

  void updateRowSet(Session paramSession, HashMappedList paramHashMappedList, int[] paramArrayOfInt, boolean paramBoolean)
    throws HsqlException
  {
    Row localRow;
    Object[] arrayOfObject;
    for (int i = paramHashMappedList.size() - 1; i >= 0; i--)
    {
      localRow = (Row)paramHashMappedList.getKey(i);
      arrayOfObject = (Object[])paramHashMappedList.get(i);
      if (localRow.isCascadeDeleted())
      {
        if (paramBoolean)
          throw Trace.error(199);
        paramHashMappedList.remove(i);
      }
      else
      {
        for (int j = 0; j < this.constraintList.length; j++)
        {
          Constraint localConstraint = this.constraintList[j];
          if (localConstraint.getType() != 3)
            continue;
          localConstraint.checkCheckConstraint(paramSession, arrayOfObject);
        }
        deleteNoCheck(paramSession, localRow, true);
      }
    }
    for (i = 0; i < paramHashMappedList.size(); i++)
    {
      localRow = (Row)paramHashMappedList.getKey(i);
      arrayOfObject = (Object[])paramHashMappedList.get(i);
      if (this.triggerLists[11] != null)
      {
        fireAll(paramSession, 11, localRow.getData(), arrayOfObject);
        checkRowDataUpdate(paramSession, arrayOfObject, paramArrayOfInt);
      }
      insertNoCheck(paramSession, arrayOfObject);
      if (this.triggerLists[8] == null)
        continue;
      fireAll(paramSession, 8, localRow.getData(), arrayOfObject);
      checkRowDataUpdate(paramSession, arrayOfObject, paramArrayOfInt);
    }
  }

  void checkRowDataInsert(Session paramSession, Object[] paramArrayOfObject)
    throws HsqlException
  {
    enforceFieldValueLimits(paramArrayOfObject, null);
    enforceNullConstraints(paramArrayOfObject);
    if (this.database.isReferentialIntegrity())
    {
      int i = 0;
      int j = this.constraintList.length;
      while (i < j)
      {
        this.constraintList[i].checkInsert(paramSession, paramArrayOfObject);
        i++;
      }
    }
  }

  void checkRowDataUpdate(Session paramSession, Object[] paramArrayOfObject, int[] paramArrayOfInt)
    throws HsqlException
  {
    enforceFieldValueLimits(paramArrayOfObject, paramArrayOfInt);
    enforceNullConstraints(paramArrayOfObject);
    for (int i = 0; i < this.constraintList.length; i++)
    {
      Constraint localConstraint = this.constraintList[i];
      if (localConstraint.getType() != 3)
        continue;
      localConstraint.checkCheckConstraint(paramSession, paramArrayOfObject);
    }
  }

  public boolean isCached()
  {
    return this.isCached;
  }

  boolean isIndexCached()
  {
    return this.isCached;
  }

  int getIndexIndex(String paramString)
  {
    Index[] arrayOfIndex = this.indexList;
    for (int i = 0; i < arrayOfIndex.length; i++)
      if (paramString.equals(arrayOfIndex[i].getName().name))
        return i;
    return -1;
  }

  Index getIndex(String paramString)
  {
    Index[] arrayOfIndex = this.indexList;
    int i = getIndexIndex(paramString);
    return i == -1 ? null : arrayOfIndex[i];
  }

  int getConstraintIndex(String paramString)
  {
    int i = 0;
    int j = this.constraintList.length;
    while (i < j)
    {
      if (this.constraintList[i].getName().name.equals(paramString))
        return i;
      i++;
    }
    return -1;
  }

  Constraint getConstraint(String paramString)
  {
    int i = getConstraintIndex(paramString);
    return i < 0 ? null : this.constraintList[i];
  }

  void removeConstraint(String paramString)
  {
    int i = getConstraintIndex(paramString);
    this.constraintList = ((Constraint[])ArrayUtil.toAdjustedArray(this.constraintList, null, i, -1));
  }

  Column getColumn(int paramInt)
  {
    return (Column)this.columnList.get(paramInt);
  }

  void renameColumn(Column paramColumn, String paramString, boolean paramBoolean)
    throws HsqlException
  {
    String str = paramColumn.columnName.name;
    int i = getColumnNr(str);
    this.columnList.setKey(i, paramString);
    paramColumn.columnName.rename(paramString, paramBoolean);
    renameColumnInCheckConstraints(str, paramString, paramBoolean);
  }

  public int[] getColumnTypes()
  {
    return this.colTypes;
  }

  public Index getIndex(int paramInt)
  {
    return this.indexList[paramInt];
  }

  public Index[] getIndexes()
  {
    return this.indexList;
  }

  CachedRow getRow(int paramInt, Node paramNode)
    throws HsqlException
  {
    if (this.isText)
    {
      CachedDataRow localCachedDataRow = (CachedDataRow)this.rowStore.get(paramInt);
      localCachedDataRow.nPrimaryNode = paramNode;
      return localCachedDataRow;
    }
    if (this.isCached)
      return (CachedRow)this.rowStore.get(paramInt);
    return null;
  }

  CachedRow getRow(int paramInt)
  {
    return (CachedRow)this.rowStore.get(paramInt);
  }

  CachedRow getRow(long paramLong)
  {
    return (CachedRow)this.rowStore.get((int)paramLong);
  }

  void removeRowFromStore(Row paramRow)
    throws HsqlException
  {
    if ((this.isCached) || ((this.isText) && (this.cache != null)))
      this.rowStore.remove(paramRow.getPos());
  }

  void releaseRowFromStore(Row paramRow)
    throws HsqlException
  {
    if ((this.isCached) || ((this.isText) && (this.cache != null)))
      this.rowStore.release(paramRow.getPos());
  }

  void commitRowToStore(Row paramRow)
  {
    if ((this.isText) && (this.cache != null))
      this.rowStore.commit(paramRow);
  }

  void indexRow(Session paramSession, Row paramRow)
    throws HsqlException
  {
    int i = 0;
    try
    {
      while (i < this.indexList.length)
      {
        this.indexList[i].insert(paramSession, paramRow, i);
        i++;
      }
    }
    catch (HsqlException localHsqlException)
    {
      Index localIndex = this.indexList[i];
      boolean bool = localIndex.isConstraint;
      i--;
      Object localObject;
      while (i >= 0)
      {
        localObject = paramRow.getNode(i);
        this.indexList[i].delete(paramSession, (Node)localObject);
        i--;
      }
      paramRow.delete();
      removeRowFromStore(paramRow);
      if (bool)
      {
        localObject = getUniqueOrPKConstraintForIndex(localIndex);
        String str = localObject == null ? localIndex.getName().name : ((Constraint)localObject).getName().name;
        throw Trace.error(104, new Object[] { str });
      }
      throw localHsqlException;
    }
  }

  void clearAllRows(Session paramSession)
  {
    for (int i = 0; i < this.indexList.length; i++)
      this.indexList[i].clearAll(paramSession);
    if (!this.isTemp)
    {
      this.identitySequence.reset();
      this.rowIdSequence.reset();
    }
  }

  void drop()
    throws HsqlException
  {
  }

  boolean isWritable()
  {
    return (!isDataReadOnly()) && (!this.database.databaseReadOnly) && ((!this.database.isFilesReadOnly()) || ((!this.isCached) && (!this.isText)));
  }

  String getCatalogName()
  {
    return this.database.getProperties().isPropertyTrue("hsqldb.catalogs") ? this.database.getURI() : null;
  }

  public String getSchemaName()
  {
    return this.tableName.schema.name;
  }

  public int getRowCount(Session paramSession)
    throws HsqlException
  {
    return getPrimaryIndex().size(paramSession);
  }

  public void resetRowId(Session paramSession)
    throws HsqlException
  {
    if (this.isCached)
      return;
    this.rowIdSequence = new NumberSequence(null, 0L, 1L, -5);
    RowIterator localRowIterator = getPrimaryIndex().firstRow(paramSession);
    while (localRowIterator.hasNext())
    {
      Row localRow = localRowIterator.next();
      int i = (int)this.rowIdSequence.getValue();
      localRow.setPos(i);
    }
  }

  Row newRow(Object[] paramArrayOfObject)
    throws HsqlException
  {
    Object localObject;
    try
    {
      if (this.isMemory)
      {
        localObject = new Row(this, paramArrayOfObject);
        int i = (int)this.rowIdSequence.getValue();
        ((Row)localObject).setPos(i);
      }
      else
      {
        localObject = CachedRow.newCachedRow(this, paramArrayOfObject);
        this.rowStore.add((CachedObject)localObject);
      }
    }
    catch (IOException localIOException)
    {
      throw new HsqlException(localIOException, Trace.getMessage(98), 98);
    }
    return (Row)localObject;
  }

  Row restoreRow(Row paramRow)
    throws HsqlException
  {
    Object localObject;
    try
    {
      if (this.isMemory)
      {
        localObject = new Row(this, paramRow.oData);
        ((Row)localObject).setPos(paramRow.getPos());
      }
      else
      {
        localObject = CachedRow.newCachedRow(this, paramRow.oData);
        ((Row)localObject).setStorageSize(paramRow.getStorageSize());
        ((Row)localObject).setPos(paramRow.getPos());
        this.rowStore.restore((CachedObject)localObject);
      }
    }
    catch (IOException localIOException)
    {
      throw new HsqlException(localIOException, Trace.getMessage(98), 98);
    }
    return (Row)localObject;
  }

  public class RowStore
    implements PersistentStore
  {
    public RowStore()
    {
    }

    public CachedObject get(int paramInt)
    {
      try
      {
        return Table.this.cache.get(paramInt, this, false);
      }
      catch (HsqlException localHsqlException)
      {
      }
      return null;
    }

    public CachedObject getKeep(int paramInt)
    {
      try
      {
        return Table.this.cache.get(paramInt, this, true);
      }
      catch (HsqlException localHsqlException)
      {
      }
      return null;
    }

    public int getStorageSize(int paramInt)
    {
      try
      {
        return Table.this.cache.get(paramInt, this, false).getStorageSize();
      }
      catch (HsqlException localHsqlException)
      {
      }
      return 0;
    }

    public void add(CachedObject paramCachedObject)
      throws IOException
    {
      Table.this.cache.add(paramCachedObject);
    }

    public void restore(CachedObject paramCachedObject)
      throws IOException
    {
      Table.this.cache.restore(paramCachedObject);
    }

    public CachedObject get(RowInputInterface paramRowInputInterface)
    {
      try
      {
        if (Table.this.isText)
          return new CachedDataRow(Table.this, paramRowInputInterface);
        CachedRow localCachedRow = new CachedRow(Table.this, paramRowInputInterface);
        return localCachedRow;
      }
      catch (HsqlException localHsqlException)
      {
        return null;
      }
      catch (IOException localIOException)
      {
      }
      return null;
    }

    public CachedObject getNewInstance(int paramInt)
    {
      return null;
    }

    public void remove(int paramInt)
    {
      try
      {
        Table.this.cache.remove(paramInt, this);
      }
      catch (IOException localIOException)
      {
      }
    }

    public void removePersistence(int paramInt)
    {
      try
      {
        Table.this.cache.removePersistence(paramInt, this);
      }
      catch (IOException localIOException)
      {
      }
    }

    public void release(int paramInt)
    {
      Table.this.cache.release(paramInt);
    }

    public void commit(CachedObject paramCachedObject)
    {
      try
      {
        if (Table.this.isText)
          Table.this.cache.saveRow(paramCachedObject);
      }
      catch (IOException localIOException)
      {
      }
    }
  }
}

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     org.hsqldb.Table
 * JD-Core Version:    0.6.0
 */