package org.hsqldb.persist;

import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import org.hsqldb.DatabaseManager;
import org.hsqldb.HsqlDateTime;
import org.hsqldb.HsqlException;
import org.hsqldb.Trace;
import org.hsqldb.lib.FileUtil;
import org.hsqldb.lib.HsqlTimer;
import org.hsqldb.lib.java.JavaSystem;

public class LockFile
{
  protected File f;
  private String cpath = null;
  protected RandomAccessFile raf;
  public static final long HEARTBEAT_INTERVAL = 10000L;
  public static final byte[] MAGIC = "HSQLLOCK".getBytes();
  protected boolean locked;
  protected static final HsqlTimer timer = DatabaseManager.getTimer();
  private Object timerTask;

  private void checkHeartbeat()
    throws Exception
  {
    String str1 = "checkHeartbeat(): ";
    String str2 = "lock file [" + this.cpath + "]";
    trace(str1 + "entered.");
    if (!this.f.exists())
    {
      trace(str1 + str2 + " does not exist. Check OK.");
      return;
    }
    if (this.f.length() != 16L)
    {
      trace(str1 + str2 + " length != 16; Check OK.");
      return;
    }
    long l;
    try
    {
      l = System.currentTimeMillis() - readHeartbeat();
    }
    catch (Exception localException)
    {
      throw new Exception(Trace.getMessage(143, true, new Object[] { localException.toString(), this.cpath }));
    }
    trace(str1 + str2 + " last heartbeat " + l + " ms ago.");
    if (Math.abs(l) < 10000L)
      throw new Exception(Trace.getMessage(144, true, new Object[] { str1, str2 }));
  }

  private void closeRAF()
    throws Exception
  {
    String str = "closeRAF(): ";
    trace(str + "entered.");
    if (this.raf == null)
    {
      trace(str + "raf was null upon entry. Exiting immediately.");
    }
    else
    {
      trace(str + "closing " + this.raf);
      this.raf.close();
      trace(str + this.raf + " closed successfully. Setting raf null");
      this.raf = null;
    }
  }

  private void setFile(File paramFile)
    throws Exception
  {
    if (isLocked())
      try
      {
        tryRelease();
      }
      catch (Exception localException)
      {
        trace(localException);
      }
    this.f = FileUtil.canonicalFile(paramFile);
    this.cpath = this.f.getPath();
    this.raf = null;
    this.locked = false;
  }

  protected boolean lockImpl()
    throws Exception
  {
    String str = "lockImpl(): ";
    trace(str + "entered.");
    FileUtil.deleteOnExit(this.f);
    return true;
  }

  private void openRAF()
    throws Exception
  {
    trace("openRAF(): entered.");
    this.raf = new RandomAccessFile(this.f, "rw");
    trace("openRAF(): got new 'rw' mode " + this.raf);
  }

  private long readHeartbeat()
    throws Exception
  {
    long l = -9223372036854775808L;
    String str1 = "readHeartbeat(): ";
    String str2 = "lock file [" + this.cpath + "]";
    trace(str1 + "entered.");
    if (!this.f.exists())
    {
      trace(str1 + str2 + " does not exist. Return  '" + l + "'");
      return l;
    }
    DataInputStream localDataInputStream = new DataInputStream(new FileInputStream(this.f));
    trace(str1 + " got new " + localDataInputStream);
    for (int i = 0; i < MAGIC.length; i++)
    {
      if (MAGIC[i] == localDataInputStream.readByte())
        continue;
      trace(str1 + str2 + " is not lock file. Return '" + l + "'");
      return l;
    }
    l = localDataInputStream.readLong();
    trace(str1 + " read:  [" + HsqlDateTime.getTimestampString(l) + "]");
    localDataInputStream.close();
    trace(str1 + " closed " + localDataInputStream);
    return l;
  }

  protected boolean releaseImpl()
    throws Exception
  {
    trace("releaseImpl(): no action: returning true");
    return true;
  }

  private void startHeartbeat()
  {
    trace("startHeartbeat(): entered.");
    if ((this.timerTask == null) || (HsqlTimer.isCancelled(this.timerTask)))
    {
      HeartbeatRunner localHeartbeatRunner = new HeartbeatRunner();
      this.timerTask = timer.schedulePeriodicallyAfter(0L, 10000L, localHeartbeatRunner, true);
      trace("startHeartbeat(): heartbeat task scheduled.");
    }
    trace("startHeartbeat(): exited.");
  }

  private void stopHeartbeat()
  {
    String str = "stopHeartbeat(): ";
    trace(str + "entered");
    if ((this.timerTask != null) && (!HsqlTimer.isCancelled(this.timerTask)))
    {
      HsqlTimer.cancel(this.timerTask);
      this.timerTask = null;
    }
    trace(str + "exited");
  }

  private void writeMagic()
    throws Exception
  {
    String str1 = "writeMagic(): ";
    String str2 = "lock file [" + this.cpath + "]";
    trace(str1 + "entered.");
    trace(str1 + "raf.seek(0)");
    this.raf.seek(0L);
    trace(str1 + "raf.write(byte[])");
    this.raf.write(MAGIC);
    trace(str1 + "wrote [\"HSQLLOCK\".getBytes()] to " + str2);
  }

  private void writeHeartbeat()
    throws Exception
  {
    String str1 = "writeHeartbeat(): ";
    String str2 = "lock file [" + this.cpath + "]";
    trace(str1 + "entered.");
    long l = System.currentTimeMillis();
    trace(str1 + "raf.seek(" + MAGIC.length + ")");
    this.raf.seek(MAGIC.length);
    trace(str1 + "raf.writeLong(" + l + ")");
    this.raf.writeLong(l);
    trace(str1 + "wrote [" + l + "] to " + str2);
  }

  public static LockFile newLockFile(String paramString)
    throws Exception
  {
    Class localClass = null;
    LockFile localLockFile;
    try
    {
      Class.forName("java.nio.channels.FileLock");
      localClass = Class.forName("org.hsqldb.persist.NIOLockFile");
      localLockFile = (LockFile)localClass.newInstance();
    }
    catch (Exception localException)
    {
      localLockFile = new LockFile();
    }
    File localFile = new File(paramString);
    FileUtil.makeParentDirectories(localFile);
    localLockFile.setFile(localFile);
    return localLockFile;
  }

  public static LockFile newLockFileLock(String paramString)
    throws HsqlException
  {
    LockFile localLockFile = null;
    try
    {
      localLockFile = newLockFile(paramString + ".lck");
    }
    catch (Exception localException1)
    {
      throw Trace.error(29, localException1.toString());
    }
    boolean bool = false;
    String str = "";
    try
    {
      bool = localLockFile.tryLock();
    }
    catch (Exception localException2)
    {
      str = localException2.toString();
    }
    if (!bool)
      throw Trace.error(1, localLockFile + ": " + str);
    return localLockFile;
  }

  public boolean equals(Object paramObject)
  {
    if (this == paramObject)
      return true;
    if ((paramObject instanceof LockFile))
    {
      LockFile localLockFile = (LockFile)paramObject;
      return this.f == null ? false : localLockFile.f == null ? true : this.f.equals(localLockFile.f);
    }
    return false;
  }

  public String getCanonicalPath()
  {
    return this.cpath;
  }

  public int hashCode()
  {
    return this.f == null ? 0 : this.f.hashCode();
  }

  public boolean isLocked()
  {
    return this.locked;
  }

  public static boolean isLocked(String paramString)
  {
    FileInputStream localFileInputStream = null;
    try
    {
      LockFile localLockFile = newLockFile(paramString);
      localLockFile.checkHeartbeat();
      if ((localLockFile.f.exists()) && (localLockFile.f.isFile()))
      {
        localFileInputStream = new FileInputStream(localLockFile.f);
        localFileInputStream.read();
      }
      i = 0;
    }
    catch (Exception localException)
    {
    }
    finally
    {
      int i;
      if (localFileInputStream != null)
        try
        {
          localFileInputStream.close();
        }
        catch (IOException localIOException)
        {
        }
    }
    return true;
  }

  public boolean isValid()
  {
    return (isLocked()) && (this.f != null) && (this.f.exists()) && (this.raf != null);
  }

  public String toString()
  {
    return super.toString() + "[file =" + this.cpath + ", exists=" + this.f.exists() + ", locked=" + isLocked() + ", valid=" + isValid() + ", " + toStringImpl() + "]";
  }

  protected String toStringImpl()
  {
    return "";
  }

  public boolean tryLock()
    throws Exception
  {
    String str = "tryLock(): ";
    trace(str + "entered.");
    if (this.locked)
    {
      trace(str + " lock already held. Returning true immediately.");
      return true;
    }
    checkHeartbeat();
    openRAF();
    this.locked = lockImpl();
    if (this.locked)
    {
      writeMagic();
      startHeartbeat();
      try
      {
        JavaSystem.runFinalizers();
        trace(str + "success for System.runFinalizersOnExit(true)");
      }
      catch (Exception localException1)
      {
        trace(str + localException1.toString());
      }
    }
    else
    {
      try
      {
        releaseImpl();
        closeRAF();
      }
      catch (Exception localException2)
      {
        trace(str + localException2.toString());
      }
    }
    trace(str + "ran to completion.  Returning " + this.locked);
    return this.locked;
  }

  public boolean tryRelease()
    throws Exception
  {
    String str1 = "tryRelease(): ";
    trace(str1 + "entered.");
    boolean bool = !this.locked;
    if (bool)
    {
      trace(str1 + "No lock held. Returning true immediately");
      return true;
    }
    try
    {
      bool = releaseImpl();
    }
    catch (Exception localException1)
    {
      trace(str1 + localException1);
    }
    if (!bool)
    {
      trace(str1 + "releaseImpl() failed. Returning false immediately.");
      return false;
    }
    trace(str1 + "releaseImpl() succeeded.");
    stopHeartbeat();
    closeRAF();
    trace(str1 + "Starting Thread.sleep(100).");
    try
    {
      Thread.sleep(100L);
    }
    catch (Exception localException2)
    {
      trace(str1 + localException2.toString());
    }
    trace(str1 + "Finished Thread.sleep(100).");
    String str2 = "[" + this.cpath + "]";
    if (this.f.exists())
    {
      trace(str1 + str2 + " exists.");
      bool = this.f.delete();
      trace(str1 + str2 + (bool ? "" : "not") + " deleted.");
      if (this.f.exists())
        trace(str1 + " WARNING!: " + str2 + "still exists.");
    }
    this.locked = (!bool);
    trace(str1 + "ran to completion.  Returning " + bool);
    return bool;
  }

  protected void trace(Object paramObject)
  {
    if (Trace.TRACE)
      Trace.printSystemOut("[" + super.toString() + "]: " + paramObject);
  }

  protected void finalize()
    throws Throwable
  {
    trace("finalize(): calling tryRelease()");
    tryRelease();
  }

  protected class HeartbeatRunner
    implements Runnable
  {
    protected HeartbeatRunner()
    {
    }

    public void run()
    {
      try
      {
        LockFile.this.trace("HeartbeatRunner.run(): writeHeartbeat()");
        LockFile.this.writeHeartbeat();
      }
      catch (Throwable localThrowable)
      {
        LockFile.this.trace("HeartbeatRunner.run(): caught Throwable: " + localThrowable);
      }
    }
  }
}

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     org.hsqldb.persist.LockFile
 * JD-Core Version:    0.6.0
 */