/*
 * Envers. http://www.jboss.org/envers
 *
 * Copyright 2008  Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT A WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License, v.2.1 along with this distribution; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * Red Hat Author(s): Adam Warski
 */
package org.jboss.envers.configuration;

import org.hibernate.cfg.Configuration;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.MappingException;
import org.jboss.envers.metadata.VersionsMetadataGenerator;
import org.jboss.envers.metadata.RevisionsInfoMetadataGenerator;
import org.jboss.envers.metadata.AnnotationsMetadataReader;
import org.jboss.envers.metadata.data.PersistentClassVersioningData;
import org.jboss.envers.synchronization.VersionsSyncManager;
import org.jboss.envers.mapper.ExtendedPropertyMapper;
import org.jboss.envers.mapper.id.IdMapper;
import org.jboss.envers.tools.graph.GraphTopologicalSort;
import org.jboss.envers.reflection.YReflectionManager;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.io.XMLWriter;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.DOMWriter;

import java.util.*;
import java.io.ByteArrayOutputStream;
import java.io.Writer;
import java.io.PrintWriter;
import java.io.IOException;

/**
 * @author Adam Warski (adam at warski dot org)
 */
public class VersionsConfiguration {
    private VersionsEntitiesConfiguration verEntCfg;
    private VersionsSyncManager versionsSyncManager;
    private EntitiesConfigurations entitiesConfigurations;

    // Should a revision be generated when a not-owned relation field changes
    private boolean generateRevisionsForCollections;

    // Should a warning, instead of an error and an exception, be logged, when an unsupported type is versioned
    private boolean warnOnUnsupportedTypes;

    //

    public VersionsEntitiesConfiguration getEntitiesCfg() {
        return verEntCfg;
    }

    public VersionsSyncManager getSyncManager() {
        return versionsSyncManager;
    }

    // todo
    private void writeDocument(Document e) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        Writer w = new PrintWriter(baos);

        try {
            XMLWriter xw = new XMLWriter(w, new OutputFormat(" ", true));
            xw.write(e);
            w.flush();
        } catch (IOException e1) {
            e1.printStackTrace();
        }

        System.out.println("-----------");
        System.out.println(baos.toString());
        System.out.println("-----------");
    }

    private YReflectionManager getReflectionManager(Configuration cfg) {
        try {
            return new YReflectionManager(cfg);
        } catch (Exception e) {
            throw new MappingException(e);
        }
    }

    private boolean isVersioned(PersistentClassVersioningData versioningData) {
        if (versioningData == null) { return false; }
        if (versioningData.propertyStoreInfo.propertyStores.size() > 0) { return true; }
        if (versioningData.propertyStoreInfo.defaultStore != null) { return true; }
        return false;
    }

    private void configureFromProperties(Properties properties) {
        String generateRevisionsForCollectionsStr = properties.getProperty("org.jboss.envers.revisionOnCollectionChange",
                "true");
        generateRevisionsForCollections = Boolean.parseBoolean(generateRevisionsForCollectionsStr);

        String warnOnUnsupportedTypesStr = properties.getProperty("org.jboss.envers.warnOnUnsupportedTypes",
                "false");
        warnOnUnsupportedTypes = Boolean.parseBoolean(warnOnUnsupportedTypesStr);        
    }

    @SuppressWarnings({"unchecked"})
    public void configure(Configuration cfg) {
        configureFromProperties(cfg.getProperties());

        YReflectionManager reflectionManager = getReflectionManager(cfg);

        verEntCfg = VersionsEntitiesConfiguration.get(cfg, reflectionManager);

        versionsSyncManager = new VersionsSyncManager(verEntCfg);

        VersionsMetadataGenerator versionsMetaGen = new VersionsMetadataGenerator(this, verEntCfg);
        DOMWriter writer = new DOMWriter();

        // Sorting the persistent class topologically - superclass always before subclass
        Iterator<PersistentClass> classes = GraphTopologicalSort.sort(new PersistentClassGraphDefiner(cfg)).iterator();

        Map<PersistentClass, PersistentClassVersioningData> pcDatas =
                new HashMap<PersistentClass, PersistentClassVersioningData>();
        Map<PersistentClass, Document> documents = new HashMap<PersistentClass, Document>();

        // First pass
        AnnotationsMetadataReader annotationsMetadataReader = new AnnotationsMetadataReader();
        while (classes.hasNext()) {
            PersistentClass pc = classes.next();
            // Collecting information from annotations on the persistent class pc
            PersistentClassVersioningData versioningData =
                    annotationsMetadataReader.getVersioningData(pc, reflectionManager);

            if (isVersioned(versioningData)) {
                pcDatas.put(pc, versioningData);

                if (versioningData.versionsTableName != null) {
                    verEntCfg.addCustomVersionsTableName(pc.getEntityName(), versioningData.versionsTableName);
                }

                Document genData = versionsMetaGen.generateFirstPass(pc, versioningData);
                documents.put(pc, genData);
            }
        }

        // Second pass
        for (PersistentClass pc : pcDatas.keySet()) {
            Document document = documents.get(pc);

            versionsMetaGen.generateSecondPass(pc, pcDatas.get(pc), document);

            //TODO
            //writeDocument(document);

            try {
                cfg.addDocument(writer.write(document));
            } catch (DocumentException e) {
                throw new MappingException(e);
            }
        }

        // Getting the entities configurations
        entitiesConfigurations = new EntitiesConfigurations(versionsMetaGen.getEntitiesConfigurations());

        // Checking if custom revision entity isn't versioned
        if (entitiesConfigurations.get(verEntCfg.getRevisionsInfoEntityName()) != null) {
            throw new MappingException("An entity annotated with @RevisionEntity cannot be versioned!");
        }

        // Only if there are any versioned classes
        if (pcDatas.size() > 0) {
            try {
                if (!verEntCfg.hasCustomRevisionInfoEntity()) {
                    RevisionsInfoMetadataGenerator revisionsMetaGenInfo = new RevisionsInfoMetadataGenerator(verEntCfg);
                    Document helperEntity = revisionsMetaGenInfo.generate();
                    //writeDocument(helperEntity);
                    cfg.addDocument(writer.write(helperEntity));
                }
            } catch (DocumentException e) {
                throw new MappingException(e);
            }
        }
    }

    public boolean isVersioned(String entityName) {
        return entitiesConfigurations.get(entityName) != null;
    }

    public IdMapper getIdMapper(String entityName) {
        return entitiesConfigurations.get(entityName).getIdMappingData().getIdMapper();
    }

    public ExtendedPropertyMapper getPropertyMapper(String entityName) {
        return entitiesConfigurations.get(entityName).getPropertyMapper();
    }

    public EntityConfiguration getEntityConfiguration(String entityName) {
        return entitiesConfigurations.get(entityName);
    }

    public String getEntityNameForVersionsEntityName(String versionsEntityName) {
        return entitiesConfigurations.getEntityNameForVersionsEntityName(versionsEntityName);
    }

    public boolean isGenerateRevisionsForCollections() {
        return generateRevisionsForCollections;
    }

    public boolean isWarnOnUnsupportedTypes() {
        return warnOnUnsupportedTypes;
    }

    public RelationDescription getRelationDescription(String entityName, String propertyName) {
        return entitiesConfigurations.getRelationDescription(entityName, propertyName);
    }

    //

    private static Map<Configuration, VersionsConfiguration> cfgs
            = new WeakHashMap<Configuration, VersionsConfiguration>();

    public synchronized static VersionsConfiguration getFor(Configuration cfg) {
        VersionsConfiguration verCfg = cfgs.get(cfg);

        if (verCfg == null) {
            verCfg = new VersionsConfiguration();
            cfgs.put(cfg, verCfg);

            verCfg.configure(cfg);
            cfg.buildMappings();
        }

        return verCfg;
    }
}
