/*
 * Envers. http://www.jboss.org/envers
 *
 * Copyright 2008  Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT A WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License, v.2.1 along with this distribution; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * Red Hat Author(s): Adam Warski
 */
package org.jboss.envers.configuration;

import org.hibernate.cfg.Configuration;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.MappingException;
import org.jboss.envers.RevisionEntity;
import org.jboss.envers.RevisionNumber;
import org.jboss.envers.RevisionTimestamp;
import org.jboss.envers.reflection.YReflectionManager;
import org.jboss.envers.reflection.YClass;
import org.jboss.envers.reflection.YProperty;
import org.jboss.envers.tools.MutableBoolean;

import java.util.*;

/**
 * Configuration of versions entities - names of fields and entities created to store versioning information, revision
 * info entity configuration etc.
 * @author Adam Warski (adam at warski dot org)
 */
public class VersionsEntitiesConfiguration {
    private String versionsTablePrefix;
    private String versionsTableSuffix;

    private String originalIdPropName;

    private String revisionPropName;
    private String revisionPropType;
    private String revisionPropPath;

    private String revisionTypePropName;
    private String revisionTypePropType;

    private String revisionsInfoEntityName;
    private String revisionsInfoIdName;
    private String revisionsInfoTimestampName;
    private String revisionsInfoTimestampType;
    private boolean customRevisionsInfoEntity;

    private RevisionInfoGenerator revisionInfoGenerator;

    private Map<String, String> customVersionsTablesNames;

    public static VersionsEntitiesConfiguration get(Configuration cfg, YReflectionManager reflectionManager) {
        Properties properties = cfg.getProperties();

        VersionsEntitiesConfiguration verEntCfg = new VersionsEntitiesConfiguration();

        verEntCfg.versionsTablePrefix = properties.getProperty("org.jboss.envers.versionsTablePrefix", "");
        verEntCfg.versionsTableSuffix = properties.getProperty("org.jboss.envers.versionsTableSuffix", "_versions");

        verEntCfg.originalIdPropName = "originalId";

        verEntCfg.revisionPropName = properties.getProperty("org.jboss.envers.revisionFieldName", "_revision");
        verEntCfg.revisionPropType = "integer";

        verEntCfg.revisionTypePropName = properties.getProperty("org.jboss.envers.revisionTypeFieldName", "_revision_type");
        verEntCfg.revisionTypePropType = "byte";

        verEntCfg.revisionsInfoEntityName = "_revisions_info";
        verEntCfg.revisionsInfoIdName = "revision_id";
        verEntCfg.revisionsInfoTimestampName = "revision_timestamp";
        verEntCfg.revisionsInfoTimestampType = "long";

        verEntCfg.customVersionsTablesNames = new HashMap<String, String>();

        verEntCfg.revisionPropPath = verEntCfg.originalIdPropName + "." + verEntCfg.revisionPropName;

        verEntCfg.customRevisionsInfoEntity = verEntCfg.configureRevisionsEntity(cfg, reflectionManager);

        return verEntCfg;
    }

    public String getVersionsTablePrefix() {
        return versionsTablePrefix;
    }

    public String getVersionsTableSuffix() {
        return versionsTableSuffix;
    }

    public String getOriginalIdPropName() {
        return originalIdPropName;
    }

    public String getRevisionPropName() {
        return revisionPropName;
    }

    public String getRevisionPropPath() {
        return revisionPropPath;
    }

    public String getRevisionPropType() {
        return revisionPropType;
    }

    public String getRevisionTypePropName() {
        return revisionTypePropName;
    }

    public String getRevisionTypePropType() {
        return revisionTypePropType;
    }

    public String getRevisionsInfoEntityName() {
        return revisionsInfoEntityName;
    }

    public String getRevisionsInfoIdName() {
        return revisionsInfoIdName;
    }

    public String getRevisionsInfoTimestampName() {
        return revisionsInfoTimestampName;
    }

    public String getRevisionsInfoTimestampType() {
        return revisionsInfoTimestampType;
    }

    public RevisionInfoGenerator getRevisionInfoGenerator() {
        return revisionInfoGenerator;
    }

    //

    public void addCustomVersionsTableName(String entityName, String tableName) {
        customVersionsTablesNames.put(entityName, tableName);
    }

    //

    public String getVersionsEntityName(String entityName) {
        return getVersionsTablePrefix() + entityName + getVersionsTableSuffix();
    }

    public boolean isVersionsEntityName(String entityName) {
        if (entityName == null) {
            return false;
        }

        return entityName.endsWith(getVersionsTableSuffix()) &&
                entityName.startsWith(getVersionsTablePrefix());
    }

    public String getVersionsTableName(String entityName, String tableName) {
        String customHistoryTableName = customVersionsTablesNames.get(entityName);
        if (customHistoryTableName == null) {
            return getVersionsTablePrefix() + tableName + getVersionsTableSuffix();
        }

        return customHistoryTableName;
    }

    public boolean isRevisionInfoEntityName(String entityName) {
        return revisionsInfoEntityName.equals(entityName);
    }

    public boolean hasCustomRevisionInfoEntity() {
        return customRevisionsInfoEntity;
    }

    //

    private void searchForRevisionInfoCfg(YClass clazz, YReflectionManager reflectionManager,
                                          MutableBoolean revisionNumberFound, MutableBoolean revisionTimestampFound) {
        YClass superclazz = clazz.getSuperclass();
        if (!"java.lang.Object".equals(superclazz.getName())) {
            searchForRevisionInfoCfg(superclazz, reflectionManager, revisionNumberFound, revisionTimestampFound);
        }

        for (YProperty property : clazz.getDeclaredProperties("field")) {
            RevisionNumber revisionNumber = property.getAnnotation(RevisionNumber.class);
            RevisionTimestamp revisionTimestamp = property.getAnnotation(RevisionTimestamp.class);

            if (revisionNumber != null) {
                if (revisionNumberFound.isSet()) {
                    throw new MappingException("Only one property may be annotated with @RevisionNumber!");
                }

                YClass revisionNumberClass = property.getType();
                if (reflectionManager.equals(revisionNumberClass, Integer.class) ||
                        reflectionManager.equals(revisionNumberClass, Integer.TYPE)) {
                    revisionsInfoIdName = property.getName();
                    revisionNumberFound.set();
                } else if (reflectionManager.equals(revisionNumberClass, Long.class) ||
                        reflectionManager.equals(revisionNumberClass, Long.TYPE)) {
                    revisionsInfoIdName = property.getName();
                    revisionNumberFound.set();

                    // The default is integer
                    revisionPropType = "long";
                } else {
                    throw new MappingException("The field annotated with @RevisionNumber must be of type " +
                            "int, Integer, long or Long");
                }
            }

            if (revisionTimestamp != null) {
                if (revisionTimestampFound.isSet()) {
                    throw new MappingException("Only one property may be annotated with @RevisionTimestamp!");
                }

                YClass revisionTimestampClass = property.getType();
                if (reflectionManager.equals(revisionTimestampClass, Long.class) ||
                        reflectionManager.equals(revisionTimestampClass, Long.TYPE)) {
                    revisionsInfoTimestampName = property.getName();
                    revisionTimestampFound.set();
                } else {
                    throw new MappingException("The field annotated with @RevisionTimestamp must be of type " +
                            "long or Long");
                }
            }
        }
    }

    @SuppressWarnings({"unchecked"})
    private boolean configureRevisionsEntity(Configuration cfg, YReflectionManager reflectionManager) {
        Iterator<PersistentClass> classes = (Iterator<PersistentClass>) cfg.getClassMappings();
        boolean revisionEntityFound = false;

        while (classes.hasNext()) {
            PersistentClass pc = classes.next();
            YClass clazz;
            try {
                clazz = reflectionManager.classForName(pc.getClassName(), this.getClass());
            } catch (ClassNotFoundException e) {
                throw new MappingException(e);
            }

            RevisionEntity revisionEntity = clazz.getAnnotation(RevisionEntity.class);
            if (revisionEntity != null) {
                if (revisionEntityFound) {
                    throw new MappingException("Only one entity may be annotated with @RevisionEntity!");
                }

                revisionEntityFound = true;

                MutableBoolean revisionNumberFound = new MutableBoolean();
                MutableBoolean revisionTimestampFound = new MutableBoolean();

                searchForRevisionInfoCfg(clazz, reflectionManager, revisionNumberFound, revisionTimestampFound);

                if (!revisionNumberFound.isSet()) {
                    throw new MappingException("An entity annotated with @RevisionEntity must have a field annotated " +
                            "with @RevisionNumber!");
                }

                if (!revisionTimestampFound.isSet()) {
                    throw new MappingException("An entity annotated with @RevisionEntity must have a field annotated " +
                            "with @RevisionTimestamp!");
                }

                revisionsInfoEntityName = pc.getEntityName();

                revisionInfoGenerator = new CustomRevisionInfoGenerator(this, pc.getMappedClass(),
                        revisionEntity.value());
            }
        }

        if (revisionInfoGenerator == null) {
            revisionInfoGenerator = new DefaultRevisionInfoGenerator(this);
        }

        return revisionEntityFound;
    }
}
