/*
 * Envers. http://www.jboss.org/envers
 *
 * Copyright 2008  Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT A WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License, v.2.1 along with this distribution; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * Red Hat Author(s): Adam Warski
 */
package org.jboss.envers.entity;

import org.jboss.envers.metadata.RevisionType;
import org.jboss.envers.exception.VersionsException;
import org.jboss.envers.configuration.VersionsConfiguration;
import org.jboss.envers.mapper.id.IdMapper;
import org.jboss.envers.reader.VersionsReaderImplementor;
import org.jboss.envers.reflection.ReflectionTools;

import java.util.Map;
import java.util.List;

/**
 * @author Adam Warski (adam at warski dot org)
 */
public class EntityInstantiator {
    private VersionsReaderImplementor versionsReader;
    private VersionsConfiguration verCfg;

    public EntityInstantiator(VersionsReaderImplementor versionsReader) {
        this.versionsReader = versionsReader;

        verCfg = versionsReader.getVerCfg();
    }

    /**
     * Creates an entity instance based on an entry from the versions table.
     * @param entityName Name of the entity, which instances should be read
     * @param versionsEntity An entry in the versions table, from which data should be mapped.
     * @param revision Revision at which this entity was read.
     * @return An entity instance, with versioned properties set as in the versionsEntity map, and proxies
     * created for collections.
     */
    public Object createInstanceFromVersionsEntity(String entityName, Map versionsEntity, Number revision) {
        if (versionsEntity == null) {
            return null;
        }
        
        if (RevisionType.DEL.equals(versionsEntity.get(verCfg.getEntitiesCfg().getRevisionTypePropName()))) {
            // This means that at the given revision the entity didn't yet exist.
            return null;
        }

        // The $type$ property holds the name of the (versions) entity
        String type = verCfg.getEntityNameForVersionsEntityName((String) versionsEntity.get("$type$"));

        if (type != null) {
            entityName = type;
        }

        Object ret;
        try {
            Class<?> cls = ReflectionTools.loadClass(entityName);
            ret = cls.newInstance();
        } catch (Exception e) {
            throw new VersionsException(e);
        }

        IdMapper idMapper = verCfg.getIdMapper(entityName);
        Map originalId = (Map) versionsEntity.get(verCfg.getEntitiesCfg().getOriginalIdPropName());

        Object primaryKey = idMapper.mapToIdFromMap(originalId);

        verCfg.getPropertyMapper(entityName).mapToEntityFromMap(ret, versionsEntity, primaryKey, versionsReader, revision);
        idMapper.mapToEntityFromMap(ret, originalId);

        return ret;
    }

    @SuppressWarnings({"unchecked"})
    public void addInstancesFromVersionsEntities(String entityName, List list, List<Map> versionsEntities, Number revision) {
        for (Map versionsEntity : versionsEntities) {
            list.add(createInstanceFromVersionsEntity(entityName, versionsEntity, revision));
        }
    }
}
