/*
 * Envers. http://www.jboss.org/envers
 *
 * Copyright 2008  Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT A WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License, v.2.1 along with this distribution; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * Red Hat Author(s): Adam Warski
 */
package org.jboss.envers.event;

import org.hibernate.event.*;
import org.hibernate.cfg.Configuration;
import org.hibernate.persister.entity.EntityPersister;
import org.jboss.envers.configuration.VersionsConfiguration;
import org.jboss.envers.configuration.RelationDescription;
import org.jboss.envers.synchronization.VersionsSync;
import org.jboss.envers.synchronization.work.AddWorkUnit;
import org.jboss.envers.synchronization.work.ModWorkUnit;
import org.jboss.envers.synchronization.work.DelWorkUnit;
import org.jboss.envers.synchronization.work.CollectionChangeWorkUnit;
import org.jboss.envers.tools.Tools;
import org.jboss.envers.mapper.id.IdMapper;

import java.io.Serializable;

/**
 * @author Adam Warski (adam at warski dot org)
 */
public class VersionsEventListener implements PostInsertEventListener, PostUpdateEventListener,
        PostDeleteEventListener, Initializable {
    private VersionsConfiguration verCfg;

    private void generateCollectionChangeWorkUnits(VersionsSync verSync, EntityPersister entityPersister,
                                                   String entityName, Object[] newState, Object[] oldState) {
        // Checking if this is enabled in configuration ...
        if (!verCfg.isGenerateRevisionsForCollections()) {
            return;
        }

        // Checks every property of the entity, if it is an "owned" relation to another entity.
        // If the value of that property changed, and the relation is bi-directional, a new revision
        // for the related entity is generated.
        String[] propertyNames = entityPersister.getPropertyNames();

        for (int i=0; i<propertyNames.length; i++) {
            String propertyName = propertyNames[i];
            RelationDescription relDesc = verCfg.getRelationDescription(entityName, propertyName);
            if (relDesc != null && relDesc.bidirectional && relDesc.relationType == RelationDescription.RelationType.TO_ONE) {
                // Checking for changes
                Object oldValue = oldState == null ? null : oldState[i];
                Object newValue = newState == null ? null : newState[i];

                if (!Tools.objectsEqual(oldValue, newValue)) {
                    IdMapper idMapper = verCfg.getIdMapper(relDesc.toEntityName);

                    // We have to generate changes both in the old collection (size decreses) and new collection
                    // (size increases).
                    if (newValue != null) {
                        Serializable id = (Serializable) idMapper.mapToIdFromEntity(newValue);
                        verSync.addWorkUnit(new CollectionChangeWorkUnit(relDesc.toEntityName, verCfg, id, newValue));
                    }
                    
                    if (oldValue != null) {
                        Serializable id = (Serializable) idMapper.mapToIdFromEntity(oldValue);
                        verSync.addWorkUnit(new CollectionChangeWorkUnit(relDesc.toEntityName, verCfg, id, oldValue));
                    }
                }
            }
        }
    }

    public void onPostInsert(PostInsertEvent event) {
        String entityName = event.getPersister().getEntityName();

        if (verCfg.isVersioned(entityName)) {
            VersionsSync verSync = verCfg.getSyncManager().get(event.getSession());

            verSync.addWorkUnit(new AddWorkUnit(event.getPersister().getEntityName(), verCfg, event.getId(),
                    event.getPersister(), event.getState()));

            generateCollectionChangeWorkUnits(verSync, event.getPersister(), entityName, event.getState(), null);
        }
    }

    public void onPostUpdate(PostUpdateEvent event) {
        String entityName = event.getPersister().getEntityName();

        if (verCfg.isVersioned(entityName)) {
            VersionsSync verSync = verCfg.getSyncManager().get(event.getSession());

            verSync.addWorkUnit(new ModWorkUnit(event.getPersister().getEntityName(), verCfg, event.getId(),
                    event.getPersister(), event.getState(), event.getOldState()));

            generateCollectionChangeWorkUnits(verSync, event.getPersister(), entityName, event.getState(), event.getOldState());
        }
    }

    public void onPostDelete(PostDeleteEvent event) {
        String entityName = event.getPersister().getEntityName();

        if (verCfg.isVersioned(entityName)) {
            VersionsSync verSync = verCfg.getSyncManager().get(event.getSession());

            verSync.addWorkUnit(new DelWorkUnit(event.getPersister().getEntityName(), verCfg, event.getId()));

            generateCollectionChangeWorkUnits(verSync, event.getPersister(), entityName, null, event.getDeletedState());
        }
    }

    @SuppressWarnings({"unchecked"})
    public void initialize(Configuration cfg) {
        verCfg = VersionsConfiguration.getFor(cfg);
    }

    public VersionsConfiguration getVerCfg() {
        return verCfg;
    }
}
