/*
 * Envers. http://www.jboss.org/envers
 *
 * Copyright 2008  Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT A WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License, v.2.1 along with this distribution; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * Red Hat Author(s): Adam Warski
 */
package org.jboss.envers.metadata;

import org.jboss.envers.reflection.YClass;
import org.jboss.envers.reflection.YProperty;
import org.jboss.envers.reflection.YReflectionManager;
import org.jboss.envers.metadata.data.PersistentClassVersioningData;
import org.jboss.envers.Versioned;
import org.jboss.envers.VersionsTable;
import org.jboss.envers.SecondaryVersionsTable;
import org.jboss.envers.SecondaryVersionsTables;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.MappingException;

/**
 * A helper class to read versioning meta-data from annotations on persistent classes.
 * @author Adam Warski (adam at warski dot org)
 */
public class AnnotationsMetadataReader {
    private void addPropertiesFromClass(YClass clazz, PersistentClassVersioningData versioningData)  {
        YClass superclazz = clazz.getSuperclass();
        if (!"java.lang.Object".equals(superclazz.getName())) {
            addPropertiesFromClass(superclazz, versioningData);
        }

        for (YProperty property : clazz.getDeclaredProperties("field")) {
            Versioned ver = property.getAnnotation(Versioned.class);
            if (ver != null) {
                versioningData.propertyStoreInfo.propertyStores.put(property.getName(), ver.modStore());
            }
        }

        for (YProperty property : clazz.getDeclaredProperties("property")) {
            Versioned ver = property.getAnnotation(Versioned.class);
            if (ver != null) {
                versioningData.propertyStoreInfo.propertyStores.put(property.getName(), ver.modStore());
            }
        }
    }

    public PersistentClassVersioningData getVersioningData(PersistentClass pc, YReflectionManager reflectionManager) {
        if (pc.getClassName() == null) {
            return null;
        }

        PersistentClassVersioningData versioningData = new PersistentClassVersioningData();

        try {
            YClass clazz = reflectionManager.classForName(pc.getClassName(), this.getClass());
            Versioned defaultVersioned = clazz.getAnnotation(Versioned.class);

            if (defaultVersioned != null) {
                versioningData.propertyStoreInfo.defaultStore = defaultVersioned.modStore();
            }

            addPropertiesFromClass(clazz, versioningData);

            VersionsTable versionsTable = clazz.getAnnotation(VersionsTable.class);
            if (versionsTable != null) {
                versioningData.versionsTableName = versionsTable.value();
                versioningData.schema = versionsTable.schema();
                versioningData.catalog = versionsTable.catalog();
            }

            // Getting information on secondary tables
            SecondaryVersionsTable secondaryVersionsTable1 = clazz.getAnnotation(SecondaryVersionsTable.class);
            if (secondaryVersionsTable1 != null) {
                versioningData.secondaryTableDictionary.put(secondaryVersionsTable1.secondaryTableName(),
                        secondaryVersionsTable1.secondaryVersionsTableName());
            }

            SecondaryVersionsTables secondaryVersionsTables = clazz.getAnnotation(SecondaryVersionsTables.class);
            if (secondaryVersionsTables != null) {
                for (SecondaryVersionsTable secondaryVersionsTable2 : secondaryVersionsTables.value()) {
                    versioningData.secondaryTableDictionary.put(secondaryVersionsTable2.secondaryTableName(),
                            secondaryVersionsTable2.secondaryVersionsTableName());
                }
            }
        } catch (ClassNotFoundException e) {
            throw new MappingException(e);
        }

        return versioningData;
    }
}
