/*
 * Envers. http://www.jboss.org/envers
 *
 * Copyright 2008  Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT A WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License, v.2.1 along with this distribution; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * Red Hat Author(s): Adam Warski
 */
package org.jboss.envers.metadata;

import org.hibernate.mapping.*;
import org.hibernate.type.*;
import org.hibernate.util.StringHelper;
import org.hibernate.MappingException;
import org.dom4j.Element;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Attribute;
import org.dom4j.tree.DefaultElement;
import org.jboss.envers.configuration.VersionsEntitiesConfiguration;
import org.jboss.envers.configuration.EntityConfiguration;
import org.jboss.envers.configuration.VersionsConfiguration;
import org.jboss.envers.mapper.*;
import org.jboss.envers.mapper.id.*;
import org.jboss.envers.mapper.id.relation.ToOneIdMapper;
import org.jboss.envers.mapper.id.relation.OneToOneIdMapper;
import org.jboss.envers.mapper.id.relation.OneToManyIdMapper;
import org.jboss.envers.tools.StringTools;
import org.jboss.envers.tools.Tools;
import org.jboss.envers.tools.HibernateVersion;
import org.jboss.envers.ModificationStore;
import org.jboss.envers.log.YLog;
import org.jboss.envers.log.YLogManager;
import org.jboss.envers.metadata.data.IdMappingData;
import org.jboss.envers.metadata.data.PersistentClassVersioningData;
import org.jboss.envers.metadata.data.PropertyStoreInfo;
import org.jboss.envers.exception.VersionsException;

import java.util.Iterator;
import java.util.Map;
import java.util.Collections;
import java.util.HashMap;

/**
 * @author Adam Warski (adam at warski dot org)
 */
public class VersionsMetadataGenerator {
    private final static Map<String, ModificationStore> EMPTY_STORE = Collections.emptyMap();

    private VersionsConfiguration verCfg;
    private VersionsEntitiesConfiguration verEntCfg;
    private Map<String, EntityConfiguration> entitiesConfigurations;

    // Map entity name -> (join descriptor -> element describing the "versioned" join)
    private Map<String, Map<Join, Element>> entitiesJoins;

    private YLog log = YLogManager.getLogManager().getLog(VersionsMetadataGenerator.class);

    public VersionsMetadataGenerator(VersionsConfiguration verCfg, VersionsEntitiesConfiguration verEntCfg) {
        this.verCfg = verCfg;
        this.verEntCfg = verEntCfg;

        entitiesConfigurations = new HashMap<String, EntityConfiguration>();
        entitiesJoins = new HashMap<String, Map<Join, Element>>();
    }

    private void addComponentClassName(Element any_mapping, Component comp) {
        if (StringHelper.isNotEmpty(comp.getComponentClassName())) {
            any_mapping.addAttribute("class", comp.getComponentClassName());
        }
    }

    private void addColumns(Element any_mapping, Iterator<Column> columns) {
        while (columns.hasNext()) {
            Column column = columns.next();
            MetadataTools.addColumn(any_mapping, column.getName(), column.getLength());
        }
    }

    @SuppressWarnings({"unchecked"})
    private void addSimpleProperty(Element parent, Property property, SimpleMapperBuilder mapper,
                                   ModificationStore store, boolean key) {
        Element prop_mapping = MetadataTools.addProperty(parent, property.getName(),
                property.getType().getName(), key);
        addColumns(prop_mapping, (Iterator<Column>) property.getColumnIterator());

        // A null mapper means that we only want to add xml mappings (while building the id mapping)
        if (mapper != null) {
            mapper.add(property.getName(), store);
        }
    }

    @SuppressWarnings({"unchecked"})
    private void addEnumProperty(Element parent, Property property, SimpleMapperBuilder mapper,
                                 ModificationStore store) {
        Element prop_mapping = parent.addElement("property");
        prop_mapping.addAttribute("name", property.getName());

        CustomType propertyType = (CustomType) property.getType();

        Element type_mapping = prop_mapping.addElement("type");
        type_mapping.addAttribute("name", propertyType.getName());

        Element type_param1 = type_mapping.addElement("param");
        type_param1.addAttribute("name", "enumClass");
        type_param1.setText(propertyType.getReturnedClass().getName());

        Element type_param2 = type_mapping.addElement("param");
        type_param2.addAttribute("name", "type");
        type_param2.setText(Integer.toString(propertyType.sqlTypes(null)[0]));

        addColumns(prop_mapping, (Iterator<Column>) property.getColumnIterator());

        mapper.add(property.getName(), store);
    }

    @SuppressWarnings({"unchecked"})
    private void addComponent(Element parent, Property property, CompositeMapperBuilder mapper, ModificationStore store,
                              String entityName, boolean firstPass) {
        Element component_mapping = null;
        Component prop_component = (Component) property.getValue();

        if (!firstPass) {
            // The required element already exists.
            Iterator<Element> iter = parent.elementIterator("component");
            while (iter.hasNext()) {
                Element child = iter.next();
                if (child.attribute("name").getText().equals(property.getName())) {
                    component_mapping = child;
                    break;
                }
            }

            if (component_mapping == null) {
                throw new VersionsException("Element for component not found during second pass!");
            }
        } else {
            component_mapping = parent.addElement("component");
            component_mapping.addAttribute("name", property.getName());

            addComponentClassName(component_mapping, prop_component);
        }

        addProperties(component_mapping, (Iterator<Property>) prop_component.getPropertyIterator(),
                mapper.addComposite(property.getName()), new PropertyStoreInfo(store, EMPTY_STORE), entityName,
                firstPass);
    }

    @SuppressWarnings({"unchecked"})
    private void changeNamesInColumnElement(Element element, Iterator<Column> columnIterator) {
        Iterator<Element> properties = element.elementIterator();
        while (properties.hasNext()) {
            Element property = properties.next();

            if ("column".equals(property.getName())) {
                Attribute nameAttr = property.attribute("name");
                if (nameAttr != null) {
                    nameAttr.setText(columnIterator.next().getName());
                }
            }
        }
    }

    @SuppressWarnings({"unchecked"})
    private void prefixNamesInPropertyElement(Element element, String prefix, Iterator<Column> columnIterator) {
        Iterator<Element> properties = element.elementIterator();
        while (properties.hasNext()) {
            Element property = properties.next();

            if ("property".equals(property.getName())) {
                Attribute nameAttr = property.attribute("name");
                if (nameAttr != null) {
                    nameAttr.setText(prefix + nameAttr.getText());
                }

                changeNamesInColumnElement(property, columnIterator);
            }
        }
    }

    @SuppressWarnings({"unchecked"})
    private void addToOne(Element parent, Property property, CompositeMapperBuilder mapper, String entityName) {
        String referencedEntityName = ((ToOne) property.getValue()).getReferencedEntityName();

        EntityConfiguration configuration = entitiesConfigurations.get(referencedEntityName);
        if (configuration == null) {
            throw new MappingException("A versioned relation to a non-versioned entity " + referencedEntityName + "!");
        }

        IdMappingData idMapping = configuration.getIdMappingData();

        String propertyName = property.getName();
        String lastPropertyPrefix = propertyName + "_";

        // Generating the id mapper for the relation
        IdMapper relMapper = idMapping.getIdMapper().prefixMappedProperties(lastPropertyPrefix);

        // Storing information about this relation
        entitiesConfigurations.get(entityName).addToOneRelation(propertyName, referencedEntityName, relMapper);

        // Adding an element to the mapping corresponding to the references entity id's
        Element properties = (Element) idMapping.getXmlRelationMapping().clone();
        properties.addAttribute("name", propertyName);

        prefixNamesInPropertyElement(properties, lastPropertyPrefix, property.getValue().getColumnIterator());
        parent.add(properties);

        // Adding mapper for the id
        mapper.addComposite(propertyName, new ToOneIdMapper(relMapper, propertyName, referencedEntityName));
    }

    @SuppressWarnings({"unchecked"})
    private void addOneToOne(Property property, CompositeMapperBuilder mapper, String entityName) {
        OneToOne propertyValue = (OneToOne) property.getValue();

        String owningReferencePropertyName = propertyValue.getReferencedPropertyName(); // mappedBy

        EntityConfiguration configuration = entitiesConfigurations.get(entityName);
        if (configuration == null) {
            throw new MappingException("A versioned relation to a non-versioned entity " + entityName + "!");
        }

        IdMappingData ownedIdMapping = configuration.getIdMappingData();

        if (ownedIdMapping == null) {
            throw new MappingException("A versioned relation to a non-versioned entity " + entityName + "!");
        }

        String propertyName = property.getName();
        String lastPropertyPrefix = owningReferencePropertyName + "_";
        String referencedEntityName = propertyValue.getReferencedEntityName();

        // Generating the id mapper for the relation
        IdMapper ownedIdMapper = ownedIdMapping.getIdMapper().prefixMappedProperties(lastPropertyPrefix);

        // Storing information about this relation
        entitiesConfigurations.get(entityName).addOneToOneRelation(propertyName, owningReferencePropertyName,
                referencedEntityName, ownedIdMapper);

        // Adding mapper for the id
        mapper.addComposite(propertyName, new OneToOneIdMapper(owningReferencePropertyName,
                referencedEntityName, propertyName));
    }

    @SuppressWarnings({"unchecked"})
    private String getMappedBy(Collection collectionValue) {
        Iterator<Property> assocClassProps =
                ((OneToMany) collectionValue.getElement()).getAssociatedClass().getPropertyIterator();

        while (assocClassProps.hasNext()) {
            Property property = assocClassProps.next();

            if (Tools.iteratorsContentEqual(property.getValue().getColumnIterator(),
                    collectionValue.getKey().getColumnIterator())) {
                return property.getName();
            }
        }

        return null;
    }

    @SuppressWarnings({"unchecked"})
    private void addOneToMany(Property property, CompositeMapperBuilder mapper, String entityName) {
        Collection propertyValue = (Collection) property.getValue();

        String owningReferencePropertyName = getMappedBy(propertyValue);
        if (owningReferencePropertyName == null) {
            throw new MappingException("Unable to read the mapped by attribute for " + property.getName());
        }

        EntityConfiguration configuration = entitiesConfigurations.get(entityName);
        if (configuration == null) {
            throw new MappingException("A versioned relation to a non-versioned entity " + entityName + "!");
        }

        IdMappingData referencingIdMapping = configuration.getIdMappingData();

        String owningEntityName = ((OneToMany) propertyValue.getElement()).getReferencedEntityName();
        String propertyName = property.getName();
        String lastPropertyPrefix = owningReferencePropertyName + "_";

        // Generating the id mapper for the relation
        IdMapper ownedIdMapper = referencingIdMapping.getIdMapper().prefixMappedProperties(lastPropertyPrefix);

        // Storing information about this relation
        entitiesConfigurations.get(entityName).addOneToManyRelation(propertyName, owningReferencePropertyName,
                owningEntityName, ownedIdMapper);

        // Adding mapper for the id
        mapper.addComposite(propertyName, new OneToManyIdMapper(owningReferencePropertyName, owningEntityName,
                propertyName));
    }

    private ModificationStore getStoreForProperty(Property property, PropertyStoreInfo propertyStoreInfo) {
        /*
         * Checks if a property is versioned, which is when:
         * - the whole entity is versioned, then the default store is not null
         * - there is a store defined for this entity, which is when this property is annotated 
         */

        ModificationStore store = propertyStoreInfo.propertyStores.get(property.getName());

        if (store == null) {
            return propertyStoreInfo.defaultStore;
        }

        return store;
    }

    @SuppressWarnings({"unchecked"})
    private void addIdProperties(Element parent, Iterator<Property> properties, SimpleMapperBuilder mapper, boolean key) {
        while (properties.hasNext()) {
            Property property = properties.next();
            Type propertyType = property.getType();
            if (!"_identifierMapper".equals(property.getName())) {
                if (propertyType instanceof ImmutableType) {
                    addSimpleProperty(parent, property, mapper, ModificationStore.FULL, key);
                } else {
                    throw new MappingException("Type not supported: " + propertyType.getClass().getName());
                }
            }
        }
    }

    @SuppressWarnings({"unchecked"})
    private void addProperties(Element parent, Iterator<Property> properties, CompositeMapperBuilder currentMapper,
                               PropertyStoreInfo propertyStoreInfo, String entityName, boolean firstPass) {
        while (properties.hasNext()) {
            Property property = properties.next();
            Type propertyType = property.getType();
            if (!"_identifierMapper".equals(property.getName())) {
                ModificationStore store = getStoreForProperty(property, propertyStoreInfo);

                if (store != null) {
                    if (propertyType instanceof ComponentType) {
                        // only first pass
                        if (firstPass) {
                            addComponent(parent, property, currentMapper, store, entityName, firstPass);
                        }
                    } else if (propertyType instanceof ImmutableType || propertyType instanceof MutableType) {
                        // only first pass
                        if (firstPass) {
                            addSimpleProperty(parent, property, currentMapper, store, false);
                        }
                    } else if (propertyType instanceof CustomType &&
                            "org.hibernate.type.EnumType".equals(propertyType.getName())) {
                        // only first pass
                        if (firstPass) {
                            addEnumProperty(parent, property, currentMapper, store);
                        }
                    } else if (propertyType instanceof ManyToOneType) {
                        // only second pass
                        if (!firstPass) {
                            addToOne(parent, property, currentMapper, entityName);
                        }
                    } else if (propertyType instanceof OneToOneType) {
                        // only second pass
                        if (!firstPass) {
                            addOneToOne(property, currentMapper, entityName);
                        }
                    } else if ("org.hibernate.type.PrimitiveByteArrayBlobType".equals(
                            propertyType.getClass().getName())) {
                        // only first pass
                        if (firstPass) {
                            addSimpleProperty(parent, property, currentMapper, store, false);
                        }
                    } else if (propertyType instanceof CustomType &&
                            ("org.hibernate.type.PrimitiveCharacterArrayClobType".equals(propertyType.getName()) ||
                                    "org.hibernate.type.StringClobType".equals(propertyType.getName()))) {
                        // only first pass
                        if (firstPass) {
                            addSimpleProperty(parent, property, currentMapper, store, false);
                        }
                    } else if ((propertyType instanceof BagType || propertyType instanceof SetType) &&
                            ((((Collection) property.getValue()).getElement() instanceof OneToMany))) {
                        // only second pass
                        if (!firstPass) {
                            addOneToMany(property, currentMapper, entityName);
                        }
                    } else {
                        String message = "Type not supported for versioning: " + propertyType.getClass().getName() +
                                ", on entity " + entityName + ", property '" + property.getName() + "'.";
                        if (verCfg.isWarnOnUnsupportedTypes()) {
                            log.warn(message);
                        } else {
                            throw new MappingException(message);
                        }
                    }
                }
            }
        }
    }

    @SuppressWarnings({"unchecked"})
    private void createJoins(PersistentClass pc, Element parent, PersistentClassVersioningData versioningData) {
        Iterator<Join> joins = pc.getJoinIterator();

        Map<Join, Element> joinElements = new HashMap<Join, Element>();
        entitiesJoins.put(pc.getEntityName(), joinElements);

        while (joins.hasNext()) {
            Join join = joins.next();

            // Determining the table name. If there is no entry in the dictionary, just constructing the table name
            // as if it was an entity (by appending/prepending configured strings).
            String originalTableName = join.getTable().getName();
            String versionedTableName = versioningData.secondaryTableDictionary.get(originalTableName);
            if (versionedTableName == null) {
                versionedTableName = verEntCfg.getVersionsEntityName(originalTableName);
            }

            String schema = versioningData.schema;
            if (StringTools.isEmpty(schema)) {
                schema = join.getTable().getSchema();
            }

            String catalog = versioningData.catalog;
            if (StringTools.isEmpty(catalog)) {
                catalog = join.getTable().getCatalog();
            }

            Element joinElement = MetadataTools.createJoin(parent, versionedTableName, schema, catalog);
            joinElements.put(join, joinElement);

            Element joinKey = joinElement.addElement("key");
            addColumns(joinKey, join.getKey().getColumnIterator());
            MetadataTools.addColumn(joinKey, verEntCfg.getRevisionPropName(), null);
        }
    }

    @SuppressWarnings({"unchecked"})
    private void addJoins(PersistentClass pc, CompositeMapperBuilder currentMapper, PropertyStoreInfo propertyStoreInfo,
                          String entityName, boolean firstPass) {
        Iterator<Join> joins = pc.getJoinIterator();

        while (joins.hasNext()) {
            Join join = joins.next();
            Element joinElement = entitiesJoins.get(entityName).get(join);

            addProperties(joinElement, join.getPropertyIterator(), currentMapper, propertyStoreInfo, entityName,
                    firstPass);
        }
    }

    @SuppressWarnings({"unchecked"})
    private IdMappingData addId(PersistentClass pc) {
        // Mapping which will be used for relations
        Element rel_id_mapping = new DefaultElement("properties");
        // Mapping which will be used for the primary key of the versions table
        Element orig_id_mapping = new DefaultElement("composite-id");

        Property id_prop = pc.getIdentifierProperty();
        Component id_mapper = pc.getIdentifierMapper();

        SimpleIdMapperBuilder mapper;
        if (id_mapper != null) {
            mapper = new MultipleIdMapper(((Component) pc.getIdentifier()).getComponentClassName());
            addIdProperties(rel_id_mapping, (Iterator<Property>) id_mapper.getPropertyIterator(), mapper, false);

            // null mapper - the mapping where already added the first time, now we only want to generate the xml
            addIdProperties(orig_id_mapping, (Iterator<Property>) id_mapper.getPropertyIterator(), null, true);
        } else if (id_prop.isComposite()) {
            Component id_component = (Component) id_prop.getValue();

            mapper = new EmbeddedIdMapper(id_prop.getName(), id_component.getComponentClassName());
            addIdProperties(rel_id_mapping, (Iterator<Property>) id_component.getPropertyIterator(), mapper, false);

            // null mapper - the mapping where already added the first time, now we only want to generate the xml
            addIdProperties(orig_id_mapping, (Iterator<Property>) id_component.getPropertyIterator(), null, true);
        } else {
            mapper = new SingleIdMapper();

            addSimpleProperty(rel_id_mapping, id_prop, mapper, ModificationStore.FULL, false);

            // null mapper - the mapping where already added the first time, now we only want to generate the xml
            addSimpleProperty(orig_id_mapping, id_prop, null, ModificationStore.FULL, true);
        }

        orig_id_mapping.addAttribute("name", verEntCfg.getOriginalIdPropName());


        // Adding the "revision number" property
        Element rev_mapping = MetadataTools.addProperty(orig_id_mapping, verEntCfg.getRevisionPropName(),
                verEntCfg.getRevisionPropType(), true);
        MetadataTools.addColumn(rev_mapping, verEntCfg.getRevisionPropName(), null);

        return new IdMappingData(mapper, orig_id_mapping, rel_id_mapping);
    }

    private void addPersisterHack(Element class_mapping) {
        String persisterClassName;
        if (HibernateVersion.get().startsWith("3.3")) {
            persisterClassName = "org.jboss.envers.entity.VersionsInheritanceEntityPersisterFor33";
        } else {
            persisterClassName = "org.jboss.envers.entity.VersionsInheritanceEntityPersisterFor32";
        }

        class_mapping.addAttribute("persister", persisterClassName);
    }

    @SuppressWarnings({"unchecked"})
    public Document generateFirstPass(PersistentClass pc, PersistentClassVersioningData versioningData) {
        Document document = DocumentHelper.createDocument();

        String schema = versioningData.schema;
        if (StringTools.isEmpty(schema)) {
            schema = pc.getTable().getSchema();
        }

        String catalog = versioningData.catalog;
        if (StringTools.isEmpty(catalog)) {
            catalog = pc.getTable().getCatalog();
        }

        String entityName = pc.getEntityName();
        String versionsEntityName = verEntCfg.getVersionsEntityName(entityName);
        String versionsTableName = verEntCfg.getVersionsTableName(entityName, pc.getTable().getName());

        // Generating a mapping for the id
        IdMappingData idMapper = addId(pc);

        Element class_mapping;
        ExtendedPropertyMapper propertyMapper;

        InheritanceType inheritanceType = InheritanceType.get(pc);
        String parentEntityName = null;

        switch (inheritanceType) {
            case NONE:
                class_mapping = MetadataTools.createEntity(document, versionsEntityName, versionsTableName,
                        schema, catalog, pc.getDiscriminatorValue());
                propertyMapper = new MultiPropertyMapper();

                // Checking if there is a discriminator column
                if (pc.getDiscriminator() != null) {
                    Element discriminator_element = class_mapping.addElement("discriminator");
                    addColumns(discriminator_element, pc.getDiscriminator().getColumnIterator());
                    discriminator_element.addAttribute("type", pc.getDiscriminator().getType().getName());

                    // If so, there is some inheritance scheme -> using the persister hack.
                    addPersisterHack(class_mapping);
                }

                // Adding the id mapping
                class_mapping.add((Element) idMapper.getXmlMapping().clone());

                // Adding the "revision type" property
                MetadataTools.addProperty(class_mapping, verEntCfg.getRevisionTypePropName(),
                    verEntCfg.getRevisionTypePropType(), false);

                break;
            case SINGLE:
                String extendsEntityName = verEntCfg.getVersionsEntityName(pc.getSuperclass().getEntityName());
                class_mapping = MetadataTools.createSubclassEntity(document, versionsEntityName, versionsTableName,
                        schema, catalog, extendsEntityName, pc.getDiscriminatorValue());

                addPersisterHack(class_mapping);

                // The id and revision type is already mapped in the parent

                // Getting the property mapper of the parent - when mapping properties, they need to be included
                parentEntityName = pc.getSuperclass().getEntityName();
                ExtendedPropertyMapper parentPropertyMapper = entitiesConfigurations.get(parentEntityName).getPropertyMapper();
                propertyMapper = new SubclassPropertyMapper(new MultiPropertyMapper(), parentPropertyMapper);

                break;
            case JOINED:
                throw new MappingException("Joined inheritance strategy not supported for versioning!");
            case TABLE_PER_CLASS:
                throw new MappingException("Table-per-class inheritance strategy not supported for versioning!");
            default:
                throw new AssertionError("Impossible enum value.");
        }

        // Mapping unjoined properties
        addProperties(class_mapping, (Iterator<Property>) pc.getUnjoinedPropertyIterator(), propertyMapper,
                versioningData.propertyStoreInfo, pc.getEntityName(), true);

        // Creating and mapping joins (first pass)
        createJoins(pc, class_mapping, versioningData);
        addJoins(pc, propertyMapper, versioningData.propertyStoreInfo, pc.getEntityName(), true);

        // Storing the generated configuration
        EntityConfiguration entityCfg = new EntityConfiguration(entityName, versionsEntityName, idMapper, 
                propertyMapper, parentEntityName);
        entitiesConfigurations.put(pc.getEntityName(), entityCfg);

        return document;
    }

    @SuppressWarnings({"unchecked"})
    public void generateSecondPass(PersistentClass pc, PersistentClassVersioningData versioningData,
                                   Document document) {
        String entityName = pc.getEntityName();

        CompositeMapperBuilder propertyMapper = entitiesConfigurations.get(entityName).getPropertyMapper();

        // Mapping unjoined properties
        addProperties(document.getRootElement().element("class"), (Iterator<Property>) pc.getUnjoinedPropertyIterator(),
                propertyMapper, versioningData.propertyStoreInfo, entityName, false);

        // Mapping joins (second pass)
        addJoins(pc, propertyMapper, versioningData.propertyStoreInfo, entityName, false);
    }

    public Map<String, EntityConfiguration> getEntitiesConfigurations() {
        return entitiesConfigurations;
    }
}
