/*
 * Envers. http://www.jboss.org/envers
 *
 * Copyright 2008  Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT A WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License, v.2.1 along with this distribution; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * Red Hat Author(s): Adam Warski
 */
package org.jboss.envers.query.criteria;

import org.hibernate.criterion.*;
import org.jboss.envers.reader.VersionsReaderImplementor;
import org.jboss.envers.exception.VersionsException;

import java.util.List;
import java.util.ArrayList;

/**
 * @author Adam Warski (adam at warski dot org)
 */
public class AggregatedFieldVersionsExpression implements VersionsCriterion, ExtendableCriterion {
    public static enum AggregatedMode {
        MAX,
        MIN
    }

    private String propertyName;
    private AggregatedMode mode;
    private List<VersionsCriterion> criterions;

    public AggregatedFieldVersionsExpression(String propertyName, AggregatedMode mode) {
        this.propertyName = propertyName;
        this.mode = mode;
        criterions = new ArrayList<VersionsCriterion>();
    }

    public AggregatedFieldVersionsExpression add(VersionsCriterion criterion) {
        criterions.add(criterion);
        return this;
    }

    public Criterion toVersionsCriterion(String entityName, VersionsReaderImplementor versionsReader)
            throws VersionsException {
        CriteriaTools.checkPropertyNotARelation(versionsReader, entityName, propertyName);

        String versionsEntityName = versionsReader.getEntitiesCfg().getVersionsEntityName(entityName);

        // This will be the aggregated criteria, containing all the specified conditions
        DetachedCriteria aggregatedCriteria = DetachedCriteria.forEntityName(versionsEntityName);

        // This will contain all the specified conditions and an equal constraing on the result of the
        // aggregated criteria.
        Conjunction conjunction = Restrictions.conjunction();

        // First adding all specified conditions both to the main criteria, as well as to the
        // aggregated one.
        for (VersionsCriterion versionsCriteria : criterions) {
            Criterion transformedCriterion = versionsCriteria.toVersionsCriterion(entityName, versionsReader);

            conjunction.add(transformedCriterion);
            aggregatedCriteria.add(transformedCriterion);
        }

        // Setting the desired projection of the aggregated criteria
        switch (mode) {
            case MIN:
                aggregatedCriteria.setProjection(Property.forName(propertyName).min());
                break;
            case MAX:
                aggregatedCriteria.setProjection(Property.forName(propertyName).max());
        }      

        // Adding the constrain on the result of the aggregated criteria
        conjunction.add(Property.forName(propertyName).eq(aggregatedCriteria));

        return conjunction;
    }
}