/*
 * Envers. http://www.jboss.org/envers
 *
 * Copyright 2008  Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT A WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License, v.2.1 along with this distribution; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * Red Hat Author(s): Adam Warski
 */
package org.jboss.envers.query.impl;

import org.jboss.envers.reader.VersionsReaderImplementor;
import org.jboss.envers.entity.EntityInstantiator;
import org.jboss.envers.query.criteria.VersionsCriterion;
import org.jboss.envers.query.VersionsQuery;
import org.jboss.envers.query.projection.VersionsProjection;
import org.jboss.envers.query.order.VersionsOrder;
import org.jboss.envers.exception.VersionsException;
import org.hibernate.criterion.Projection;
import org.hibernate.criterion.Order;
import org.hibernate.Criteria;
import org.hibernate.FlushMode;
import org.hibernate.CacheMode;
import org.hibernate.LockMode;

import javax.persistence.NonUniqueResultException;
import javax.persistence.NoResultException;
import java.util.List;
import java.util.ArrayList;

/**
 * @author Adam Warski (adam at warski dot org)
 */
public abstract class AbstractVersionsQuery implements VersionsQuery {
    protected VersionsReaderImplementor versionsReader;
    protected EntityInstantiator entityInstantiator;
    protected List<VersionsCriterion> criterions;

    protected String entityName;
    protected String versionsEntityName;
    protected Criteria versionsCriteria;

    protected boolean hasProjection;
    protected boolean hasOrder;

    protected AbstractVersionsQuery(VersionsReaderImplementor versionsReader, Class<?> cls) {
        this.versionsReader = versionsReader;

        criterions = new ArrayList<VersionsCriterion>();
        entityInstantiator = new EntityInstantiator(versionsReader);

        entityName = cls.getName();
        versionsEntityName = versionsReader.getEntitiesCfg().getVersionsEntityName(entityName);
        versionsCriteria = versionsReader.getSession().createCriteria(versionsEntityName, "e");
    }

    public abstract List list() throws VersionsException;

    public List getResultList() throws VersionsException {
        return list();
    }

    public Object getSingleResult() throws VersionsException, NonUniqueResultException, NoResultException {
        List result = list();

        if (result == null || result.size() == 0) {
            throw new NoResultException();
        }

        if (result.size() > 1) {
            throw new NonUniqueResultException();
        }

        return result.get(0);
    }

    public VersionsQuery add(VersionsCriterion criterion) {
        criterions.add(criterion);
        return this;
    }

    // Various options

    public VersionsQuery setProjection(Projection projection) {
        hasProjection = true;
        versionsCriteria.setProjection(projection);
        return this;
    }

    public VersionsQuery setProjection(VersionsProjection projection) {
        hasProjection = true;
        versionsCriteria.setProjection(projection.getProjection(entityName, versionsReader));
        return this;
    }

    public VersionsQuery addOrder(Order order) {
        hasOrder = true;
        versionsCriteria.addOrder(order);
        return this;
    }

    public VersionsQuery addOrder(VersionsOrder order) {
        hasOrder = true;
        versionsCriteria.addOrder(order.getOrder(entityName, versionsReader));
        return this;
    }

    public VersionsQuery setMaxResults(int maxResults) {
        versionsCriteria.setMaxResults(maxResults);
        return this;
    }

    public VersionsQuery setFirstResult(int firstResult) {
        versionsCriteria.setFirstResult(firstResult);
        return this;
    }

    public VersionsQuery setCacheable(boolean cacheable) {
        versionsCriteria.setCacheable(cacheable);
        return this;
    }

    public VersionsQuery setCacheRegion(String cacheRegion) {
        versionsCriteria.setCacheRegion(cacheRegion);
        return this;
    }

    public VersionsQuery setComment(String comment) {
        versionsCriteria.setComment(comment);
        return this;
    }

    public VersionsQuery setFlushMode(FlushMode flushMode) {
        versionsCriteria.setFlushMode(flushMode);
        return this;
    }

    public VersionsQuery setCacheMode(CacheMode cacheMode) {
        versionsCriteria.setCacheMode(cacheMode);
        return this;
    }

    public VersionsQuery setTimeout(int timeout) {
        versionsCriteria.setTimeout(timeout);
        return this;
    }

    public VersionsQuery setLockMode(LockMode lockMode) {
        versionsCriteria.setLockMode(lockMode);
        return this;
    }
}
