/*
 * Envers. http://www.jboss.org/envers
 *
 * Copyright 2008  Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT A WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License, v.2.1 along with this distribution; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * Red Hat Author(s): Adam Warski
 */
package org.jboss.envers.reader;

import org.hibernate.NonUniqueResultException;
import org.hibernate.Session;
import org.hibernate.Query;
import org.hibernate.engine.SessionImplementor;
import org.jboss.envers.configuration.VersionsConfiguration;
import org.jboss.envers.configuration.VersionsEntitiesConfiguration;
import org.jboss.envers.exception.VersionsException;
import org.jboss.envers.exception.NotVersionedException;
import org.jboss.envers.exception.RevisionDoesNotExistException;

import javax.persistence.NoResultException;

import static org.jboss.envers.tools.ArgumentsTools.*;
import org.jboss.envers.query.VersionsQueryCreator;
import org.jboss.envers.query.VersionsRestrictions;
import org.jboss.envers.query.RevisionProperty;

import java.util.*;

/**
 * @author Adam Warski (adam at warski dot org)
 */
public class VersionsReaderImpl implements VersionsReaderImplementor {
    private VersionsConfiguration verCfg;
    private VersionsEntitiesConfiguration entitiesCfg;
    private SessionImplementor sessionImplementor;
    private Session session;

    public VersionsReaderImpl(VersionsConfiguration verCfg, Session session,
                              SessionImplementor sessionImplementor) {
        this.verCfg = verCfg;
        this.sessionImplementor = sessionImplementor;
        this.session = session;

        entitiesCfg = verCfg.getEntitiesCfg();
    }

    private void checkSession() {
        if (!session.isOpen()) {
            throw new IllegalStateException("The associated entity manager is closed!");
        }
    }

    public SessionImplementor getSessionImplementor() {
        return sessionImplementor;
    }

    public Session getSession() {
        return session;
    }

    public VersionsConfiguration getVerCfg() {
        return verCfg;
    }

    public VersionsEntitiesConfiguration getEntitiesCfg() {
        return entitiesCfg;
    }

    @SuppressWarnings({"unchecked"})
    public <T> T find(Class<T> cls, Object primaryKey, Number revision) throws
            IllegalArgumentException, NotVersionedException, IllegalStateException {
        checkNotNull(cls, "Entity class");
        checkNotNull(primaryKey, "Primary key");
        checkNotNull(revision, "Entity revision");
        checkPositive(revision, "Entity revision");
        checkSession();

        String entityName = cls.getName();

        if (!verCfg.isVersioned(entityName)) {
            throw new NotVersionedException(entityName + " is not versioned!");
        }

        try {
            return (T) createQuery().forEntitiesAtRevision(cls, revision)
                .add(VersionsRestrictions.idEq(primaryKey)).getSingleResult();
        } catch (NoResultException e) {
            return null;
        } catch (NonUniqueResultException e) {
            throw new VersionsException(e);
        }
    }

    public Object findOneReferencing(Class<?> cls, String owningEntityName, String owningReferencePropertyName,
                                     Object ownedEntityId, Number revision) {
        try {
            return createQuery().forEntitiesAtRevision(cls, revision)
                    .add(VersionsRestrictions.relatedIdEq(owningReferencePropertyName, ownedEntityId)).getSingleResult();
        } catch (NoResultException e) {
            return null;
        } catch (NonUniqueResultException e) {
            throw new VersionsException("Many versions results for one-to-one relationship: (" + owningEntityName +
                    ", " + owningReferencePropertyName + ")");
        }
    }

    @SuppressWarnings({"unchecked"})
    public Object findManyReferencing(Class<?> cls, String owningReferencePropertyName, Object ownedEntityId,
                                      Number revision, Class<? extends Collection> collectionClass) {
        List queryResult = createQuery().forEntitiesAtRevision(cls, revision)
                .add(VersionsRestrictions.relatedIdEq(owningReferencePropertyName, ownedEntityId)).getResultList();

        if (collectionClass.isAssignableFrom(queryResult.getClass())) {
            return queryResult;
        } else {
            Collection result;
            try {
                result = collectionClass.newInstance();
            } catch (Exception e) {
                throw new VersionsException(e);
            }

            result.addAll(queryResult);

            return result;
        }
    }

    @SuppressWarnings({"unchecked"})
    public List<Number> getRevisions(Class<?> cls, Object primaryKey)
            throws IllegalArgumentException, NotVersionedException, IllegalStateException {
        // todo: if a class is not versioned from the beginning, there's a missing ADD rev - what then?
        checkNotNull(cls, "Entity class");
        checkNotNull(primaryKey, "Primary key");
        checkSession();

        String entityName = cls.getName();

        if (!verCfg.isVersioned(entityName)) {
            throw new NotVersionedException(entityName + " is not versioned!");
        }

        return createQuery().forRevisionsOfEntity(cls, false)
                .setProjection(RevisionProperty.revisionNumber())
                .add(VersionsRestrictions.idEq(primaryKey))
                .getResultList();
    }

    public Date getRevisionDate(Number revision) throws IllegalArgumentException, RevisionDoesNotExistException,
            IllegalStateException{
        checkNotNull(revision, "Entity revision");
        checkPositive(revision, "Entity revision");
        checkSession();

        StringBuilder queryStr = new StringBuilder();
        queryStr.append("select rev.").append(entitiesCfg.getRevisionsInfoTimestampName())
                .append(" from ").append(entitiesCfg.getRevisionsInfoEntityName())
                .append(" rev where ").append(entitiesCfg.getRevisionsInfoIdName()).append(" = :_revision_number");

        Query query = session.createQuery(queryStr.toString()).setParameter("_revision_number", revision);

        try {
            Long timestamp = (Long) query.uniqueResult();
            if (timestamp == null) {
                throw new RevisionDoesNotExistException(revision);
            }

            return new Date(timestamp);
        } catch (NonUniqueResultException e) {
            throw new VersionsException(e);
        }
    }

    public Number getRevisionNumberForDate(Date date) {
        checkNotNull(date, "Date of revision");
        checkSession();

        StringBuilder queryStr = new StringBuilder();
        queryStr.append("select max(rev.").append(entitiesCfg.getRevisionsInfoIdName())
                .append(") from ").append(entitiesCfg.getRevisionsInfoEntityName())
                .append(" rev where ").append(entitiesCfg.getRevisionsInfoTimestampName()).append(" <= :_revision_date");

        Query query = session.createQuery(queryStr.toString()).setParameter("_revision_date", date.getTime());

        try {
            Number res = (Number) query.uniqueResult();
            if (res == null) {
                throw new RevisionDoesNotExistException(date);
            }

            return res;
        } catch (NonUniqueResultException e) {
            throw new VersionsException(e);
        }
    }

    @SuppressWarnings({"unchecked"})
    public <T> T findRevision(Class<T> revisionEntityClass, Number revision) throws IllegalArgumentException,
            RevisionDoesNotExistException, IllegalStateException {
        checkNotNull(revision, "Entity revision");
        checkPositive(revision, "Entity revision");
        checkSession();

        StringBuilder queryStr = new StringBuilder();
        queryStr.append("select rev from ")
                .append(entitiesCfg.getRevisionsInfoEntityName())
                .append(" rev where ")
                .append(entitiesCfg.getRevisionsInfoIdName())
                .append(" = :_revision_number");

        Query query = session.createQuery(queryStr.toString()).setParameter("_revision_number", revision);

        try {
            T revisionData = (T) query.uniqueResult();

            if (revisionData == null) {
                throw new RevisionDoesNotExistException(revision);
            }

            return revisionData;
        } catch (NonUniqueResultException e) {
            throw new VersionsException(e);
        }
    }

    public VersionsQueryCreator createQuery() {
        return new VersionsQueryCreator(this);
    }
}
