/*
 * Envers. http://www.jboss.org/envers
 *
 * Copyright 2008  Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT A WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License, v.2.1 along with this distribution; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * Red Hat Author(s): Adam Warski
 */
package org.jboss.envers.synchronization;

import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.FlushMode;
import org.jboss.envers.synchronization.work.VersionsWorkUnit;
import org.jboss.envers.tools.Pair;

import javax.transaction.Synchronization;
import java.util.*;

/**
 * @author Adam Warski (adam at warski dot org)
 */
public class VersionsSync implements Synchronization {
    private VersionsSyncManager manager;
    private Session session;
    private Number revision;
    private Transaction transaction;
    private LinkedList<VersionsWorkUnit> workUnits;
    private Queue<VersionsWorkUnit> undoQueue;
    private Map<Pair<String, Object>, VersionsWorkUnit> usedIds;

    public VersionsSync(VersionsSyncManager manager, Session session) {
        this.manager = manager;
        this.session = session;

        transaction = session.getTransaction();
        workUnits = new LinkedList<VersionsWorkUnit>();
        undoQueue = new LinkedList<VersionsWorkUnit>();
        usedIds = new HashMap<Pair<String, Object>, VersionsWorkUnit>();
    }

    private void generateRevision(Session session) {
        revision = (Number) session.save(manager.getEntitiesCfg().getRevisionsInfoEntityName(),
                manager.getEntitiesCfg().getRevisionInfoGenerator().newRevision());
    }

    private void removeWorkUnit(VersionsWorkUnit vwu) {
        workUnits.remove(vwu);
        if (vwu.isPerformed()) {
            // If this work unit has already been performed, it must be deleted (undone) first.
            undoQueue.offer(vwu);
        }
    }

    public void addWorkUnit(VersionsWorkUnit vwu) {
        if (vwu.containsWork()) {
            Object entityId = vwu.getEntityId();
            String entityName = vwu.getEntityName();
            Pair<String, Object> usedIdsKey = Pair.make(entityName, entityId);

            if (usedIds.containsKey(usedIdsKey)) {
                VersionsWorkUnit other = usedIds.get(usedIdsKey);

                // The entity with entityId has two work units; checking which one should be kept.
                switch (vwu.dispatch(other)) {
                    case FIRST:
                        // Simply not adding the second
                        break;

                    case SECOND:
                        removeWorkUnit(other);
                        usedIds.put(usedIdsKey, vwu);
                        workUnits.offer(vwu);
                        break;

                    case NONE:
                        removeWorkUnit(other);
                }
            } else {
                usedIds.put(usedIdsKey, vwu);
                workUnits.offer(vwu);
            }
        }
    }

    private void executeInSession(Session session) {
        if (revision == null) {
            generateRevision(session);
        }

        VersionsWorkUnit vwu;

        // First undoing any performed work units
        while ((vwu = undoQueue.poll()) != null) {
            vwu.undo(session);
        }

        while ((vwu = workUnits.poll()) != null) {
            vwu.perform(session, revision);
        }
    }

    public void beforeCompletion() {
        if (workUnits.size() == 0 && undoQueue.size() == 0) {
            return;
        }

        if (FlushMode.isManualFlushMode(session.getFlushMode())) {
            Session temporarySession = null;
            try {
                temporarySession = session.getSessionFactory().openSession(session.connection());

                executeInSession(temporarySession);

                temporarySession.flush();
            } finally {
                if (temporarySession != null) {
                    temporarySession.close();
                }
            }
        } else {
            executeInSession(session);

            // Explicity flushing the session, as the auto-flush may have already happened.
            session.flush();
        }
    }

    public void afterCompletion(int i) {
        manager.remove(transaction);
    }
}
