package org.jboss.envers.test.integration.onetomany;

import org.jboss.envers.test.integration.AbstractEntityTest;
import org.jboss.envers.test.entities.onetomany.CollectionRefEdEntity;
import org.jboss.envers.test.entities.onetomany.CollectionRefIngEntity;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;
import org.hibernate.ejb.Ejb3Configuration;

import javax.persistence.EntityManager;
import java.util.Arrays;
import java.util.Set;
import java.util.HashSet;
import java.util.Collections;

/**
 * @author Adam Warski (adam at warski dot org)
 */
public class BasicCollection extends AbstractEntityTest {
    private Integer ed1_id;
    private Integer ed2_id;

    private Integer ing1_id;
    private Integer ing2_id;

    public void configure(Ejb3Configuration cfg) {
        cfg.addAnnotatedClass(CollectionRefEdEntity.class);
        cfg.addAnnotatedClass(CollectionRefIngEntity.class);
    }

    @BeforeClass(dependsOnMethods = "init")
    public void initData() {
        EntityManager em = getEntityManager();

        CollectionRefEdEntity ed1 = new CollectionRefEdEntity(1, "data_ed_1");
        CollectionRefEdEntity ed2 = new CollectionRefEdEntity(2, "data_ed_2");

        CollectionRefIngEntity ing1 = new CollectionRefIngEntity(3, "data_ing_1", ed1);
        CollectionRefIngEntity ing2 = new CollectionRefIngEntity(4, "data_ing_2", ed1);

        // Revision 1
        em.getTransaction().begin();

        em.persist(ed1);
        em.persist(ed2);

        em.persist(ing1);
        em.persist(ing2);

        em.getTransaction().commit();

        // Revision 2
        em.getTransaction().begin();

        ing1 = em.find(CollectionRefIngEntity.class, ing1.getId());
        ed2 = em.find(CollectionRefEdEntity.class, ed2.getId());

        ing1.setReference(ed2);

        em.getTransaction().commit();

        // Revision 3
        em.getTransaction().begin();

        ing2 = em.find(CollectionRefIngEntity.class, ing2.getId());
        ed2 = em.find(CollectionRefEdEntity.class, ed2.getId());

        ing2.setReference(ed2);

        em.getTransaction().commit();

        //

        ed1_id = ed1.getId();
        ed2_id = ed2.getId();

        ing1_id = ing1.getId();
        ing2_id = ing2.getId();
    }

    @Test
    public void testRevisionsCounts() {
        assert Arrays.asList(1, 2, 3).equals(getVersionsReader().getRevisions(CollectionRefEdEntity.class, ed1_id));
        assert Arrays.asList(1, 2, 3).equals(getVersionsReader().getRevisions(CollectionRefEdEntity.class, ed2_id));

        assert Arrays.asList(1, 2).equals(getVersionsReader().getRevisions(CollectionRefIngEntity.class, ing1_id));
        assert Arrays.asList(1, 3).equals(getVersionsReader().getRevisions(CollectionRefIngEntity.class, ing2_id));
    }

    private <T> Set<T> makeSet(T... objects) {
        Set<T> ret = new HashSet<T>();
        //noinspection ManualArrayToCollectionCopy
        for (T obj : objects) { ret.add(obj); }
        return ret;
    }

    @Test
    public void testHistoryOfEdId1() {
        CollectionRefIngEntity ing1 = getEntityManager().find(CollectionRefIngEntity.class, ing1_id);
        CollectionRefIngEntity ing2 = getEntityManager().find(CollectionRefIngEntity.class, ing2_id);

        CollectionRefEdEntity rev1 = getVersionsReader().find(CollectionRefEdEntity.class, ed1_id, 1);
        CollectionRefEdEntity rev2 = getVersionsReader().find(CollectionRefEdEntity.class, ed1_id, 2);
        CollectionRefEdEntity rev3 = getVersionsReader().find(CollectionRefEdEntity.class, ed1_id, 3);

        assert rev1.getReffering().equals(makeSet(ing1, ing2));
        assert rev2.getReffering().equals(makeSet(ing2));
        assert rev3.getReffering().equals(Collections.EMPTY_SET);
    }

    @Test
    public void testHistoryOfEdId2() {
        CollectionRefIngEntity ing1 = getEntityManager().find(CollectionRefIngEntity.class, ing1_id);
        CollectionRefIngEntity ing2 = getEntityManager().find(CollectionRefIngEntity.class, ing2_id);

        CollectionRefEdEntity rev1 = getVersionsReader().find(CollectionRefEdEntity.class, ed2_id, 1);
        CollectionRefEdEntity rev2 = getVersionsReader().find(CollectionRefEdEntity.class, ed2_id, 2);
        CollectionRefEdEntity rev3 = getVersionsReader().find(CollectionRefEdEntity.class, ed2_id, 3);

        assert rev1.getReffering().equals(Collections.EMPTY_SET);
        assert rev2.getReffering().equals(makeSet(ing1));
        assert rev3.getReffering().equals(makeSet(ing1, ing2));
    }

    @Test
    public void testHistoryOfEdIng1() {
        CollectionRefEdEntity ed1 = getEntityManager().find(CollectionRefEdEntity.class, ed1_id);
        CollectionRefEdEntity ed2 = getEntityManager().find(CollectionRefEdEntity.class, ed2_id);

        CollectionRefIngEntity rev1 = getVersionsReader().find(CollectionRefIngEntity.class, ing1_id, 1);
        CollectionRefIngEntity rev2 = getVersionsReader().find(CollectionRefIngEntity.class, ing1_id, 2);
        CollectionRefIngEntity rev3 = getVersionsReader().find(CollectionRefIngEntity.class, ing1_id, 3);

        assert rev1.getReference().equals(ed1);
        assert rev2.getReference().equals(ed2);
        assert rev3.getReference().equals(ed2);
    }

    @Test
    public void testHistoryOfEdIng2() {
        CollectionRefEdEntity ed1 = getEntityManager().find(CollectionRefEdEntity.class, ed1_id);
        CollectionRefEdEntity ed2 = getEntityManager().find(CollectionRefEdEntity.class, ed2_id);

        CollectionRefIngEntity rev1 = getVersionsReader().find(CollectionRefIngEntity.class, ing2_id, 1);
        CollectionRefIngEntity rev2 = getVersionsReader().find(CollectionRefIngEntity.class, ing2_id, 2);
        CollectionRefIngEntity rev3 = getVersionsReader().find(CollectionRefIngEntity.class, ing2_id, 3);

        assert rev1.getReference().equals(ed1);
        assert rev2.getReference().equals(ed1);
        assert rev3.getReference().equals(ed2);
    }
}