/*
 * Copyright 2011 JBoss, by Red Hat, Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.errai.enterprise.rebind;

import java.lang.annotation.Annotation;
import java.util.List;
import java.util.Set;

import javax.enterprise.event.Observes;
import javax.enterprise.util.TypeLiteral;

import org.jboss.errai.bus.client.api.Local;
import org.jboss.errai.bus.client.api.Message;
import org.jboss.errai.bus.client.framework.MessageBus;
import org.jboss.errai.codegen.framework.Context;
import org.jboss.errai.codegen.framework.Parameter;
import org.jboss.errai.codegen.framework.Statement;
import org.jboss.errai.codegen.framework.builder.AnonymousClassStructureBuilder;
import org.jboss.errai.codegen.framework.builder.BlockBuilder;
import org.jboss.errai.codegen.framework.meta.MetaMethod;
import org.jboss.errai.codegen.framework.meta.MetaParameter;
import org.jboss.errai.codegen.framework.util.Bool;
import org.jboss.errai.codegen.framework.util.Refs;
import org.jboss.errai.codegen.framework.util.Stmt;
import org.jboss.errai.enterprise.client.cdi.AbstractCDIEventCallback;
import org.jboss.errai.enterprise.client.cdi.CDIProtocol;
import org.jboss.errai.enterprise.client.cdi.api.CDI;
import org.jboss.errai.ioc.client.api.CodeDecorator;
import org.jboss.errai.ioc.rebind.ioc.IOCDecoratorExtension;
import org.jboss.errai.ioc.rebind.ioc.InjectUtil;
import org.jboss.errai.ioc.rebind.ioc.InjectableInstance;

/**
 * Generates the boiler plate for @Observes annotations use in GWT clients.<br/>
 * Basically creates a subscription for a CDI event type that invokes on the annotated method.
 *
 * @author Heiko Braun <hbraun@redhat.com>
 * @author Mike Brock <cbrock@redhat.com>
 * @author Christian Sadilek <csadilek@redhat.com>
 */
@CodeDecorator
public class ObservesExtension extends IOCDecoratorExtension<Observes> {

  public ObservesExtension(Class<Observes> decoratesWith) {
    super(decoratesWith);
  }

  @Override
  public Statement generateDecorator(InjectableInstance<Observes> instance) {
    final Context ctx = instance.getInjectionContext().getProcessingContext().getContext();
    final MetaMethod method = instance.getMethod();
    final MetaParameter parm = instance.getParm();

    if (!method.isPublic()) {
      instance.ensureMemberExposed();
    }

    final String parmClassName = parm.getType().getFullyQualifiedName();
    final Statement bus = instance.getInjectionContext().getInjector(MessageBus.class).getType(instance);
    final String subscribeMethodName = method.isAnnotationPresent(Local.class) ? "subscribeLocal" : "subscribe";

    final String subject = CDI.getSubjectNameByType(parmClassName);
    final Annotation[] qualifiers = InjectUtil.extractQualifiers(instance).toArray(new Annotation[0]);
    final List<String> qualifierNames = CDI.getQualifiersPart(qualifiers);

    AnonymousClassStructureBuilder callBack = Stmt.newObject(AbstractCDIEventCallback.class).extend();

    BlockBuilder<AnonymousClassStructureBuilder> callBackBlock;
    if (qualifierNames != null) {
      callBackBlock = callBack.initialize();
      for (String qualifierName : qualifierNames) {
        callBackBlock.append(Stmt.loadClassMember("qualifierSet").invoke("add", qualifierName));
      }
      callBack = callBackBlock.finish();
    }

    callBackBlock = callBack.publicOverridesMethod("callback", Parameter.of(Message.class, "message"))
            .append(Stmt.declareVariable("msgQualifiers", new TypeLiteral<Set<String>>() {
            },
                    Stmt.loadVariable("message").invoke("get", Set.class, CDIProtocol.Qualifiers)))
            .append(Stmt
                    .if_(Bool.or(
                            Stmt.loadClassMember("qualifierSet").invoke("equals", Refs.get("msgQualifiers")),
                            Bool.and(Bool.equals(Refs.get("msgQualifiers"), null),
                                    Stmt.loadClassMember("qualifierSet").invoke("isEmpty"))))
                    .append(instance.callOrBind(Stmt.loadVariable("message").invoke("get", parm.getType().asClass(), CDIProtocol.BeanReference)))
                    .finish());

    return Stmt.create(ctx).nestedCall(bus).invoke(subscribeMethodName, subject, callBackBlock.finish().finish());
  }
}