package org.jboss.fresh.deployer;

/**
 * mbean that registers diferent scripting engines
 */

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.SAXParser;

import org.apache.bsf.BSFManager;
import org.apache.bsf.util.event.EventAdapterRegistry;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;
import org.jboss.fresh.deployer.ScriptingCentral;

import java.io.InputStream;
import java.io.FileInputStream;
import java.net.URL;
import java.util.StringTokenizer;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;
import java.util.LinkedList;


public class ScriptingService extends RegistryNamingBinder implements ScriptingServiceMBean {

	private String cfgPath;
	private List langs = new LinkedList();
	private List languages = new LinkedList();

	public String getName() {
		return "Scripting Services";
	}

	// if not set we use a context classloader
	public void setConfigPath(String val) {
		cfgPath = val;
	}
	
	public String getConfigPath() {
		return cfgPath;
	}
	
	
	// The format of the config file
	/*
	
	<scripting>
		<provider language="javascript"  ext="js"  class="com.ibm.bsf.engines.javascript.JavaScriptEngine" />
		<provider language="jython"      ext="jy"  class="com.ibm.bsf.engines.jython.JythonEngine" />
		<provider language="beanshell"   ext="bsh" class="bsh.util.BeanShellBSFEngine" />
		<provider language="judoscript"  ext="jud;judo" class="com.judoscript.BSFJudoEngine" />

		<event-adaper listener-class="" adapter-class="" />
		<event-adaper listener-class="" adapter-class="" />
		<event-adaper listener-class="" adapter-class="" />
	</scripting>
	
	*/
	
	public void doStart() throws Exception {
		// lookup the config file and initialize stuff
		
		SAXParserFactory factory = SAXParserFactory.newInstance();
		SAXParser parser = factory.newSAXParser();
	
		
		InputStream in = null;
		
		if(cfgPath!=null) {
			in = new FileInputStream(cfgPath);
		} else {
		        ClassLoader cl = Thread.currentThread().getContextClassLoader();
	        	URL url = cl.getResource("fresh-scripting.xml");
	        	if (url != null) {
		        	in = url.openStream();
		        }
		}
		
		parser.parse(in, new DefaultHandler() {
		
			int level = 0;
			
			public void startElement(String uri, String localName, String qName, Attributes attrs) throws SAXException {
		
			   if(level==1) {
				if(qName.equals("provider")) {
	
					String clazz = null;
					String lang = null;
					String ext = null;


					int len = attrs.getLength();
					for(int i=0; i<len; i++) {
						String aname = attrs.getQName(i);
						String   val = attrs.getValue(i);
	
						if(aname.equals("language")) {
							lang = val;
						} else if(aname.equals("ext")) {
							ext = val;
						} else if(aname.equals("class")) {
							clazz = val;
						} else {
							throw new SAXException("Element provider does not support attribute: " + aname);
						}
					}

					addProvider(lang, ext, clazz);

				} else if(qName.equals("event-adapter")) {

					String listener = null;
					String adapter = null;

					int len = attrs.getLength();
					for(int i=0; i<len; i++) {
						String aname = attrs.getQName(i);
						String   val = attrs.getValue(i);
	
						if(aname.equals("listener-class")) {
							listener = val;
						} else if(aname.equals("adapter-class")) {
							adapter = val;
						} else {
							throw new SAXException("Element event-adapter does not support attribute: " + aname);
						}
					}
					
					registerEventAdapter(listener, adapter);

				} else {
					throw new SAXException("Unknown element: " + qName);
				}
				
			   } else if(level==0) {
				if(!qName.equals("scripting"))
					throw new SAXException("Unknown element: " + qName);

			   }

			   level++;
			}
			
			public void endElement(String uri, String localName, String qName) {
			   level--;
			}
		});
	

		super.doStart();
	
	}


	public BSFManager getManager(String lang) {
		Language scriptLang = getLanguage(lang);		
		if(scriptLang==null) throw new RuntimeException("Language not supported: " + lang);
		
		BSFManager manager=new BSFManager();
		BSFManager.registerScriptingEngine(scriptLang.getIdentifier(),scriptLang.getScriptEngine(),scriptLang.getExtensions());

		return manager;
	}


	public void registerEventAdapter(String listenerClass, String adapterClass) {

		ClassLoader cl = Thread.currentThread().getContextClassLoader();
		Class cls1 = null;
		Class cls2 = null;

		try {
			cls1 = cl.loadClass(listenerClass);
		} catch(Exception ex) {
			throw new RuntimeException("Failed to load class: " + listenerClass, ex);
		}

		try {
			cls2 = cl.loadClass(adapterClass);	
		} catch(Exception ex) {
			throw new RuntimeException("Failed to load class: " + adapterClass, ex);
		}

		EventAdapterRegistry.register(cls1, cls2);
	}


	public void addProvider(String language, String filext, String clazz) {

		if(language==null) throw new IllegalArgumentException("Language can't be null");
		if(clazz==null) throw new IllegalArgumentException("Class can't be null");

		// parse file extensions;
		LinkedList exts = new LinkedList();
		String [] aexts;
		if(filext!=null) {
			StringTokenizer st = new StringTokenizer(filext, "; ");
			while(st.hasMoreTokens()) {
				exts.add(st.nextToken());
			}
		}

		aexts = new String[exts.size()];
		Iterator it = exts.iterator();
		for(int i=0; it.hasNext(); i++) {
			aexts[i]=(String)it.next();
		}

	        languages.add(new Language(language, aexts, clazz));
	        langs.add(language);
	}


	private Language getLanguage(String lang){

	        Iterator it = languages.iterator();
	        while(it.hasNext()) {
	        	Language l = (Language)it.next();
	        	if(l.getIdentifier().equals(lang)) return l;
	        }

		return null;
	}

	public List getSupportedLanguages() {
		return new ArrayList(langs);
	}


	protected String getBindClass() {
		return ScriptingCentral.class.getName();
	}

	protected Object classToInstance(Class c) {
		setServiceObject( this );
		//return new ScriptingServiceJMXProxy(getRegisteredName());
		//return java.lang.reflect.Proxy.newProxyInstance(Thread.currentThread().getContextClassLoader(), new Class [] {com.parsek.cp2.scripting.ScriptingCentral.class}, new JMXProxyHandler(getRegisteredName()));
		return this;
	}
 
 	
 	// 
 	public boolean isLanguageSupported(String name) {
		Iterator it = new ArrayList(langs).iterator();
		while(it.hasNext()) {
			if(it.next().equals(name)) return true;
		}
		
		return false;
	}
	
 	// if we support aliases we return a logical name of the language
 	public String getLanguageForName(String name) {
		Iterator it = new ArrayList(langs).iterator();
		while(it.hasNext()) {
			if(it.next().equals(name)) return name;
		}
		
		return null;
 	}
 	
 	// we try determine a language for this extension
 	public String getLanguageForExtension(String ext) {
 		Iterator it = new ArrayList(languages).iterator();
		while(it.hasNext()) {
			Language lng = (Language)it.next();
			String [] extz = lng.getExtensions();
			for(int i=0; i<extz.length; i++) {
				if(extz[i].equals(ext)) return lng.getIdentifier();
			}
		}

		return null;
 	}



    protected boolean bindByReference() {
	return true;
    }


    public class Language {

        private String identifier;
        private String[] extensions;
        private String scriptEngine;

        public Language(String identifier,String[] extensions, String scriptEngine){
            this.identifier=identifier;
            this.extensions=extensions;
            this.scriptEngine=scriptEngine;
        }

        public String getIdentifier(){
            return identifier;
        }
        public String[] getExtensions(){
            return extensions;
        }
        public String getScriptEngine(){
            return scriptEngine;
        }

    }

}