/**
 *Gives scripting capabilities to CP2 shell
 *  - supports javascript, jython and beanshell (uses Rhino, Jython and BeanShell)
 *
 * @author  derganc
 */
package org.jboss.fresh.shell.commands;


import org.jboss.fresh.vfs.impl.VFSInputStream;
import org.jboss.fresh.vfs.impl.SecureVFS;
import org.jboss.fresh.vfs.VFS;
import org.jboss.fresh.vfs.FileName;
import org.jboss.fresh.ctx.Context;
import org.jboss.fresh.deployer.ScriptingCentral;
import org.jboss.fresh.io.BufferInputStream;
import org.jboss.fresh.io.BufferOutputStream;
import org.jboss.fresh.io.BufferReader;
import org.jboss.fresh.io.BufferWriter;
import org.jboss.fresh.io.InBuffer;
import org.jboss.fresh.io.OutBuffer;
import org.jboss.fresh.shell.AbstractExecutable;
import org.jboss.fresh.io.IOUtils;

import java.io.PrintWriter;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.Writer;
import java.util.Properties;

import org.apache.bsf.BSFException;
import org.apache.bsf.BSFManager;


public class ScriptExe extends AbstractExecutable {
	private static transient org.apache.log4j.Logger log = org.apache.log4j.Logger.getLogger(ScriptExe.class);

	//GLAVNA METODA RAZREDA
	public void process(String execName, String[] params) throws java.lang.Exception {
		log.debug("entered");

		String langName = null, fname = "unknown";

		//Variables to be declared to the BSFManager (are available in scripts under aliases)
		Writer out = null;
		Reader in = null;
		BufferInputStream inStream = null;
		BufferOutputStream outStream = null;
		Context context = null;
		Properties props = null;

		boolean isFile = false;

		if (helpRequested()) {
			printHelp();
			return;
		}


		PrintWriter err = new PrintWriter(new BufferWriter(getStdOut()), true);

		// now lookup ScriptingServices
		ScriptingCentral sc = (ScriptingCentral) getShell().getContext().get("ScriptingServices");
		if (sc == null) {
			if (canThrowEx()) {
				throw new RuntimeException("ScriptingServices not bound in context.");
			} else {
				err.println("ScriptingServices not bound in context.");
				return;
			}
		}

        String file = null;
        String vfile = null;
		Reader reader = null;
		String enc = shell.getEnvProperty("ENCODING");

		String [] args = null;


        if(params.length >0) {
            int i = 0;
            String temp = params[i];

            if("-f".equals(temp) || "--file".equals(temp)) {
                if(i<params.length-1) {
                    file = params[++i];
                } else {
                    error("File name missing after parameter: " + temp);
                    return;
                }
            } else if("-v".equals(params[0]) || "--vfile".equals(params[0])) {
                if(i<params.length-1) {
                    vfile = params[++i];
                } else {
                    error("File name missing after parameter: " + temp);
                    return;
                }
            }
        }


        if(file != null || vfile != null) {
            args = new String[params.length-2];
            for(int i=2; i<params.length; i++) {
                args[i-2] = params[i];
            }


            fname = file != null ? file : vfile;

            String ext = IOUtils.getExt(fname);
            if (ext == null) {
                if (canThrowEx()) {
                    throw new RuntimeException("Can't determine scripting language - file has no extension.");
                } else {
                    err.println("Can't determine scripting language - file has no extension.");
                    return;
                }
            }


            langName = sc.getLanguageForExtension(ext);
            if (langName == null) {
                if (canThrowEx()) {
                    throw new RuntimeException("No scripting language registered for the specified extensions: " + ext);
                } else {
                    err.println("No scripting language registered for the specified extensions: " + ext);
                    return;
                }
            }

            InputStream ins = null;
            if(file != null) {
                ins = new FileInputStream(file);
            } else {
                VFS vfs = shell.getVFS();
                FileName pwd = new FileName(shell.getEnvProperty("PWD"));
                FileName path = new FileName(vfile);
                if (path.isRelative())
                    path = pwd.absolutize(path);

                path = vfs.resolve(shell.getUserCtx(), path, false);

                ins = new VFSInputStream(new SecureVFS(vfs, shell.getUserCtx()), path.toString());
            }

            if (enc == null || enc.trim().length() == 0) {
                reader = new BufferedReader(new InputStreamReader(ins));
            } else {
                reader = new BufferedReader(new InputStreamReader(ins, enc));
            }

            isFile = true;

        } else {
            args = params;

            InputStream ins = new BufferInputStream(getStdIn());
            if (enc == null || enc.trim().length() == 0) {
                reader = new BufferedReader(new InputStreamReader(ins));
            } else {
                reader = new BufferedReader(new InputStreamReader(ins, enc));
            }

            langName = ((BufferedReader) reader).readLine();

            if (langName.length() > 2)
                langName = langName.substring(2).trim();

            String lngName = sc.getLanguageForName(langName);
            if (lngName == null) {
                if (canThrowEx()) {
                    throw new RuntimeException("Scripting language is not supported: " + langName);
                } else {
                    err.println("Scripting language is not supported: " + langName);
                    return;
                }
            }

            langName = lngName;
        }


		BSFManager manager = sc.getManager(langName);

		inStream = getStdInStream();
		outStream = getStdOutStream();
		context = shell.getContext();
		props = shell.getEnvProperties();
		out = new PrintWriter(new BufferedWriter(new BufferWriter(getStdOut())), true);
		in = new BufferedReader(new BufferReader(getStdIn()));
		InBuffer stdin = getStdIn();
		OutBuffer stdout = getStdOut();

		manager.declareBean("args", args, args.getClass());
		manager.declareBean("shell", shell, shell.getClass());
		manager.declareBean("instream", inStream, inStream.getClass());
		manager.declareBean("outstream", outStream, outStream.getClass());
		manager.declareBean("context", context, context.getClass());
		manager.declareBean("props", props, props.getClass());
		manager.declareBean("out", out, out.getClass());
		manager.declareBean("cin", in, in.getClass());
		manager.declareBean("stdin", stdin, stdin.getClass());
		manager.declareBean("stdout", stdout, stdout.getClass());
		manager.declareBean("console", System.out, System.out.getClass());

		log.debug("Info: Scripting language: " + langName);
		try {
			manager.exec(langName, fname, (isFile ? 1 : 2), 0,
					IOUtils.readerToString(reader, 50000));
        } catch (BSFException ex) {
            Throwable e = ex;
            Throwable cause = ex.getTargetException();
            if(cause != null)
                e = cause;

			if (canThrowEx()) {
				log.error(ex.getMessage(), e);
                wrapup(in, inStream, reader, out, outStream, manager);
                throw new RuntimeException("Exception while executing a script: ", e);
			} else {
				Throwable t = e.getCause();
                err.println("Exception while executing a script: " + e + " (" + (t != null ? String.valueOf(t) : "") + ")");
                wrapup(in, inStream, reader, out, outStream, manager);
                return;
			}

        } catch (Throwable ex) {
			if (canThrowEx()) {
				log.error(ex.getMessage(), ex);
                wrapup(in, inStream, reader, out, outStream, manager);
                throw new RuntimeException("Exception while executing a script: ", ex);
			} else {
				err.println("Exception while executing a script: " + ex.toString() + " (" + (ex.getCause() != null ? ex.getCause().toString() : "") + ")");
				wrapup(in, inStream, reader, out, outStream, manager);
                return;
			}
		}

        wrapup(in, inStream, reader, out, outStream, manager);
		//finalize
/*
		out.flush();
		outStream.flush();
        out.close();
        reader.close();
		outStream.close();
		inStream.close();
		in.close();
		manager.terminate();
*/
		log.debug("done");
	}

    private void wrapup(Reader in, BufferInputStream inStream, Reader reader, Writer out, BufferOutputStream outStream, BSFManager manager) {
        try {
            in.close();
        } catch(Exception ex) {}

        try {
            inStream.close();
        } catch(Exception ex) {}

        try {
            reader.close();
        } catch(Exception ex) {}

        try {
            out.flush();
        } catch(Exception ex) {}

        try {
            outStream.flush();
        } catch(Exception ex) {}

        /*
        try {
            out.close();
        } catch(Exception ex) {}

        try {
            outStream.close();
        } catch(Exception ex) {}
        */

        manager.terminate();
    }


    private void printHelp() {
        PrintWriter out = new PrintWriter(new BufferWriter(getStdOut()));
        out.print("Usage: script [-ex] [--help] [-f/--file | -v/--vfile] SCRIPTFILENAME\n");
        out.print("		Excutable for runing scripts\n");
        out.print("		--help : this help\n");
        out.print("	 -f/--file : read from native filesystem\n");
        out.print("                Usage examples:\n");
        out.print("                     script arg1 arg2 &< stream\n");
        out.print("                     script -v javascriptFile.js arg1 arg2\n");
        out.print("                     script -f javascriptFile.js arg1 arg2\n");
        out.print("\n");
        out.print("Available variables:\n");
        out.print("    args      ...  array of String - CLI parameters\n");
        out.print("    shell     ...  org.jboss.fresh.shell.impl.ShellImpl\n");
        out.print("    instream  ...  java.io.InputStream wrapped around stdin\n");
        out.print("    outstream ...  java.io.OutputStream wrapped around stdout\n");
        out.print("    context   ...  org.jboss.fresh.ctx.Context - shell.getContext()\n");
        out.print("    props     ...  java.util.Properties - shell.getEnvProperties()\n");
        out.print("    out       ...  java.io.PrintWriter - wrapped around stdout\n");
        out.print("    cin       ...  java.io.BufferedReader - wrapped around stdin\n");
        out.print("    stdin     ...  org.jboss.fresh.io.InBuffer\n");
        out.print("    stdout    ...  org.jboss.fresh.io.OutBuffer\n");
        out.print("    console   ...  java.lang.System.out - java.io.PrintStream\n");
        out.print("\n");
        out.print("NOTE: -If using a script file scripting language is determined by extension!\n");
        out.print("      -If using a stream scripting language is determined by the first line!\n\n");
        out.close();
        log.debug("done");
    }
}