/**
 *   All the methods of this interface are called by <code>Pool</code> instance on which this <code>PoolFactory</code>
 *   is installed.
 */
package org.jboss.fresh.pool.pool;

public interface PoolFactory {

	/**
	 *  Called once when this <code>PoolFactory</code> instance is installed on the pool - when pool is created.
	 */
	public void registerPool(Pool p);


	/**
	 *  Called once when this <code>PoolFactory</code> instance is uninstalled from the pool - when pool is destroyed.
	 */
	public void unregisterPool();


	/**
	 *  Pool requests us to create a new instance of the pooled resource
	 */
	public Object createObject() throws Exception;


	/**
	 *  Pool requests us to do all cleanup of object specific data for the object being deleted from the pool.
	 */
	public void deleteObject(Object o) throws Exception;


	/**
	 *  Pool informs us that the object is about to be checked out from the pool. We can reinitialize it...
	 */
	public void retrieveObject(Object o) throws Exception;


	/**
	 *  Pool informs us that the object has been returned to the pool. We can cleanup any active state - reinit it.
	 */
	public void returnObject(Object o) throws Exception;


	/**
	 *  Pool requests us to fix the object that was returned to the pool
	 *  through badCheckout method. If fixing it is not possible return a
	 *  new object. The requirement is that the object need to be as good
	 *  as new.
	 * 	@param o Object to be recycled 
	 *  @param extra Optional PoolFactory implementation specific object that gives more information about how to recicle the object  
	 */
	public Object recycleObject(Object o, Object extra) throws Exception;


	/**
	 *  This method is called when pool is being shutdown and there are still objects checkedout
	 *  By passing all the checked-out objects at once a factory has a chance to possibly first gracefully instruct 
	 *  the pooled resource that it should finish, and only if it does not finish in some time it should then forcefully
	 *  be destroyed.
	 *
	 *  While this method is being processed objects may be checked in and passed to returnObject or they can be returned as bad
	 *  and destroyed via deleteObject. That means that pooled objects themselves need to be thread safe on destroy/remove operations
	 *  or factory needs to ensure serial access.
	 */
	public void killObjects(java.util.Set set);

}
