package org.jboss.fresh.pool.pool.impl;

import org.jboss.fresh.pool.pool.Pool;
import org.jboss.fresh.pool.pool.PoolFactory;
import org.jboss.fresh.pool.pool.PoolException;
import org.jboss.fresh.pool.pool.IllegalPoolOperationException;
import org.jboss.fresh.pool.pool.PoolIsFullException;
import org.jboss.fresh.pool.pool.PoolTimeoutException;

import java.util.List;
import java.util.LinkedList;
import java.util.HashSet;

public class PoolImpl implements Pool {

	private static final org.apache.log4j.Logger log = org.apache.log4j.Logger.getLogger(PoolImpl.class);
	private String name;
	private PoolFactory factory;

	private int size = 0;
	private int minsize = 0;
	private int maxsize = 100;
	private int incr = 1;

	private boolean started = false;
	private boolean blocking = true;
	
	private long timeout = 30000;  // default is 30 secs

	private LinkedList checked;
	private LinkedList unchecked;


	public PoolImpl(String name, PoolFactory factory) {
		this.name = name;
		this.factory = factory;
		factory.registerPool(this);
	}

	public int getSize() {
		return size;
	}

	public int getMinSize() {
		return minsize;
	}

	public void setMinSize(int val) {
		minsize = val;
	}

	public int getMaxSize() {
		return maxsize;
	}

	/**
	 *  This method is synchronized to notify potential other thread blocked on checkout to wakeup
	 */
	public synchronized void setMaxSize(int val) {
		maxsize = val;
		notify();
	}

	public List getDiagnostics() {
		return null;
	}

	public void setIncrement(int num) {
		incr = num;
	}
	
	public void setNonBlocking(boolean val) {
		blocking = !val;
	}
	
	public void setTimeout(long millis) {
		timeout = millis;
	}	
	/**
	  *  grow pool but prevent overshooting
	  */
	private void grow(int gincr) throws PoolException {
		if(gincr > 0) {
			int max = maxsize - size;
			gincr = gincr < max ? gincr : max;
			for(int i=0; i < gincr; i++, size++) {
				try {
					unchecked.addLast(factory.createObject());
				} catch(Throwable ex) {
					throw new PoolException("Error occured while creating an object.", ex);
				}
			}
		} else if(gincr < 0) {
			int max = unchecked.size();

			gincr = -gincr;
			gincr = gincr < max ? gincr : max;

			for(int i=0; i < gincr; i++, size--) {
				Object o = unchecked.removeFirst();
				try {
					factory.deleteObject(o);  // TEST : make exception here and see that size has really decreased
				} catch(Throwable ex) {
					// we don't really know if any cleanup happened on the object or not
					// it's safer to keep it removed
					size--;
					throw new PoolException("Error occured while deleting an object. Object was thrown out of the pool anyway: " + o, ex);
				}
			}
		}
	}


	public synchronized Object checkout() throws PoolException, InterruptedException {

		if(!started) throw new PoolException("Pool has been stopped.");

		if(unchecked.size() == 0) {

			if(size < maxsize) {
				// grow by specified increment or the max you can
				grow(incr);
			} else {
				if (blocking) {

					// waitLoop
					// if maxSize gets increased or objects get checked in we can do something
					long expires = System.currentTimeMillis() + timeout;
					long togo = timeout;

					while(togo > 0) {
						wait(togo);
						if(!started) throw new PoolException("Pool has been stopped.");
						if(unchecked.size() > 0) {
							break;
						} else if(size < maxsize) {
							grow(incr);
							break;
						} else {
							togo = expires - System.currentTimeMillis();
							if(togo <= 0) throw new PoolTimeoutException("Timed out while checking out.");
						}
					}
				} else {
					throw new PoolIsFullException("All objects are checked out and the pool is full. Can't add any more.");
				}
			}
		}

		Object o = unchecked.removeLast();
		checked.addLast(o);
		try {
			factory.retrieveObject(o);
		} catch(Throwable ex) {
			// we just put it back
			unchecked.addLast(o);
			checked.removeLast();
			throw new PoolException("Error occured while retrieving an object: " + o, ex);
		}
		
		return o;
	}


	public synchronized void checkin(Object o) throws PoolException {

		if(!checked.remove(o)) throw new IllegalPoolOperationException("Trying to return an object that is not checked out.");

		try {
			factory.returnObject(o);
		} catch(Throwable ex) {
			// considered not returned - that is safer that presuming that the return succeeded
			checked.addLast(o);
			throw new PoolException("Error occured while returning an object: " + o, ex);
		}
	
		unchecked.addLast(o);

		// if pool was dynamically reconfigured then now it can be reduced in size
		if(size > maxsize) grow(maxsize-size);

		notify(); // if someone is waiting to checkout
	}


	public Object badCheckout(Object o) throws PoolException {
		return badCheckout(o, null);
	}


	public synchronized Object badCheckout(Object o, Object extra) throws PoolException {
		// destroy this object and get a new one
		if(!checked.remove(o)) throw new IllegalPoolOperationException("Trying to return an object that is not checked out.");
		size--;

		// if exception happens here the object is treated as removed
		try {
			o = factory.recycleObject(o, extra);
		} catch(Throwable ex) {
			throw new PoolException("An error occured while recycling an object: " + o, ex);
		}

		checked.addLast(o);
		size++;

		try {
			factory.retrieveObject(o);
		} catch(Throwable ex) {
			// put it in unchecked
			checked.removeLast();
			unchecked.addLast(o);
			throw new PoolException("Error occured while retrieving an object: " + o, ex);
		}
		return o;
	}

	public synchronized void invalidate(Object o) throws PoolException {
		// destroy this object
		if(!checked.remove(o)) throw new IllegalPoolOperationException("Trying to return an object that is not checked out.");
		size--;		

		try {
			factory.deleteObject(o);
		} catch(Throwable ex) {
			throw new PoolException("An error occured while deleting an object: " + o, ex);
		}
	}


	public synchronized void start() throws PoolException {
		if(started) throw new IllegalPoolOperationException("Pool has already been started.");

		unchecked = new LinkedList();
		checked = new LinkedList();

		grow(minsize);

		started = true;
	}

	public boolean isStarted() {
		return started;
	}

	public int getCheckedOutCount() {
		return checked.size();
	}

	public void stop() throws PoolException {
		HashSet kill = null;

		synchronized(this) {
			if(checked==null) return;
	
			if(checked.size() > 0) {
				log.warn("WARN: There are still objects checked out from the pool.");
				kill = new HashSet(checked);
			}
			// throw new IllegalPoolOperationException("Can't stop the pool - there are objects still checked out.");

			grow(-size);

			started = false;
			unchecked = null;
			checked = null;

			notifyAll();
		}

		if(kill!=null) factory.killObjects(kill);

		// we don't really care if some exception happens here
		factory.unregisterPool();
	}

    Object [] getCheckedInAndCheckedOut() {
        synchronized(this) {
            Object [] res = new Object[2];
            res[0] = new LinkedList(checked);
            res[1] = new LinkedList(unchecked);
            return res;
        }
    }
}