package org.jboss.fresh.vfs;

import org.jboss.fresh.naming.PathExpression;

import java.io.Serializable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.StringTokenizer;
import java.util.Collections;


public class FileName implements PathExpression, Serializable {

	private final static char DELIM = '/';
	private final static String ROOT = Character.toString(DELIM);

	private String name;
	private boolean absolute;
	private final LinkedList parsed;
	private final List unmodifiable;
	private int hash;


	public FileName(String name) {

		if ((name == null) || "".equals(name))
			throw new IllegalArgumentException("File path can't be empty.");

		this.name = name;
		absolute = name.charAt(0) == DELIM;

		LinkedList l = new LinkedList();
		StringTokenizer st = new StringTokenizer(name, Character.toString(DELIM));
		for (int i = 0; st.hasMoreTokens(); i++) {
			String token = st.nextToken();
			if (token.length() == 0) continue;
			checkPathToken(token);
			l.add(token.intern());
		}
		parsed = l;
		unmodifiable = Collections.unmodifiableList(parsed);
	}


	public FileName(List l, boolean absolute) {
		Iterator it = l.iterator();
		parsed = new LinkedList();
		unmodifiable = Collections.unmodifiableList(parsed);
		while (it.hasNext()) {
			String token = it.next().toString();
			if (token.length() == 0) continue;
			parsed.add(token.intern());
		}

		this.absolute = absolute;
	}


	public void checkPathToken(String name) {
		if (name.indexOf(DELIM) >= 0 || name.indexOf("*") >= 0 || name.indexOf("?") >= 0)
			throw new IllegalArgumentException("Illegal character in file name (" + name + ")");
	}

	public FileName absolutize(String name) {
		// this is need for absolutize( root.getPath() ) to work properly
		return absolutize(new FileName((name == null) ? ROOT : name));
	}


	public FileName absolutize(FileName relName) {
		LinkedList resultParsed = new LinkedList(parsed);
		// here we resolve each token
		LinkedList relParsed = new LinkedList(relName.parsed);
		int len = relParsed.size();

		for (int i = 0; i < len; i++) {
			String next = (String) relParsed.get(i);
			if ("..".equals(next)) {
				// going up from root is not allowed
				if (resultParsed.isEmpty())
					throw new IllegalArgumentException("Attempt to go up from root.");
				resultParsed.removeLast();

			} else if (".".equals(next)) {
				// do nothing

			} else {
				resultParsed.add(next);
			}
		}

		return new FileName(resultParsed, absolute);
	}


	public FileName getPath() {

		FileName result = null;
		int len = parsed.size();

		if (len == 0) {
			// we're returning parent of root
			result = null;

		} else if (len == 1) {
			if (absolute)
				result = new FileName(ROOT);
			else
				result = null;

		} else {
			LinkedList path = new LinkedList(parsed);
			path.removeLast();
			result = new FileName(path, absolute);
		}

		return result;
	}


	public String getName() {
		String name = null;

		if (parsed.size() == 0)
			name = null;
		else
			name = (String) parsed.getLast();

		return name;
	}


	public boolean isRoot() {
		return parsed.isEmpty();
	}


	/**
	 * Please note that this method returns an <strong>UNMODIFIABLE</strong> list!
	 * @return
	 */
	public List getParsed() {
		return unmodifiable;
	}

	public boolean isParsedEmpty() {
		return parsed.isEmpty();
	}


	public boolean equals(Object obj) {
		boolean result = false;

		if (obj instanceof FileName) {
			FileName compareTo = (FileName) obj;

			if ((compareTo.isAbsolute() != absolute) || (!compareTo.parsed.equals(parsed)))
				result = false;
			else
				result = true;
		}

		return result;
	}


	public FileName getBase(int i) {
		// returns path resulting from first i tokens
		List base = parsed.subList(0, i);
		return new FileName(base, absolute);
	}

	public FileName getSuffix(int i) {
		// returns path resulting from first i tokens
		List base = parsed.subList(i, parsed.size());
		return new FileName(base, absolute);
	}

	public int size() {
		return parsed.size();
	}

	public FileName relativeTo(String name) {
		return relativeTo(new FileName(name));
	}


	// what you need to pass to name.absolutize to come from name to this
	// The presumption is that all strings are automatically cononicalized
	// no /./ or /../ components in the name
	// Also we presumed all strings inside parsed are interned ... we make sure of that
	// in other methods
	// Kanonizirati relativno ime je precej bedno...
	public FileName relativeTo(FileName name) {
		// you find common root
		// it can be "/"
		// if name is higher above than we are then it's ok, cause we need
		// to add to name to get here
		// if name is further down from us or we even have to go up and down again
		// to get there then the name will be either all /../ or it will also go down

		FileName result = null;

		// compare them from root down
		if (name.isRoot()) {
			LinkedList l = new LinkedList(parsed);
			if (l.isEmpty())
				l.add(".");
			result = new FileName(l, false);

		} else if (isRoot()) {
			// give it as many .. as there is parsed length
			LinkedList l = new LinkedList();
			Iterator it = name.iterateTokens();
			while (it.hasNext()) {
				l.add("..");
				it.next();
			}
			result = new FileName(l, false);

		} else {
			// zdaj pa...
			int i;
			LinkedList startPath = name.parsed;
			LinkedList endPath = parsed;
			LinkedList l = new LinkedList();

			for (i = 0; true; i++) {
				// najdi tocko razhajanja
				// preveri ce sploh je znotraj vectorja, ce ni, potem smo ce pri razhajanju
				String startPathToken = null;
				String endPathToken = null;

				if (i < parsed.size())
					startPathToken = (String) startPath.get(i);

				if (i < name.parsed.size())
					endPathToken = (String) endPath.get(i);

				if (startPathToken == null || endPathToken == null)
					break; // the point where they don't match any more

				if (endPathToken != startPathToken) // they don't match
					break;
			}

			for (int j = i; j < endPath.size(); j++) {
				l.add("..");
			}

			for (int j = i; j < startPath.size(); j++) {
				l.add(startPath.get(j));
			}

			if (l.isEmpty())
				l.add(".");

			result = new FileName(l, false);
		}

		return result;
	}


	/**
	 * our object is immutable so caching the hashcode is easy
	 */
	public int hashCode() {

		if (hash == 0) {
			hash = 17;
			hash = 37 * hash + (absolute ? 1 : 0);
			int tmp = 0;
			if (parsed != null)
				tmp = parsed.hashCode();

			hash = 37 * hash + tmp;
/*
			tmp = 0;
			if( vfs != null )
				tmp = vfs.hashCode();
			hash = 37 * hash + tmp;
*/
//System.out.println("CALCULATED HASH");
//Thread.currentThread().dumpStack();

		} else {
			int tmp0 = 17;
			tmp0 = 37 * tmp0 + (absolute ? 1 : 0);
			int tmp = 0;
			if (parsed != null)
				tmp = parsed.hashCode();

			tmp0 = 37 * tmp0 + tmp;

/*			tmp = 0;
			if ( vfs != null )
				tmp = vfs.hashCode();

			tmp0 = 37 * tmp0 + tmp;
*/
			if (tmp0 != hash)
				throw new RuntimeException(" * * * * * * * *   BUMMER  * * * * * * * * *");
		}


//System.out.println("hashCode: " + hash + "(" + this + ")" + super.hashCode());

		return hash;
	}

	public boolean isAbsolute() {
		return absolute;
	}

	public boolean isRelative() {
		return !absolute;
	}


	public boolean isParent(FileName parent) {
		boolean result = true;

		Iterator itParent = parent.iterateTokens();
		Iterator itChild = parsed.iterator();

		while (itParent.hasNext()) {
			if (!itChild.hasNext() || !itParent.next().equals(itChild.next())) {
				result = false;
				break;
			}
		}

		return result;
	}


	public String toString() {
		StringBuffer buf = new StringBuffer();

		if (absolute)
			buf.append(DELIM);
		Iterator it = parsed.iterator();
		while (it.hasNext()) {
			String pathToken = (String) it.next();
			buf.append(pathToken);
			if (it.hasNext())
				buf.append(DELIM);
		}

		return buf.toString();
	}


	/**
	 * Please note that this method returns an <strong>UNMODIFIABLE</strong> iterator!
	 * @return
	 */
	public Iterator iterateTokens() {
		return unmodifiable.iterator();
	}

	public Object getFinalToken() {
		return getName();
	}

	public PathExpression getParentExpression() {
		return getPath();
	}

	/**
	 * Please note that this method returns an <strong>UNMODIFIABLE</strong> list!
	 * @return
	 */
	public List getTokens() {
		return unmodifiable;
	}

	public PathExpression absolutize(Object obj) {
		return absolutize((String) obj);
	}

}