package org.jboss.fresh.vfs;

import java.util.Collection;
import java.util.List;


/**
 *  VFS provides file meta and content persistence and also transactions
 *  functionality over all methods. All VFS implementations should check at the
 *  beginning of every transaction-regarding method, if transaction is active.
 *  If not, it should throw an appropriate VFSException. All exceptions thrown
 *  in VFS are thrown only to prevent data integrity corruption and to ensure
 *  fail safety or to apply security restrictions. If parameters are invalid,
 *  unexpected exception will be thrown; one lever higher must be ensured that
 *  in that case rollback is called and data integrity remains untouched. All
 *  validition in VFS is only for data safety. Any other validation should be
 *  implemented at higher level.
 */
public interface VFS {

    /*  We haven't discussed this one yet; we can forget about it for now.
    public void setVFSContext(VFSContext ctx);
    public VFSContext getVFSContext();
    */

    /**
     *  Name must not be null. If exception is thrown, it means something is wrong
     *  in the meta persistence layer or access is not allowed. This is all a
     *  thrown exception is allowed to indicate.
     *
     * @param  ctx               User's context.
     * @param  name              File's filename.
     * @param  direct            If false, any final links will be resolved.
     * @return                   Whether file exists or not.
     * @exception  VFSException  Description of the Exception
     */
    public boolean exists(UserCtx ctx, FileName name, boolean direct) throws VFSException;

    /*
    /**
     *  Name must not be null. If exception is occured, it indicates meta
     *  persistence problem or security restrictions. If name represent link, then
     *  target FileInfo object is also returned, in extra field. Returns valid
     *  Collection object with path resolution thread paths. It is empty only when
     *  name was root. If name does not exist (only parent or grand-parent does),
     *  it returns partial resolution (as much as it gets).
     *
     *@param  ctx               User context.
     *@param  name              Name of file.
     *@return                   List of path resolution steps.
     /
    public List resolvePath( UserCtx uctx, FileName name ) throws VFSException;
    */

    /**
     *  Name must not be null. If exception is occured, it indicates meta
     *  persistence problem or security restrictions. Returns valid fully loaded
     *  FileInfo object. If object doesn't exist, returns null.
     *
     * @param  ctx               User's context.
     * @param  name              File's filename.
     * @param  direct            If false, a final link in the filename will be
     *      resolved.
     * @return                   The fileInfo value
     * @exception  VFSException  Description of the Exception
     */
    public FileInfo getFileInfo(UserCtx ctx, FileName name, boolean direct) throws VFSException;

    /**
     *  Name must not be null and the file must exist. If exception is occured, it
     *  indicates meta persistence problem or security restrictions. Returns valid
     *  Collection object filled with child FileInfo objects.
     *
     * @param  ctx     User's context.
     * @param  name    Parent's filename.
     * @param  direct  If false, a final link in the parent file will be resolved.
     * @return         List of child FileInfo objects.
     */
    public List list(UserCtx ctx, FileName name, boolean direct) throws VFSException;

    /**
     *  File must not be null and must not exist. Only name, type and mime must be
     *  specified. Optional can be specified also linkto, attributes and roles.
     *  CreateDate, lastModified, length, complete, tag and orderIndex are set by
     *  vfs. If exception is occured, it indicates meta persistence problem or
     *  security restrictions. Returns tag string for this file.
     *
     * @param  ctx   User's context.
     * @param  file  Description of the Parameter
     * @return       Tag indication current state of file.
     */
    public String createFile(UserCtx ctx, FileInfo file) throws VFSException;

    /**
     *  File must not be null and must exist. Everything except name (&parent&path)
     *  will not be updated. If exception is occured, it indicates meta persistence
     *  problem or security restrictions.
     *
     * @param  ctx   User's context.
     * @param  file  Description of the Parameter
     * @return       Tag indication state of file after update.
     */
    public String updateFile(UserCtx ctx, FileInfo file) throws VFSException;

    /**
     *  File must not be null and must exist. File meta infos and file's content
     *  will be permanently removed. Removal is recursive. All children meta infos
     *  and content's will be removed. If exception is occured, it indicates
     *  persistence problem or security restrictions or that file doesn't exist.
     *
     * @param  ctx     User's context.
     * @param  name    Filename of the existing, to-be-removed file.
     * @param  direct  If false, a final link in the file will be resolved.
     */
    public void remove(UserCtx ctx, FileName name, boolean direct) throws VFSException;

    /**
     *  FileOpInfo must be filled with tag, filename (file must exist), buf and
     *  complete. If exception is occured, it indicates persistence problem or
     *  security restrictions. Returns FileWriteInfo filled with new tag.
     *
     * @param  ctx               Description of the Parameter
     * @param  info              Description of the Parameter
     * @return                   Description of the Return Value
     * @exception  VFSException  Description of the Exception
     */
    public FileWriteInfo write(UserCtx ctx, FileOpInfo info) throws VFSException;

    /**
     *  FileOpInfo must be filled with tag (not mandatory if offset is 0), filename
     *  (file must exist) and offset. If exception is occured, it indicates
     *  persistence problem or security restrictions. Returns FileReadInfo filled
     *  with buffer, length and information if there are any more records available
     *  for this file.
     *
     * @param  uctx              Description of the Parameter
     * @param  info              Description of the Parameter
     * @return                   Description of the Return Value
     * @exception  VFSException  Description of the Exception
     */
    public FileReadInfo read(UserCtx uctx, FileOpInfo info) throws VFSException;

    /**
     *  Resolves the filename up to the final target.
     *
     * @param  ctx       User's context.
     * @param  filename  Filename to be resolved.
     * @param  partial   If true, then last token in path is not resolved. Useful
     *      when there's a need to resolve to-be-created file, etc.
     * @return           Resolved filename.
     */
    public FileName resolve(UserCtx ctx, FileName filename, boolean partial) throws VFSException;

    /**
     *  Returns all the links that have the given target.
     *
     * @param  ctx      User's context.
     * @param  target   Links to this target is to be looked up.
     * @param  partial  If true, then last token in the target is not resolved.
     * @return          Collection of FileInfos of type link and target as given.
     */
    public Collection getLinks(UserCtx ctx, FileName target, boolean partial) throws VFSException;

    /**
     *  Tells if a file has content or not..
     *
     * @param  ctx     User's context.
     * @param  file    File to be checked for content.
     * @param  direct  If false, then the filename is resolved all the way up to
     *      the final target.
     * @return         True if file has any content.
     */
    public boolean hasContent(UserCtx ctx, FileName file, boolean direct) throws VFSException;

    /**
     *  Moves or renames file from oldPath to newPath. Moving/Renaming occurs
     *  recursively. Meaning all children file's will be affected.
     *
     * @param  ctx      User's context.
     * @param  oldName  The filename of the file to be renamed or moved.
     * @param  newName  The target filename.
     * @param  direct   If false, then the filename is resolved all the way up to
     *      the final target.
     */
    public void move(UserCtx ctx, FileName oldName, FileName newName, boolean direct) throws VFSException;

    /**
     *  Positions file right before given file in the same directory.
     *
     * @param  ctx   User's context.
     * @param  file  File to be reordered.
     * @param  next  File to be used as relative destination.
     * @return       The orderIndex of reordered file in parent directory.
     */
    public float moveBefore(UserCtx ctx, FileName file, FileName next) throws VFSException;
}

