package org.jboss.fresh.vfs.impl;

import org.jboss.fresh.vfs.*;

import java.io.IOException;
import java.io.OutputStream;


public class LazyVFSOutputStream extends OutputStream {

    private SecureVFS vfs;
    private FileName filename;
    private boolean append;
    private boolean closed;

    private boolean first = true;
    private String tag;

    private byte[] buff;
    private int tlen;
    private boolean needCreate = false;


    public LazyVFSOutputStream(SecureVFS vfs, String filename) throws IOException {
        this(vfs, filename, false);
    }

    public LazyVFSOutputStream(SecureVFS vfs, String fname, boolean append) throws IOException {

        try {
            this.vfs = vfs;
            this.filename = new FileName(fname);
            this.append = append;

            // need to locate this file if it exists and not append overwrite it

            FileInfo info = vfs.getFileInfo(filename, false);
            // if it does not exist create an empty one
            if (info == null) { // file does not exist
                needCreate = true;
            } else {
                tag = info.getTag();
            }

        } catch (Throwable ex) {
            throw new VFSIOException(ex);
        }

    }


    protected void fileCreate() throws Exception {
        //if (!org.jboss.fresh.util.TxSupport.isActive()) org.jboss.fresh.util.TxSupport.begin();
        FileInfo info = new FileInfo(filename, FileInfo.TYPE_FILE);
        info.setMime("x-application/octet-stream");
        tag = vfs.createFile(info);
        needCreate = false;
    }


    public void close() throws IOException {
        if (closed) return;  // if it has already been flushed we must not do it again.

        closed = true;
        flush();
    }


    public void flush() throws IOException {
        // write what you have buffered
        // if nothing buffered, write nothing
        try {


            if (tlen == 0) return;
//System.out.println("[VFSOutputStream] flush: tlen: " + tlen);
            // append to the current location there

            if (needCreate)
                fileCreate();
// DEBUGGING CODE
//		else {
//			FileInfo info=vfs.getFileInfo(filename, false);
//			if(info==null) throw new RuntimeException("**** BUMMER !!!");
//		}


            FileOpInfo inf = new FileOpInfo();
            inf.tag = tag;
            inf.filename = filename;

// DEBUG CODE >>>
// Check if file exists:
//		FileInfo finf=vfs.getFileInfo(filename, false);
//		if(finf==null) {
//			System.out.println("[VFSOutputStream] flush: File with a name " + filename + " does not exist.");
//		}
// <<<
            if (first && !append) { // here we decide weather to overwrite
//System.out.println("[LazyVFSOutputStream] Set OVERWRITE mode.");
                inf.append = false;
                first = false;
            } else {
//System.out.println("[LazyVFSOutputStream] Set APPEND mode.");
                inf.append = true;
            }

            inf.complete = closed;

            byte[] bf;

            if (tlen != buff.length) {
                bf = new byte[tlen];
                System.arraycopy(buff, 0, bf, 0, tlen);
            } else {
                bf = buff;
            }

            inf.buf = bf;
// DEBUG CODE >>>
//System.out.println("[VFSOutputStream] flush: FileInfo: ");
//System.out.println(inf);
// <<<
            FileWriteInfo retinf = vfs.write(inf);
            tag = retinf.tag;

            tlen = 0;
        } catch (Throwable ex) {
            ex.printStackTrace();
            throw new VFSIOException(ex);
        }
    }

    // don't use this. Always wrap BufferedOutputStream around
    public void write(int b) throws IOException {
        write(new byte[]{(byte) b});
    }

    public void write(byte[] buf) throws IOException {
        write(buf, 0, buf.length);
    }

    public void write(byte[] buf, int off, int len) throws IOException {
        flush(); // flush previous one, save this one

        int sz;
        if (buff == null || buff.length < len) { // increase buffer if necessary
//System.out.println("[VFSOutputStream] write: increasing buffer size.");

            if (buf.length > len)
                sz = buf.length;
            else
                sz = len;
            buff = new byte[sz];
        }

        System.arraycopy(buf, off, buff, 0, len);
//System.out.println("[VFSOutputStream] write: copied " + len + " bytes to the buffer.");
        tlen = len;
    }
}