/**
 * Contains generic CRUD methods to add, read, update and remove (singleton) resources. Some methods just execute the
 * underlying DMR operations, other methods also interact with the user by showing (confirmation) dialogs.
 */
class CrudOperations {

    /**
     * Opens an add-resource dialog for the given resource type. The dialog contains fields for all required request
     * properties. When clicking "Add", a new resource is added using the specified address. After the resource has been
     * added a success message is displayed and the callback is executed.
     * 
     * @param {string} type       The human readable resource type used in the dialog header and success message.
     * @param {AddressTemplate|ResourceAddress|string} address    The address for the add operation. Must end in <code>&lt;resource type&gt;=*</code>.
     * @param {string[]} attributes Additional attributes (besides the required attributes) which should be part of the
     * add-resource dialog. May be null or empty.
     * @param {function(name: string, address: ResourceAddress)} callback   The callback executed after the resource has been added.
     */
    addDialog(type, address, attributes, callback) {}

    /**
     * Executes an add operation using the specified name and payload. After the resource has been added a success
     * message is displayed and the callback is executed.
     * 
     * @param {string} type     The human readable resource type used in the dialog header and success message.
     * @param {string} name     The resource name which is part of the add operation.
     * @param {AddressTemplate|ResourceAddress|string} address  The address for the add operation. Must end in <code>&lt;resource type&gt;=*</code>.
     * @param {ModelNode} payload  The optional payload of the add operation (may be null or undefined).
     * @param {function(name: string, address: ResourceAddress)} callback The callback executed after the resource has been added.
     */
    add(type, name, address, payload, callback) {}

    /**
     * Opens an add-resource dialog for the given singleton resource type. The dialog contains fields for all required
     * request properties. When clicking "Add", a new singleton resource is added using the specified address template.
     * After the singleton resource has been added a success message is displayed and the callback is executed.
     * 
     * @param {string} type       The human readable resource type used in the dialog header and success message.
     * @param {Object} address    The address for the add operation. Must end in <code>&lt;resource type&gt;=&lt;resource
     * name&gt;</code>.
     * @param {string[]} attributes Additional attributes (besides the required attributes) which should be part of the
     * add-resource dialog. May be null or empty.
     * @param {function(address: ResourceAddress)} callback   The callback executed after the singleton resource has been added.
     */
    addSingletonDialog(type, address, attributes, callback) {}

    /**
     * Executes an add operation using the specified payload. After the resource has been added a success message is
     * displayed and the callback is executed.
     * 
     * @param {string} type     The human readable resource type used in the dialog header and success message.
     * @param {AddressTemplate|ResourceAddress|string} address  The address for the add operation. Must end in <code>&lt;resource type&gt;=&lt;resource
     * name&gt;</code>.
     * @param {ModelNode} payload  The optional payload of the add operation (may be null or undefined).
     * @param {function(address: ResourceAddress)} callback The callback executed after the singleton resource has been added.
     */
    addSingleton(type, address, payload, callback) {}

    /**
     * Executes a read-resource operation on the specified address and passes the result to the callback.
     * 
     * @param {AddressTemplate|ResourceAddress|string} address  The address for the read-resource operation operation.
     * @param {function(result: ModelNode)} callback The callback which gets the result of the read-resource operation.
     */
    read(address, callback) {}

    /**
     * Executes a recursive read-resource operation on the specified address and passes the result to the callback.
     * 
     * @param {AddressTemplate|ResourceAddress|string} address  The address for the read-resource operation operation.
     * @param {function(result: ModelNode)} callback The callback which gets the result of the read-resource operation.
     */
    readRecursive(address, callback) {}

    /**
     * Executes a read-children-resources operation on the specified address and passes the result as {@link Property}
     * array to the callback.
     * 
     * @param {AddressTemplate|ResourceAddress|string} address   The address for the read-children-resources operation.
     * @param {string} childType The child resource type.
     * @param {function(children: Property[])} callback  The callback which gets the result of the read-children-resources operation.
     */
    readChildren(address, childType, callback) {}

    /**
     * Writes the changed values to the specified resource. After the resource has been saved a standard success message
     * is displayed and the callback is executed. If the change set is empty, a warning message is displayed and the
     * callback is executed.
     * 
     * @param {string} type      The human readable resource type used in the success message.
     * @param {string} name      The resource name.
     * @param {AddressTemplate|ResourceAddress|string} address   The address for the operation.
     * @param {{key: string, value: object}} changeSet A key-value map containing the changes to the resource.
     * @param {function()} callback  The callback executed after the resource has been saved.
     */
    save(type, name, address, changeSet, callback) {}

    /**
     * Writes the changed values to the specified singleton resource. After the resource has been saved a standard
     * success message is displayed and the callback is executed. If the change set is empty, a warning message is
     * displayed and the callback is executed.
     * 
     * @param {string} type      The human readable resource type used in the success message.
     * @param {AddressTemplate|ResourceAddress|string} address   The address for the operation.
     * @param {{key: string, value: object}} changeSet A key-value map containing the changes to the resource.
     * @param {function()} callback  The callback executed after the singleton resource has been saved.
     */
    saveSingleton(type, address, changeSet, callback) {}

    /**
     * Shows a confirmation dialog and removes the resource if confirmed by the user. After the resource has been
     * removed a success message is displayed and the callback is executed.
     * 
     * @param {string} type     The human readable resource type used in the success message.
     * @param {string} name     The resource name.
     * @param {AddressTemplate|ResourceAddress|string} address  The address for the operation.
     * @param {function()} callback The callback executed after the resource has been removed.
     */
    remove(type, name, address, callback) {}

    /**
     * Shows a confirmation dialog and removes the singleton resource if confirmed by the user. After the resource has
     * been removed a success message is displayed and the callback is executed.
     * 
     * @param {string} type     The human readable resource type used in the success message.
     * @param {AddressTemplate|ResourceAddress|string} address  The address for the operation.
     * @param {function()} callback The callback executed after the resource has been removed.
     */
    removeSingleton(type, address, callback) {}
}

/**
 * Represents an extension written in JavaScript.
 */
class Extension {

    /**
     * Creates a header extension.
     * 
     * @param {string} name       A unique name of the extension.
     * @param {string} title      The title of the menu item in the header.
     * @param {function()} entryPoint The entrypoint of the header extension.
     * 
     * @return {Extension} the extension which can be registered using the {@link ExtensionRegistry}.
     */
    static header(name, title, entryPoint) {}

    /**
     * Creates a footer extension.
     * 
     * @param {string} name       A unique name of the extension.
     * @param {string} title      The title of the menu item in the footer.
     * @param {function()} entryPoint The entrypoint of the footer extension.
     * 
     * @return {Extension} the extension which can be registered using the {@link ExtensionRegistry}.
     */
    static footer(name, title, entryPoint) {}
}

/**
 * Provides access to all important classes. Acts as an entry point for the JavaScript API.
 */
class Core {

    /**
     * @return {CrudOperations} crud operations
     */
    get crud() {}

    /**
     * @return {Dispatcher} dispatcher
     */
    get dispatcher() {}

    /**
     * @return {Endpoints} endpoints
     */
    get endpoints() {}

    /**
     * @return {Environment} environment
     */
    get environment() {}

    /**
     * @return {ExtensionRegistry} extension registry
     */
    get extensionRegistry() {}

    /**
     * @return {MetadataProcessor} metadata processor
     */
    get metadataProcessor() {}

    /**
     * @return {MetadataRegistry} metadata registry
     */
    get metadataRegistry() {}

    /**
     * @return {StatementContext} statement context
     */
    get statementContext() {}

    /**
     * @return {Core} the singleton core instance.
     */
    static getInstance() {}

    /**
     * Creates and returns a dialog builder using the specified title.
     * 
     * @param {string} title The dialog title.
     * 
     * @return {DialogBuilder} a builder to create dialogs
     */
    dialog(title) {}

    /**
     * Shows an error message.
     * 
     * @param {string} message The error message.
     */
    error(message) {}

    /**
     * Returns a new form builder for a {@link ModelNode}.
     * 
     * @param {Metadata|AddressTemplate|string} meta The metadata for the form.
     * 
     * @return {FormBuilder} the form builder
     */
    form(meta) {}

    /**
     * Shows an info message.
     * 
     * @param {string} message The info message.
     */
    info(message) {}

    /**
     * Returns a new form builder for a {@link NamedNode}.
     * 
     * @param {Metadata|AddressTemplate|string} meta The metadata for the form.
     * 
     * @return {FormBuilder} the form builder
     */
    namedForm(meta) {}

    /**
     * Returns a new table builder for a {@link NamedNode}.
     * 
     * @param {Metadata|AddressTemplate|string} meta The metadata for the table.
     * 
     * @return {TableBuilder} the table builder
     */
    namedTable(meta) {}

    /**
     * Returns a new operation builder.
     * 
     * @param {AddressTemplate|ResourceAddress|string} address The address.
     * @param {string} name    The operation name.
     * 
     * @return {OperationBuilder} the operation builder
     */
    operation(address, name) {}

    /**
     * Shows a success message.
     * 
     * @param {string} message The success message.
     */
    success(message) {}

    /**
     * Returns a new table builder for a {@link ModelNode}.
     * 
     * @param {Object} meta The metadata for the table.
     * 
     * @return {TableBuilder} the table builder
     */
    table(meta) {}

    /**
     * Shows a warning message.
     * 
     * @param {string} message The warning message.
     */
    warning(message) {}
}

/**
 * Registry to manage HAL extensions written in JavaScript.
 */
class ExtensionRegistry {

    /**
     * Registers an extension. Use this method to register your extension.
     * <p>
     * If the extension is already registered, this method will do nothing.
     * 
     * @param {Extension} extension the extension to register.
     */
    register(extension) {}

    /**
     * Injects the script and stylesheets of an extension. This method is used during development. Normally you don't
     * have to call this method.
     * 
     * @param {string} script      the extension's script.
     * @param {string[]} stylesheets an optional list of stylesheets.
     */
    inject(script, stylesheets) {}
}

